/****************************************************************************
                              ObjWidgetBase.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements a base-class for all TCL-aware objects...
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "ObjWidgetBase.h"

/********************************************************************
 * ObjWidgetBase()
 ********************************************************************/
ObjWidgetBase::ObjWidgetBase(
        Tcl_Interp        *interp,
        Uint32             objc,
        Tcl_Obj           *const objv[]) :
    ok(0),
    d_interp(interp)
{
    if (objc < 2) {
        Tcl_AppendResult(interp, "too few args", 0);
        return;
    }

    d_instName = Tcl_GetString(objv[1]);

    d_cmdToken = Tcl_CreateObjCommand(interp, Tcl_GetString(objv[1]),
            &ObjWidgetBase::InstCmd, this, NULL);

    if (!d_cmdToken) {
        Tcl_AppendResult(interp, "cannot create command ",
                Tcl_GetString(objv[1]), 0);
        return;
    }

    Tcl_SetObjResult(interp, Tcl_NewStringObj(d_instName.value(), -1));

    ok = 1;
}

/********************************************************************
 * ~ObjWidgetBase()
 ********************************************************************/
ObjWidgetBase::~ObjWidgetBase()
{
    Tcl_DeleteCommandFromToken(d_interp, d_cmdToken);
}


/********************************************************************
 * InstCmd()
 ********************************************************************/
int ObjWidgetBase::InstCmd(
        Uint32        objc,
        Tcl_Obj      *const objv[])
{
    fprintf(stderr, "ERROR :: Baseclass InstCmd called\n");

    return TCL_ERROR;
}

/********************************************************************
 * InstCmd()
 ********************************************************************/
int ObjWidgetBase::InstCmd(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               objc,
        Tcl_Obj          *const objv[])
{
    ObjWidgetBase   *obj = (ObjWidgetBase *)clientData;
    return obj->InstCmd(objc, objv);
}

/********************************************************************
 * CmdSwitch()
 *
 * This static routine does the work of parsing cmds...
 ********************************************************************/
Int32 ObjWidgetBase::CmdSwitch(ObjWidgetBase::CmdStruct *cmds, const char *cmd)
{
    int i = 0;
    int cmdIdx = 0; 
    int foundCmd = -1;

    while(cmds[i].cmdStr[0] != 0) {
        while ((cmds[i].cmdStr[cmdIdx] != 0) && 
               (cmd[cmdIdx] != 0) && 
               (cmd[cmdIdx] == cmds[i].cmdStr[cmdIdx])) cmdIdx++;
        if (cmds[i].cmdStr[cmdIdx] || cmd[cmdIdx]) {
            i++;
            cmdIdx = 0;
        } else {
            foundCmd = cmds[i].cmdIdx;
            break;
        }
    }
    return foundCmd;
}

/********************************************************************
 * CmdSwitch()
 ********************************************************************/
Int32 ObjWidgetBase::CmdSwitch(const char *cmd)
{
    CmdStruct *cmds = getCmdStructs();
    Int32      ret;

    if (!cmds) {
        Tcl_AppendResult(d_interp, "no commands registered for ", 
                d_instName.value(), 0);
        return -1;
    }

    if ((ret = ObjWidgetBase::CmdSwitch(cmds, cmd)) == -1) {
        Uint32 idx = 0;
        Tcl_AppendResult(d_interp, "no sub-cmd ", cmd, ": possible are ", 0);

        while (cmds[idx].cmdStr[0]) {
            Tcl_AppendResult(d_interp, cmds[idx].cmdStr, " ", 0);
            idx++;
        }
    }

    return ret;
}

/********************************************************************
 * getCmdStructs()
 ********************************************************************/
ObjWidgetBase::CmdStruct *ObjWidgetBase::getCmdStructs()
{
    return 0;
}

/********************************************************************
 * WidgetInit()
 ********************************************************************/
int ObjWidgetBase::WidgetInit(
        Tcl_Interp        *interp,
        const char        *widgetName,
        const char        *widgetVersion,
        const char        *widgetCmd,
        Tcl_ObjCmdProc    *cmdProc)
{
    Tcl_Command cmd;
    if (Tcl_InitStubs(interp, "8.1", 0) == NULL) {
        Tcl_AppendResult(interp, "stubs init failed...", 0);
        return TCL_ERROR;
    }

    WidgetMgr_Init(interp);
    Tcl_PkgProvide(interp, (char *)widgetName, (char *)widgetVersion);

    WidgetMgr_AddType(interp, widgetName);

    cmd = Tcl_CreateObjCommand(interp, (char *)widgetCmd, cmdProc, 0, 0);
    if (!cmd) {
        Tcl_AppendResult(interp, "cannot create command ", widgetCmd, 0);
        return TCL_ERROR;
    }

    return TCL_OK;
}

/********************************************************************
 * suff_args()
 ********************************************************************/
int ObjWidgetBase::suff_args(int argc, int req_argc)
{
    if (argc < req_argc) {
        char  _p[16];
        sprintf(_p, "%d", req_argc);
        Tcl_AppendResult(d_interp, "too few args - want ", _p, 0);
        return 0;
    }
    return 1;
}


