/*****************************************************************************
 *                               SdbReader.cc
 * Author: Matthew Ballance
 * Desc:   Implements a signal reader. A signal reader accesses a Signal DB
 *         for its signal values.
 *
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *****************************************************************************/
#include "SdbReader.h"
#include "TclBitVector.h"
#include "tree_node.h"
#include "DesignDB.h"
#include "tclOldConfig.h"
#include "WidgetManager.h"
#include "SigDB.h"
#include "SdbSignal.h"
#include "DFIO.h"
#include "DFIOTrace.h"
#include "CursorMgr.h"
#include "SdbrCursor.h"
#include "CallbackMgr.h"

static int SdbReader_InstCmd(
    ClientData        clientData,
    Tcl_Interp       *interp,
    int               argc,
    char            **argv);

class SdbDesc {
    public:
        SdbDesc(SigDB *sdb, SdbReader *sdbr) {
            this->sdb    = sdb;
            this->sdbr   = sdbr;
        }

        ClientData         sdb_callback_id;
        SdbReader         *sdbr;
        SigDB             *sdb;
};

/******************************************************************
 * SDBR_SigDataUpdateCB
 ******************************************************************/
int SdbReader::SDBR_SigDataUpdateCB(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               objc,
        Tcl_Obj          *const objv[])
{
    SdbDesc    *sdbd = (SdbDesc *)clientData;

    sdbd->sdbr->maxTimeValid = 0;

    CallbackMgr_Invoke(CBTYPE_SDBR_SIGDATA_UPDATE, 
            sdbd->sdbr->instName.value(), objc, objv);

    return TCL_OK;
}

/******************************************************************
 * SdbReader()
 ******************************************************************/
SdbReader::SdbReader(
        Tcl_Interp         *interp,
        Uint32              argc,
        Char              **argv) : instName(argv[1])
{
    this->interp = interp;
    okay = 0;

    sdbName                = 0;
    sdb                    = 0;
    SDBR_SigDataUpdateCbId = 0;

    d_cbReqs = 0;

    Tcl_InitHashTable(&d_sigHash, TCL_STRING_KEYS);

    if (Configure(argc-2, &argv[2], 0) != TCL_OK) {
        return;
    }

    WidgetMgr_AddInst(interp, WIDGET_TYPE_SDBR, argv[1], this);

    sdbdList = new Vector<SdbDesc>();

    cursorMgr = new CursorMgr(interp, this);

    sigId = 1;

    resolution = -9;

    Tcl_CreateCommand(interp, argv[1], 
            (Tcl_CmdProc *)SdbReader_InstCmd, this, NULL);

    Tcl_SetObjResult(interp, Tcl_NewStringObj(argv[1], -1));

    maxTimeValid = 0;
    okay = 1;
}

/******************************************************************
 * ~SdbReader()
 *
 * TODO: Add destructor...
 ******************************************************************/
SdbReader::~SdbReader(void)
{
    Uint32 i;

    if (SDBR_SigDataUpdateCbId) {
        CallbackMgr_DeleteCb(SDBR_SigDataUpdateCbId);
    }

    for (i=0; i<sdbdList->length(); i++) {
        CallbackMgr_DeleteCb(sdbdList->idx(i)->sdb_callback_id);
        delete sdbdList->idx(i);
    }

    delete sdbdList;
}

static Char *prv_ResStrings[] = {
/*    0   1    2     3     4     5     6     7     8     9     10    11    12 */
     "", "s", "ms", "ms", "us", "us", "us", "ns", "ns", "ns", "ps", "ps", "ps",
     ""
};

/******************************************************************
 * get_resString()
 ******************************************************************/
const Char *SdbReader::get_resString()
{
    Int32 tmp_res = -resolution;
    const Char *resStr;

    if (tmp_res > 12) {
        fprintf(stderr, "ERROR: tmp_res = %d\n", tmp_res);
    } else {
        resStr = prv_ResStrings[tmp_res];
    }

    return resStr;
}

/******************************************************************
 * Configure()
 ******************************************************************/
Int32 SdbReader::Configure(Uint32 argc, Char **argv, Uint32 flags)
{
    Int32 ret;

    ret = Tcl_ConfigureWidget(interp, 0, getConfigSpec(), argc, argv,
            (char *)this, flags);

    if (optionSpecified(Opt_SdbName) && sdbName) {
        sdb = (SigDB *)WidgetMgr_GetObjHandle(interp, "SDB", sdbName);
        if (!sdb) {
            Tcl_AppendResult(interp, "no SDB ", sdbName, 0);
            return TCL_ERROR;
        }

        addSDB(sdb);
    }

    return ret;
}

/******************************************************************
 * InstCmd()
 ******************************************************************/
Int32 SdbReader::InstCmd(Uint32 argc, Char **argv)
{
    Uint32                  cmd, i, tolerance, idx;
    Tcl_Obj                *list;
    SdbrCursor             *cursor;
    Vector<SdbrCursor>     *cursors;

    cmd = CmdSwitch(sdbReaderCmds, argv[1]);

    switch (cmd) {
        case SRC_Config:
            return Configure(argc-2, &argv[2], TK_CONFIG_ARGV_ONLY);
            break;

        case SRC_Cget:
            return Tcl_ConfigureValue(interp, 0, getConfigSpec(),
                    (char *)this, argv[2], 0);
            break;

        /*************************************************************
         * ClearCutBuffer
         *************************************************************/
        case SRC_ClearCutBuffer:
            for (i=0; i<d_cutBuffer.length(); i++) {
                delete d_cutBuffer.idx(i);
            }
            d_cutBuffer.empty();
            break;

        /*************************************************************
         * AppendCutBuffer
         *************************************************************/
        case SRC_AppendCutBuffer:
            return appendCutBuffer(argc-1, &argv[1]);
            break;

        /*************************************************************
         * PutCutBuffer
         *************************************************************/
        case SRC_PutCutBuffer:
            return putCutBuffer(argc-1, &argv[1]);
            break;

        /*************************************************************
         * Trace Manipulation Commands
         *************************************************************/
        case SRC_Add:
            return addSignal(argc-1, &argv[1]);
        break;

        case SRC_AddSep:
            return addSeparator(argc-1, &argv[1]);
        break;

        case SRC_Move:
            return moveSignal(
                    strtoul(argv[2], 0, 0),
                    strtoul(argv[3], 0, 0));
            break;

        case SRC_List:
            list = Tcl_NewListObj(0, 0);

            for (i=0; i<d_signals.length(); i++) {
                Tcl_ListObjAppendElement(interp, list, 
                        Tcl_NewStringObj(d_signals.idx(i)->id, -1));
            }
            Tcl_SetObjResult(interp, list);
        break;


        /*************************************************************
         * Cursor Commands
         *************************************************************/
        case SRC_AddCursor:
            cursor = cursorMgr->addCursor();
            Tcl_AppendResult(interp, cursor->getInstName(), 0);
        break;

        case SRC_GetCursor:
            if (argc > 3) {
                tolerance = strtoul(argv[3], 0, 0);
            } else {
                tolerance = 5;
            }

            cursor = cursorMgr->nearestCursor(strtoul(argv[2], 0, 0),
                    0, 0xFFFFFFFF, tolerance);

            if (cursor) {
                Tcl_AppendResult(interp, cursor->getInstName(), 0);
            } else {
                Tcl_AppendResult(interp, "", 0);
            }
            break;

        case SRC_GetCursorList:
            list = Tcl_NewListObj(0, 0);
            cursors = cursorMgr->getCursors();
            for (i=0; i<cursors->length(); i++) {
                if (cursors->idx(i)) {
                    Tcl_ListObjAppendElement(interp, list, 
                         Tcl_NewStringObj(cursors->idx(i)->getInstName(),-1));
                }
            }
            Tcl_SetObjResult(interp, list);
            break;



        default:
            Tcl_AppendResult(interp, "no command ", argv[1], 0);
            return TCL_ERROR;
        break;
    }

    return TCL_OK;
}

/******************************************************************
 * SigListUpdate
 *
 * Invokes all SigListUpdate callbacks on this SDBR instance
 ******************************************************************/
void SdbReader::SigListUpdate(void)
{
    setupCB(CBReq_SigList);
}

/******************************************************************
 * update()
 ******************************************************************/
void SdbReader::update(void)
{
    SigListUpdate();
}

/******************************************************************
 * addSeparator()
 ******************************************************************/
int SdbReader::addSeparator(Uint32 argc, Char **argv)
{
    SdbSignal        *sdbSig;
    Char              id[1024];
    Char             *name = argv[1];
    Tcl_HashEntry    *entry;
    Int32             itmp;

    sprintf(id, "%s.@Sig%d@", instName.value(), sigId++);

    sdbSig = new SdbSignal(this, interp, id, name, 0, 0, 0);
    d_signals.append(sdbSig); 
   
    entry = Tcl_CreateHashEntry(&d_sigHash, id, &itmp);
    Tcl_SetHashValue(entry, sdbSig);

    sdbSig->Separator(1);

    SigListUpdate();

    /**** NOTE: update causes TCL commands to be executed, which
     **** causes the result string to be null'd out
     ****/
    Tcl_AppendResult(interp, id, 0);

    return TCL_OK;
}

/******************************************************************
 * isSigSep()
 ******************************************************************/
int SdbReader::isSigSep(Char *sigId)
{
    SdbSignal        *sdbSig;
    Tcl_HashEntry    *entry;

    if ((entry = Tcl_FindHashEntry(&d_sigHash, sigId))) {
        sdbSig = (SdbSignal *)Tcl_GetHashValue(entry);
        Tcl_SetObjResult(interp, Tcl_NewIntObj(sdbSig->isSeparator()));
        return TCL_OK;
    }

    Tcl_AppendResult(interp, "No signal ", sigId, 0);
    return TCL_ERROR;
}

/******************************************************************
 * moveSignal()
 ******************************************************************/
int SdbReader::moveSignal(Uint32 srci, Uint32 desti)
{
    SdbSignal   *srcSig;
    Int32        i, src = srci, dest = desti;
    if (src >= d_signals.length() || dest >= d_signals.length()) {
        Tcl_AppendResult(interp, "Too few args", 0);
        return TCL_ERROR;
    }

    if (src == dest) {
        return TCL_OK;
    }

    /**** Procedure to move:
     **** - Get src ptr
     **** - shift dest->src
     **** - place src ptr in dest
     ****/
    srcSig = d_signals.idx(src);

    /**** Move dest->src  
     ****/
    if (src > dest) {
        for (i=src-1; i>=dest; i--) {
            d_signals.setIdx(i+1, d_signals.idx(i));
        }
    } else {
        /**** Move src->dest
         ****/
        for (i=src+1; i<=dest; i++) {
            d_signals.setIdx(i-1, d_signals.idx(i));
        }
    }
    d_signals.setIdx(dest, srcSig);

    SigListUpdate();
    return TCL_OK;
}

/******************************************************************
 * addSDB()
 ******************************************************************/
void SdbReader::addSDB(SigDB *sdb)
{
    SdbDesc *sdbd;
    Uint32   i;

    for (i=0; i<sdbdList->length(); i++) {
        if (sdb == sdbdList->idx(i)->sdb) {
            break;
        }
    }

    if (i == sdbdList->length()) {
        maxTimeValid = 0;
        sdbd = new SdbDesc(sdb, this);
        sdbdList->append(sdbd);

        sdb->addClient(this);

        /**** Add callbacks... ****/
        CallbackMgr_AddCb(CBTYPE_SDB_SIGDATA_UPDATE, 
                sdb->getInstName(),
                SdbReader::SDBR_SigDataUpdateCB, sdbd, 
                &sdbd->sdb_callback_id);
    }
}

/******************************************************************
 * removeSDB()
 ******************************************************************/
void SdbReader::removeSDB(SigDB *sdb)
{
    Uint32      i;
    SdbDesc    *sdbd;

    /**** Find the sdb desc ****/
    for (i=0; i<sdbdList->length(); i++) {
        if (sdb == sdbdList->idx(i)->sdb) {
            sdbd = sdbdList->idx(i);
            break;
        }
    }

    if (i < sdbdList->length()) {
        CallbackMgr_DeleteCb(sdbd->sdb_callback_id);
        sdb->removeClient(this);
        sdbdList->remove(sdbd);
        delete sdbd;
    } else {
        fprintf(stderr, "ERROR: SDB 0x%08x not a client of SDBR 0x%08x\n",
                sdb, this);
    }
}

/******************************************************************
 * appendCutBuffer()
 ******************************************************************/
int SdbReader::appendCutBuffer(Uint32 argc, char **argv)
{
    Uint32 i;
    Tcl_HashEntry    *entry;
    SdbSignal        *sig;

    for (i=1; i<argc; i++) {
        if (!(entry = Tcl_FindHashEntry(&d_sigHash, argv[i]))) {
            Tcl_AppendResult(interp, "no signal named ", argv[i], 0);
            return TCL_ERROR;
        } 
        sig = (SdbSignal *)Tcl_GetHashValue(entry);
        Tcl_DeleteHashEntry(entry);

        d_signals.remove(sig);
        d_cutBuffer.append(sig);
    }
    update();

    return TCL_OK;
}

/******************************************************************
 * putCutBuffer()
 *
 * Expect to be given an index into the signals array.
 ******************************************************************/
int SdbReader::putCutBuffer(Uint32 argc, char **argv)
{
    Uint32          i, i_idx;
    Int32           itmp;
    SdbSignal      *idx;
    Tcl_HashEntry  *entry;

    if (!strcmp(argv[1], "end")) {
        i_idx = d_signals.length();
    } else {
        if (!(entry = Tcl_FindHashEntry(&d_sigHash, argv[1]))) {
            Tcl_AppendResult(interp, "No signal ", argv[1], 0);
            return TCL_ERROR;
        }
        idx = (SdbSignal *)Tcl_GetHashValue(entry);

        for (i=0; i<d_signals.length(); i++) {
            if (d_signals.idx(i) == idx) {
                break;
            }
        }

        i_idx = i;
    }

    d_tmplist.empty();

    /**** Now, insert signals at the indicated spot... 
     **** - copy signals from idx on to tmplist
     **** - append cutBuf to signal list
     **** - append tmplist signals to signals list
     ****/
    for (i=i_idx; i<d_signals.length(); i++) {
        d_tmplist.append(d_signals.idx(i));
    }
    d_signals.setLength(i_idx);

    for (i=0; i<d_cutBuffer.length(); i++) {
        d_signals.append(d_cutBuffer.idx(i));
        entry = Tcl_CreateHashEntry(&d_sigHash, d_cutBuffer.idx(i)->id, &itmp);
        Tcl_SetHashValue(entry, d_cutBuffer.idx(i));
    }

    for (i=0; i<d_tmplist.length(); i++) {
        d_signals.append(d_tmplist.idx(i));
    }

    d_tmplist.empty();
    d_cutBuffer.empty();
    update();

    return TCL_OK;
}

/******************************************************************
 * addSignal()
 *
 * Adds a signal to this SDBR and (if it doesn't already exist in
 * the SDB) to the SDB.
 *
 * NOTE: argc/argv are offsets of the 'add' command. Therefore,
 * argv[0] == 'add'
 ******************************************************************/
int SdbReader::addSignal(Uint32 argc, Char **argv)
{
    Char               *path = argv[1];
    Vector<DFIOTrace>  *traceVect = 0;
    DFIOTrace          *trace;
    Uint32              i;
    SdbSignal          *sdbSig;
    Char                id[1024];
    Tcl_Obj            *list;
    SigDB              *use_sdb = sdb, *sdbp;
    Tcl_HashEntry      *entry;
    Int32               itmp;

    /**** Look for options... May specify that a different SDB is the
     **** source of the signal data...
     ****/
    for (i=2; i<argc; i++) {
        if (argv[i][0] == '-') {
            if (String::equal(argv[i], "-sdb")) {
                i++;
                use_sdb = (SigDB *)WidgetMgr_GetObjHandle(
                        interp, "SDB", argv[i]);
                if (!use_sdb) {
                    Tcl_AppendResult(interp, "unknown SDB ", argv[i], 0);
                    return TCL_ERROR;
                }
            } else {
                Tcl_AppendResult(interp, "unknown 'add' option ", argv[i], 0);
                return TCL_ERROR;
            }
        }
    }

    if (!use_sdb) {
        Tcl_AppendResult(interp, "no sdb specified", 0);
        return TCL_ERROR;
    }

    addSDB(use_sdb);

    if (use_sdb->get_resolution() < resolution) {
        resolution = use_sdb->get_resolution();
    }
    
    traceVect = use_sdb->addSignal(path);

    list = Tcl_NewListObj(0, 0);
    for (i=0; i<traceVect->length(); i++) {
        trace = traceVect->idx(i);

        sprintf(id, "%s.@Sig%d@", instName.value(), sigId++);
        sdbSig = new SdbSignal(this, interp, id, trace->get_name(), 
                trace->get_msb(), trace->get_lsb(), trace);
        sdbSig->setSdbName(use_sdb->getSdbId());
        trace->AddClient(sdbSig);
        d_signals.append(sdbSig); 

        entry = Tcl_CreateHashEntry(&d_sigHash, id, &itmp);
        Tcl_SetHashValue(entry, sdbSig);

        Tcl_ListObjAppendElement(interp, list, 
                Tcl_NewStringObj(id, -1));
    }

    /**** Lookout... May modify the TCL result... ****/
    SigListUpdate();

    Tcl_SetObjResult(interp, list);

    return TCL_OK;
}

/******************************************************************
 *
 ******************************************************************/
char *SdbReader::nextSignal()
{
    sprintf(d_idBuf, "%s.@Sig%d@", instName.value(), sigId++);

    return d_idBuf;
}

/******************************************************************
 * addSignal()
 ******************************************************************/
void SdbReader::addSignal(SdbSignal *sig)
{
    Tcl_HashEntry         *entry;
    Int32                  itmp;

    d_signals.append(sig);

    entry = Tcl_CreateHashEntry(&d_sigHash, sig->id, &itmp);
    Tcl_SetHashValue(entry, sig);

    SigListUpdate();
}

/******************************************************************
 * delSignal()
 ******************************************************************/
void SdbReader::delSignal(SdbSignal *sig)
{
    Uint32         i, sigIdx;
    Tcl_HashEntry *entry;

    d_signals.remove(sig);

    SigListUpdate();

    if ((entry = Tcl_FindHashEntry(&d_sigHash, sig->id))) {
        Tcl_DeleteHashEntry(entry);
    }
}

/******************************************************************
 * sigIndex()
 ******************************************************************/
Uint32 SdbReader::sigIndex(SdbSignal *sig)
{
    Uint32    i;

    for (i=0; i<d_signals.length(); i++) {
        if (sig == d_signals.idx(i)) {
            return i;
        }
    }

    return 0;
}

/******************************************************************
 * getSigList()
 ******************************************************************/
void SdbReader::getSigList(void)
{
    Uint32  i;
    Char    buf[64];

    for (i=0; i<d_signals.length(); i++) {
        sprintf(buf, "%s ", d_signals.idx(i)->id);
        Tcl_AppendResult(interp, buf, 0);
    }
}

/******************************************************************
 * findEdge()
 ******************************************************************/
int SdbReader::findEdge(Char *sigId, Uint32 time, Uint32 fwd, Uint32 *nextTime)
{
    SdbSignal            *sdbSig; 
    Vector<DFIOValChg>   *vals;
    Tcl_HashEntry        *entry;


    if (!(entry = Tcl_FindHashEntry(&d_sigHash, sigId))) {
        fprintf(stderr, "ERROR: No signal w/id %s\n", sigId);
        return TCL_ERROR;
    } else {
        sdbSig = (SdbSignal *)Tcl_GetHashValue(entry);
    }

    vals = sdbSig->dfioTrace->getValue(time, time, 
            (fwd)?DFIOTrace::GVF_OnePlus:DFIOTrace::GVF_OneMinus);

    if (vals->length() > 1) {
        if (fwd) {
            *nextTime = vals->idx(1)->changeTime;
        } else {
            *nextTime = vals->idx(0)->changeTime;
        }
    } else {
        *nextTime = time;
    }

    delete vals;

    return TCL_OK;
}

/*****************************************************************************
 *****************************************************************************
 ****                      Tcl Interface Functions
 *****************************************************************************
 *****************************************************************************/

/******************************************************************
 * SdbReader_InstCmd()
 ******************************************************************/
static int SdbReader_InstCmd(
    ClientData        clientData,
    Tcl_Interp       *interp,
    int               argc,
    char            **argv)
{
    SdbReader    *sdbReader = (SdbReader *)clientData;

    return sdbReader->InstCmd(argc, argv);
}

/******************************************************************
 * TclCmd_SdbReader()
 *
 * 
 * Command is of format:
 * sdb_reader <sigDb>.<readerName> [options]
 ******************************************************************/
static int TclCmd_SdbReader(ClientData        clientData,
                            Tcl_Interp       *interp,
                            int               argc,
                            char            **argv
                           )
{
    SdbReader *sdbr;

    if (argc < 2) {
        Tcl_AppendResult(interp, 
            "wrong # args... sdbr <path> [options]", NULL);
        return TCL_ERROR;
    }

    sdbr = new SdbReader(interp, argc, argv);

    if (!sdbr->okay) {
        return TCL_ERROR;
    }

    return TCL_OK;
}

/******************************************************************
 * getMaxTime()
 ******************************************************************/
Uint32 SdbReader::getMaxTime()
{
    Uint32    i, tmp;
    SigDB    *sdbp;

    if (maxTimeValid) {
        return maxTime;
    }

    maxTime = 0;
    
    /**** Otherwise, go through each SDB... ****/
    for (i=0; i<sdbdList->length(); i++) {
        tmp = sdbdList->idx(i)->sdb->getMaxTime();

        if (tmp > maxTime) {
            maxTime = tmp;
        }
    }

    maxTimeValid = 1;

    return maxTime;
}

/******************************************************************
 * setupCB()
 ******************************************************************/
void SdbReader::setupCB(CBReq req)
{
    if (!d_cbReqs) {
        Tcl_DoWhenIdle(&SdbReader::IdleProc, this);
    }

    d_cbReqs |= req;
}

/******************************************************************
 * getConfigSpec()
 ******************************************************************/
Tk_ConfigSpec *SdbReader::getConfigSpec()
{
    static Tk_ConfigSpec   sdbrSigSpec[] = {
        {TK_CONFIG_STRING, "-sdb", "sdb", "SDB",
            (char *)NULL, Tk_Offset(SdbReader, sdbName), 0},
        {TK_CONFIG_END, (char *)NULL, (char *)NULL, (char *)NULL,
            (char *)NULL, 0, 0}
    };
    return sdbrSigSpec;
}

/******************************************************************
 * IdleProc()
 ******************************************************************/
void SdbReader::IdleProc(ClientData clientData)
{
    SdbReader *sdbr = (SdbReader *)clientData;

    if (sdbr->d_cbReqs & CBReq_SigList) {
        CallbackMgr_Invoke(CBTYPE_SDBR_SIGLIST_UPDATE,
                sdbr->instName.value(), 0, 0);
    }

    sdbr->d_cbReqs = 0;
}

/******************************************************************
 * SdbReader_Init()
 * Registers the SdbReader type with the WidgetMgr.
 ******************************************************************/
extern "C" int SdbReader_Init(Tcl_Interp *interp)
{
    WidgetMgr_AddType(interp, WIDGET_TYPE_SDBR);

    CallbackMgr_AddCbType(CBTYPE_SDBR_SIGLIST_UPDATE);
    CallbackMgr_AddCbType(CBTYPE_SDBR_SIGDATA_UPDATE);

    Tcl_CreateCommand(interp, "sdbr", 
            (Tcl_CmdProc *)TclCmd_SdbReader, NULL, NULL);

    return TCL_OK;
}

char SdbReader::d_idBuf[1024];
