/*****************************************************************************
 *                           WidgetManager.cc
 * Author: Matthew Ballance 
 * Desc:   Tracks the native widgets created. The names of the native 
 *         widgets are also tracked... 
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *****************************************************************************/
#include "WidgetManager.h"
#include "CmdSwitcher.h"

#include <unistd.h>

class WidgetType {
public:
    String            typeName;
    Tcl_HashTable     hashTable;

    WidgetType(const Char *type);
    voidPtr GetInst(const Char *instName);
    void    SetInst(const Char *instName, voidPtr inst);
    const Char *InstName(voidPtr inst);
};

/********************************************************************
 * WidgetType()
 ********************************************************************/
WidgetType::WidgetType(const Char *type) : 
    typeName(type)
{
    Tcl_InitHashTable(&hashTable, TCL_STRING_KEYS);
}

/********************************************************************
 * GetInst
 ********************************************************************/
voidPtr WidgetType::GetInst(const Char *instName)
{
    Tcl_HashEntry    *hashEntry;

    if (!(hashEntry = Tcl_FindHashEntry(&hashTable, instName))) {
        return 0;
    }

    return Tcl_GetHashValue(hashEntry);
}

/********************************************************************
 * SetInst()
 ********************************************************************/
void WidgetType::SetInst(const Char *instName, voidPtr inst)
{
    Tcl_HashEntry    *hashEntry;
    int               newPtr;

    hashEntry = Tcl_CreateHashEntry(&hashTable, instName, &newPtr);
    Tcl_SetHashValue(hashEntry, inst);
}

/********************************************************************
 * InstName()
 ********************************************************************/
const Char *WidgetType::InstName(voidPtr inst)
{
    Tcl_HashSearch    search;
    Tcl_HashEntry    *hashEntry;
    voidPtr           eInst;
    Char             *hashKey;

    /**** Cycle through each instance in the table... ****/

    hashEntry = Tcl_FirstHashEntry(&hashTable, &search);

    while (hashEntry) {
        eInst = Tcl_GetHashValue(hashEntry);
        if (inst == eInst) {
            hashKey = Tcl_GetHashKey(&hashTable, hashEntry);
            if (strcmp("@@default@@", hashKey) != 0) {
                return hashKey;
            }
        }
        hashEntry = Tcl_NextHashEntry(&search);
    }
    return 0;
}

static Vector<WidgetType>     *Globl_WidgetTypes = 0;

/*********************************************************
 * WidgetMgr_GetMgr()
 *********************************************************/
static Vector<WidgetType> *WidgetMgr_GetMgr()
{
    if (!Globl_WidgetTypes) {
        Globl_WidgetTypes = new Vector<WidgetType>();
    }

    return Globl_WidgetTypes;
}

/*********************************************************
 * PrvWidgetMgr_GetType()
 *********************************************************/
static WidgetType *PrvWidgetMgr_GetType(const Char *typeName)
{
    Uint32        i;
    WidgetType   *type = 0;
    Vector<WidgetType>   *WidgetTypes = WidgetMgr_GetMgr();

    for (i=0; i<WidgetTypes->length(); i++) {
        type = WidgetTypes->idx(i);

        if (type->typeName.equal(typeName)) {
            break;
        }
        type = 0;
    }

    return type;
}

/*********************************************************
 * WidgetMgr_AddType()
 *********************************************************/
int WidgetMgr_AddType(
    Tcl_Interp     *interp,
    const Char     *typeName
    )
{
    WidgetType    *widType = new WidgetType(typeName);
    Vector<WidgetType>   *WidgetTypes = WidgetMgr_GetMgr();

    WidgetTypes->append(widType);

    return TCL_OK;
}

/*********************************************************
 * WidgetMgr_AddInst()
 *********************************************************/
int WidgetMgr_AddInst(
    Tcl_Interp     *interp,
    const Char     *widgetType,
    const Char     *widgetInstName,
    voidPtr         widgetPtr
    )
{
    Uint32         nameLen = strlen(widgetType), i; 
    int            newPtr;
    Tcl_HashEntry *inst;
    WidgetType    *type;

    type = PrvWidgetMgr_GetType(widgetType);

    if (!type) {
        fprintf(stderr, "\tType %s doesn't exist\n", widgetType);
        return 0;
    } else {
        type->SetInst(widgetInstName, widgetPtr);
    }

    return 1;
}

/********************************************************************
 * WidgetMgr_DelInst()
 ********************************************************************/
int WidgetMgr_DelInst(
        Tcl_Interp        *interp,
        const char        *widgetType,
        const char        *widgetInst)
{
    Tcl_HashEntry   *entry;
    WidgetType      *widType = 0;
    Uint32           i;
    Vector<WidgetType>   *WidgetTypes = WidgetMgr_GetMgr();

    for (i=0; i<WidgetTypes->length(); i++) {
        if (WidgetTypes->idx(i)->typeName.equal(widgetType)) {
            break;
        }
    }

    if (i == WidgetTypes->length()) {
        return 0;
    } else {
        widType = WidgetTypes->idx(i);
    }

    if (!(entry = Tcl_FindHashEntry(&widType->hashTable, widgetInst))) {
        return 0;
    }

    Tcl_DeleteHashEntry(entry);

    return 1;
}

typedef enum {
    WMC_NameObj = 1,
    WMC_Default,
    WMC_NumCmds
};

static CmdSwitchStruct    widget_mgr_cmds[] = {
    {"name_obj",          WMC_NameObj     },
    {"default",           WMC_Default     },
    {"",                  0}
};


/*********************************************************
 * WidgetMgr_TclCmd()
 *
 *********************************************************/
static int WidgetMgr_TclCmd(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        char            **argv)
{
    Int32          cmd, dummy;
    Char          *typeName, *newName;
    const Char    *instName;
    Uint32         i;
    WidgetType    *type;
    Tcl_HashEntry *hashEntry, *newEntry;
    voidPtr        instHndl;
    Vector<WidgetType>   *WidgetTypes = WidgetMgr_GetMgr();

    if (argc < 2) {
        Tcl_AppendResult(interp, "too few args ", 0);
        CmdSwitch_AppendAll(interp, widget_mgr_cmds);
        return TCL_ERROR;
    }

    cmd = CmdSwitch(widget_mgr_cmds, argv[1]);
    switch (cmd) {
        /**** widget_mgr name_obj <type> <obj_inst> <obj_name> ****/
        case WMC_NameObj:
            if (argc < 5) {
                Tcl_AppendResult(interp, "too few args", 0);
                return TCL_ERROR;
            }

            typeName = argv[2];
            instName = argv[3];
            newName  = argv[4];

            /**** Find type-desc ****/
            for (i=0; i<WidgetTypes->length(); i++) {
                type = WidgetTypes->idx(i);

                if (type->typeName.equal(typeName)) {
                    break;
                }
                type = 0;
            }

            if (!type) {
                Tcl_AppendResult(interp, "cannot find type ", typeName, 0);
                return TCL_ERROR;
            }

            /**** Find the existing instance... ****/
            if (!(hashEntry = Tcl_FindHashEntry(&type->hashTable,
                            instName))) {
                Tcl_AppendResult(interp, "existing inst ", instName, 
                        " doesn't exist", 0);
                return TCL_ERROR;
            }

            /**** Add the new name... ****/
            if (!(newEntry = Tcl_FindHashEntry(&type->hashTable, newName))) {
                newEntry = Tcl_CreateHashEntry(&type->hashTable, 
                        newName, &dummy);
            }

            Tcl_SetHashValue(newEntry, Tcl_GetHashValue(hashEntry));
            break;

        /**** default <type>
         **** default <type> instance
         ****/
        case WMC_Default:
            if (argc < 3) {
                Tcl_AppendResult(interp, "too few arguments", 0);
                return TCL_ERROR;
            }

            if (!(type = PrvWidgetMgr_GetType(argv[2]))) {
                Tcl_AppendResult(interp, "no type ", argv[2], 0);
                return TCL_ERROR;
            }


            /**** Set default... ****/
            if (argc == 4) {
                if (!(instHndl = type->GetInst(argv[3]))) {
                    Tcl_AppendResult(interp, "no instance ", argv[3], 0);
                }
                type->SetInst("@@default@@", instHndl);
            } else {
                if (!(instHndl = type->GetInst("@@default@@"))) {
                    Tcl_AppendResult(interp, "no default instance", 0);
                }
                /**** Now, need to cycle through each instance in the hash... 
                 ****/
                if (!(instName = type->InstName(instHndl))) {
                    Tcl_SetObjResult(interp, Tcl_NewStringObj("", -1));
                    return TCL_OK;
                }

                Tcl_SetObjResult(interp, Tcl_NewStringObj(instName, -1));
            }
            break;

        default:
            Tcl_AppendResult(interp, "Unknown sub-command", 0);
            return TCL_ERROR;
            break;
    }

    return TCL_OK;
}

/*********************************************************
 * WidgetMgr_Init()
 *********************************************************/
int WidgetMgr_Init(
    Tcl_Interp     *interp
    )
{
    Vector<WidgetType>   *WidgetTypes = WidgetMgr_GetMgr();

    Tcl_CreateCommand(interp, "widget_mgr", 
            (Tcl_CmdProc *)WidgetMgr_TclCmd, 0, 0);

    return TCL_OK;
}

/*********************************************************
 * WidgetMgr_GetObjHandle()
 *********************************************************/
voidPtr WidgetMgr_GetObjHandle(
    Tcl_Interp     *interp,
    const Char     *typeName,
    const Char     *instName
    )
{
    Uint32 i, nameLen = strlen(typeName);
    Tcl_HashEntry *entry;
    WidgetType    *widType = 0;
    ClientData     objPtr = 0;
    Vector<WidgetType>   *WidgetTypes = WidgetMgr_GetMgr();

    for (i=0; i<WidgetTypes->length(); i++) {
        if (WidgetTypes->idx(i)->typeName.equal(typeName, nameLen)) {
            break;
        }
    }

    if (i == WidgetTypes->length()) {
        return 0;
    } else {
        widType = WidgetTypes->idx(i);
    }

    entry = Tcl_FindHashEntry(&widType->hashTable, instName);

    if (entry == 0) {
        return 0;
    } else {
        objPtr = Tcl_GetHashValue(entry);
    } 

    return objPtr; 
}



/*********************************************************
 * WidgetMgr_GetAllObjHandles()
 *********************************************************/
Vector<void> *WidgetMgr_GetObjHAndle(
    Tcl_Interp        *interp,
    const Char        *typeName
    )
{
    Uint32 i, nameLen = strlen(typeName);
    Tcl_HashEntry  *entry;
    Tcl_HashSearch  search;
    WidgetType    *widType;
    ClientData     objPtr = 0;
    Vector<void>  *retVect;
    Vector<WidgetType>   *WidgetTypes = WidgetMgr_GetMgr();

    for (i=0; i<WidgetTypes->length(); i++) {
        if (WidgetTypes->idx(i)->typeName.equal(typeName, nameLen)) {
            break;
        }
    }

    retVect = new Vector<void>();

    if (i == WidgetTypes->length()) {
        return retVect;
    } else {
        widType = WidgetTypes->idx(i);
    }

    entry = Tcl_FirstHashEntry(&widType->hashTable, &search);
    while (entry) {
        retVect->append((voidPtr)Tcl_GetHashValue(entry));
        entry = Tcl_NextHashEntry(&search); 
    }

    return retVect;
}


