#****************************************************************************
#*                          help_browser.tcl
#*
#* Author: Matthew Ballance
#* Desc:   
#* <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
#*
#*    This source code is free software; you can redistribute it
#*    and/or modify it in source code form under the terms of the GNU
#*    General Public License as published by the Free Software
#*    Foundation; either version 2 of the License, or (at your option)
#*    any later version.
#*
#*    This program is distributed in the hope that it will be useful,
#*    but WITHOUT ANY WARRANTY; without even the implied warranty of
#*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*    GNU General Public License for more details.
#*
#*    You should have received a copy of the GNU General Public License
#*    along with this program; if not, write to the Free Software
#*    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#*
#* </Copyright>
#****************************************************************************

package provide help_browser 1.0

namespace eval HelpBrowser {
    namespace export HelpBrowser
    variable  DocumentList ""

    variable biggray [image create photo biggray -data {
    R0lGODdhPAA+APAAALi4uAAAACwAAAAAPAA+AAACQISPqcvtD6OctNqLs968+w+G4kiW5omm
    6sq27gvH8kzX9o3n+s73/g8MCofEovGITCqXzKbzCY1Kp9Sq9YrNFgsAO///}]

    variable smgray  [image create photo smgray -data {
    R0lGODdhOAAYAPAAALi4uAAAACwAAAAAOAAYAAACI4SPqcvtD6OctNqLs968+w+G4kiW5omm
    6sq27gvH8kzX9m0VADv/}]

    variable nogifbig [image create photo nogifbig -data {
    R0lGODdhJAAkAPEAAACQkADQ0PgAAAAAACwAAAAAJAAkAAACmISPqcsQD6OcdJqKM71PeK15
    AsSJH0iZY1CqqKSurfsGsex08XuTuU7L9HywHWZILAaVJssvgoREk5PolFo1XrHZ29IZ8oo0
    HKEYVDYbyc/jFhz2otvdcyZdF68qeKh2DZd3AtS0QWcDSDgWKJXY+MXS9qY4+JA2+Vho+YPp
    FzSjiTIEWslDQ1rDhPOY2sXVOgeb2kBbu1AAADv/}]

    variable nogifsm  [image create photo nogifsm -data {
    R0lGODdhEAAQAPEAAACQkADQ0PgAAAAAACwAAAAAEAAQAAACNISPacHtD4IQz80QJ60as25d
    3idKZdR0IIOm2ta0Lhw/Lz2S1JqvK8ozbTKlEIVYceWSjwIAO///}]

}

proc ivi_file_join args {
    global tcl_platform

    set res [eval file join $args]

    if {$tcl_platform(platform) == "windows"} {
        regsub { } $res {\\ } res
    }

    return $res
}

#********************************************************************
#* SelectCB
#********************************************************************
proc HelpBrowser::SelectCB {path args} {
    upvar #0 $path data
    set file [$data(w:doc_tree) itemcget [lindex $args 1] -data]
    HelpBrowser::LoadFile $path $file
}

#********************************************************************
#* 
#********************************************************************
proc pad_cb {path W} {
    upvar #0 $path data
    $data(w:hsbpad) configure -width [winfo reqwidth $W]
}

proc t_proc args {

    puts "args = $args"
}

#********************************************************************
#* HelpBrowser
#********************************************************************
proc HelpBrowser::HelpBrowser {path args} {
    global $path

    frame $path
    bind $path <Destroy> [list HelpBrowser::Close $path]

    array set $path {_dummy _dummy}
    upvar #0 $path data

    #****************************************************************
    #* Create the main menu-bar
    #****************************************************************
    set data(w:menu) [MenuMgr::create menubar HelpBrowser::Main $path.mb \
        -var [list w $path]]
    regsub {\.help$} $path {} toplev

    $toplev config -menu $data(w:menu)

#    set bframe       [frame $path.bframe]

    #****************************************************************
    #* PanedWindow
    #****************************************************************
    set data(w:paned) [PanedWindow $path.paned]
    set data(w:tree_pane) [$data(w:paned) add -weight 1]
    set data(w:html_pane) [$data(w:paned) add -weight 3]

    pack $data(w:paned) -expand yes -fill both

    #****************************************************************
    #* Create tree
    #****************************************************************
    set data(w:tree_scroll) [ScrollContainer $data(w:tree_pane).scroll]
    set frame [$data(w:tree_scroll) subwidget frame]
        
#    set data(w:doc_tree) [Tree $frame.doc_tree \
#        -relief sunken -borderwidth 1 -width 24 -highlightthickness 0 \
#        -redraw 0 -background white]

    set data(w:doc_tree) [tree_widget $frame.doc_tree \
         -redraw 0 -background white -browsecmd \
         [list HelpBrowser::SelectCB $path]]


    set vsb [$data(w:tree_scroll) subwidget vsb]
    set hsb [$data(w:tree_scroll) subwidget hsb]

    $data(w:doc_tree) configure -yscrollcommand "$vsb set"
    $data(w:doc_tree) configure -xscrollcommand "$hsb set"

    $vsb configure -command "$data(w:doc_tree) yview"
    $hsb configure -command "$data(w:doc_tree) xview"

    HelpBrowser::SetupTree $path

#    $data(w:doc_tree) bindText <Double-ButtonPress-1>  \
#        [list HelpBrowser::SelectCB $path]
#    bind $data(w:doc_tree) <Double-ButtonPress-1>  \
#        [list HelpBrowser::SelectCB $path]

    $data(w:doc_tree) configure -redraw 1

    pack $data(w:doc_tree) -expand yes -fill both
    pack $data(w:tree_scroll) -expand yes -fill both

    #****************************************************************
    #* HTML Pane
    #****************************************************************
    set hbframe      [frame $data(w:html_pane).hbframe]
    set hframe       [frame $hbframe.hframe]
    set hsbframe     [frame $hbframe.hsbframe]

    set data(w:html) [html $hframe.html \
        -relief groove \
        -bg white \
        -width 320 \
        -imagecommand [list HelpBrowser::ImageCmd $path] \
    ]

    set data(w:vsb)  [scrollbar $hframe.vsb -orient vertical \
        -command [list $data(w:html) yview]]


    set data(w:hsbpad) [frame $hsbframe.hsbpad]
    set data(w:hsb)  [scrollbar $hsbframe.hsb -orient horizontal \
        -command [list $data(w:html) xview]]

    bind $data(w:vsb) <Configure> "pad_cb $path %W"

    pack $data(w:hsbpad) -side right -fill both
    pack $data(w:hsb) -side right -fill x

    pack $hsbframe -side bottom -fill x
    pack $hframe   -side bottom -fill both -expand yes

    $data(w:html) configure \
        -yscrollcommand [list $data(w:vsb) set] \
        -xscrollcommand [list $data(w:hsb) set]

    pack $data(w:vsb)  -side right -fill y
    pack $data(w:html) -side right -fill both -expand yes

    pack $data(w:hsb) -side bottom -fill x
    pack $hbframe -side bottom -fill both -expand yes

#    pack $bframe -expand yes -fill both

    bind $data(w:html).x <1> [list HelpBrowser::HrefBinding $path %x %y]

    set data(LastFile) ""
    set data(showImages) 1

    HelpBrowser::SetupMain $path


    rename $path ::$path:cmd
    proc ::$path { cmd args } \
        "return \[eval HelpBrowser::cmd $path \$cmd \$args\]"

    return $path
}

#********************************************************************
#* Close
#********************************************************************
proc HelpBrowser::Close {path} {
    global $path
    destroy $path
    unset $path
}

#********************************************************************
#* MoveBigImage
#********************************************************************
proc HelpBrowser::MoveBigImage {path b} {
    upvar #0 $path data

    if {![info exists data(BigImages:$b)]} return
    $b copy $data(BigImages:$b)
    image delete $data(BigImages:$b)
    unset data(BigImages:$b)
    update
}

#********************************************************************
#* ImageCmd
#********************************************************************
proc HelpBrowser::ImageCmd {args} {
    set path [lindex $args 0]
    upvar #0 $path data
    variable smgray 

    set args [lrange $args 1 end]

    if {!$data(showImages)} {
        return $smgray
    }

    set fn [lindex $args 0]
    if {[info exists data(OldImages:$fn)]} {
        set data(Images:$fn) $data(OldImages:$fn)
        unset data(OldImages:$fn)
        return $data(Images:$fn)
    }

    if {[catch {image create photo -file $fn} img]} {
        return $smgray
    }
    if {[image width $img]*[image height $img]>20000} {
        set b [image create photo -width [image width $img] \
           -height [image height $img]]
        set data(BigImages:$b) $img
        set img $b
        after idle [list HelpBrowser::MoveBigImage $path $b]
    }
    set data(Images:$fn) $img
    return $img
}

#********************************************************************
#* SetupTreeHelper
#*
#* sub_list is
#* [<list_of_sections>]
#*
#* section is:
#* [<sect_info_list> <sub_sect_list>]
#*
#* sect_info_list is:
#* [<file_sect_spec> <sect_name> <list_of_sections>]
#********************************************************************
proc HelpBrowser::SetupTreeHelper {path space p_node sub_list} {
    upvar #0 $path data

    foreach sect_info $sub_list {
        set sect_file [lindex $sect_info 0]
        set sect_name [lindex $sect_info 1]

        set node_id [file tail $sect_file]
        regsub {#} $node_id {_} node_id

        $data(w:doc_tree) insert end $p_node "$p_node/$node_id" \
            -text $sect_name -data $sect_file

        HelpBrowser::SetupTreeHelper $path "    $space" "$p_node/$node_id" \
            [lindex $sect_info 2]
    }
}

#********************************************************************
#* SetupTree
#*
#* A document is:
#* [<doc_id> <doc_data_list>]
#*
#* doc_data_list is:
#* [<doc_info_list> <doc_data_list>]
#*
#* doc_info_list is:
#* [<doc_file_name> <doc_title>]
#*
#********************************************************************
proc HelpBrowser::SetupTree {path} {
    upvar #0 $path data
    variable DocumentList

    foreach doc $DocumentList {
        set doc_id [lindex $doc 0]
        set top_list [lindex $doc 1]
        set info_list [lindex $top_list 0]

        set doc_title     [lindex $info_list 1]
        set doc_filename  [lindex $info_list 0]

        $data(w:doc_tree) insert end root $doc_id \
            -text $doc_title -data $doc_filename
        HelpBrowser::SetupTreeHelper $path "    " $doc_id [lindex $top_list 1]
    }
}

#********************************************************************
#* history
#********************************************************************
proc HelpBrowser::history {cmd title location} {
    upvar #0 $path data

}

#********************************************************************
#* SetupMain
#********************************************************************
proc HelpBrowser::SetupMain {path} {
    upvar #0 $path data
    global   IVI_HOME

    HelpBrowser::Clear $path

    HelpBrowser::LoadFile $path \
        [ivi_file_join $IVI_HOME modules help_browser doc_main.html]
}

#********************************************************************
#* Clear
#********************************************************************
proc HelpBrowser::Clear {path} {
    upvar #0 $path data

    $data(w:html) clear

    #**** TODO :: Clear locally-help images...
}

#********************************************************************
#* LoadFile
#*
#*
#* NOTE: $name may be either a file path or a file path with a
#*       document offset appended...
#********************************************************************
proc HelpBrowser::LoadFile {path name} {
    upvar #0 $path data

    set namel [split $name {#}]
    set filename [lindex $namel 0]

    if {$filename == "MasterIndex"} {
        HelpBrowser::LoadMasterIndex $path
    } else {
        if {[catch {open $filename r} fp]} {
            tk_messageBox -icon error -message $fp -type ok
            set content {}
        } else {
            fconfigure $fp -translation binary
            set content [read $fp [file size $filename]]
            close $fp
        }

        if {$content == ""} {
            return
        }

        HelpBrowser::Clear $path

        set data(LastFile) $filename
        $data(w:html) configure -base $filename
        $data(w:html) parse $content

        if {[llength $namel] > 1} {
            update
            $data(w:html) yview [lindex $namel 1]
        }
    }
}

#********************************************************************
#* HrefBinding
#********************************************************************
proc HelpBrowser::HrefBinding {path x y} {
    upvar #0 $path data

    set new [$data(w:html) href $x $y]

    if {$new != ""} {
        set pattern "$data(LastFile)#"

        set len [string length $pattern]
        incr len -1
        if {[string range $new 0 $len] == $pattern} {
            incr len
            update
            $data(w:html) yview [string range $new $len end]
        } else {
            HelpBrowser::LoadFile $path $new
        }
    }
}

#********************************************************************
#* ScanStructure
#*
#* Look for <LI> tags.
#* Create a node entry for each <LI> tag.
#* If a <UL> tag is found, recurse deeper
#* If a </UL> tag is found, return the current list...
#********************************************************************
proc HelpBrowser::ScanStructure {space fp doc_base} {

    set contents ""
    set line_list ""

    set scanning 1
    while {$scanning} {
        if {[gets $fp line] <= 0} {
            if {[eof $fp]} {
                break
            }
        }

        #**** Found a <LI> line... Formats as:
        #**** <LI><A NAME="STRING" HREF="LinkSpec">NodeTitle</A>
        if {[regexp {^<LI>} $line]} {
            if {$line_list != ""} {
                lappend contents $line_list
                set line_list ""
            }

            regsub {<LI><A NAME=".*" HREF="} $line {} line2

            #**** line2 now holds: LinkSpec">NodeTitle</A>
            regsub {">.*</A>.*} $line2 {} link_spec

            regsub {.*">} $line2 {} node_title_tmp
            #**** node_title_tmp contains: NodeTitle</A>

            regsub {</A>} $node_title_tmp {} node_title

            set line_list [list [ivi_file_join $doc_base $link_spec] $node_title]
        } elseif {[regexp {^</UL>} $line]} {
            #**** End of a section, return what we've got...
            break
        } elseif {[regexp {^<UL>} $line]} {
            lappend line_list \
                [HelpBrowser::ScanStructure "    $space" $fp $doc_base]

            lappend contents $line_list
            set line_list ""
        }
    }

    if {$line_list != ""} {
        lappend contents $line_list
    }

    return $contents
}

#********************************************************************
#* ScanTocFile
#********************************************************************
proc HelpBrowser::ScanTocFile {filename} {
    set limit 100

    set fp [open $filename r]

    set doc_elems [file split $filename]
    set doc_base [eval ivi_file_join [lrange $doc_elems 0 \
        [expr [llength $doc_elems] - 2]]]

    set file_title ""
    set file_structure ""

    #**** Look for a <TITLE> tag...
    for {set i 0} {$i < $limit} {incr i} {
        if {[gets $fp line] <= 0} {
            if {[eof $fp]} {
                break
            }
        }

        #**** The format of the TITLE line will probably be:
        #**** <TITLE>TitleString - Table of Contents</TITLE>
        if {[regexp {^<TITLE>} $line]} {
            regsub {<TITLE>} $line {} line
            regsub {</TITLE>} $line {} line

            regsub { - Table of Contents} $line {} line

            set file_title $line
            break
        }
    }

    lappend file_structure [list $filename $file_title]

    #**** Now, try to build a hierarchical structure for the entries
    #**** in this file...
    set scanning 1
    while {$scanning == 1} {
        if {[gets $fp line] <= 0} {
            if {[eof $fp]} {
                break
            }
        }

        #**** When we find a <UL> tag, we enter a new hierarchy level...
        if {[regexp {^<UL>} $line]} {
            lappend file_structure \
                [HelpBrowser::ScanStructure "    " $fp $doc_base]
            break
        }
    }

    return $file_structure
}

#********************************************************************
#* LocateDocuments
#********************************************************************
proc HelpBrowser::LocateDocuments args {
    variable DocumentList
    global   IVI_HOME
    global   tcl_platform

    set docdir [ivi_file_join $IVI_HOME doc]
    set docdirs ""

    #**** Find all directories in the doc-root

    set join_path  [ivi_file_join $IVI_HOME doc *]
    if [catch {set files [glob -type d $join_path]}] {
        ivi_puts "ERROR: It appears that no documentation is installed"
        ivi_puts "    Expecting documentation in [ivi_file_join $IVI_HOME doc]"
        set files ""
        return
    }

    foreach node $files {
        if {[file isdirectory $node]} {
            lappend docdirs $node
        }
    }

    #**** Now, look for *_toc.html files in these directories
    set toc_files ""
    foreach dir $docdirs {
        lappend toc_files [glob [ivi_file_join $dir *_toc.html]]
    }

    #**** Finally, we have a list of the available documents... Now,
    #**** we need to find out what the docs are titled and build an
    #**** array of DocTitle/DocPath pairs...
    foreach toc_file $toc_files {
        #**** Scan this file for a <TITLE> tag...
        if {$tcl_platform(platform) == "windows"} {
            regsub { } $toc_file {\\ } toc_file
        }
        set structure [HelpBrowser::ScanTocFile $toc_file]

        set file_name [file tail $toc_file]
        regsub {_toc.html} $file_name {} file_name

        lappend DocumentList [list $file_name $structure]
    }
}

#********************************************************************
#* help_browser_Init
#********************************************************************
proc help_browser_Init args {
    global IVI_HOME
    global auto_path

    lappend auto_path [ivi_file_join $IVI_HOME lib Img1.2]

    if [catch {package require Img} res] {
        ivi_puts "WARNING: Img library couldn't be found ($res)"
        ivi_puts "    This means that images will not be shown"
    }


    MenuMgr::load \
        [file join $IVI_HOME modules help_browser help_menus.mtf]

    #**** Setup binding for cursor-change when mouse-over links
    bind HtmlClip <Motion> {
        set parent [winfo parent %W]
        set url [$parent href %x %y]
        if {[string length $url] > 0} {
            $parent configure -cursor hand2
        } else {
            $parent configure -cursor {}
        }
    }

    #**** Scan the doc directory for available documents...

    HelpBrowser::LocateDocuments
}



