/****************************************************************************
 *                         mk_pkg_index.c
 * Author: Matthew Ballance
 * Desc:   Replacement for TCL's package command. Gives more flexibility in
 *         naming a package...
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "config.h"

typedef struct strlist {
    char    **items;
    int       num_items;
    int       max_items;
} strlist;


/********************************************************************
 * strlist_init()
 ********************************************************************/
static void strlist_init(strlist *sl)
{
    memset(sl, 0, sizeof(strlist));
}
/********************************************************************
 * strlist_free()
 ********************************************************************/
static void strlist_free(strlist *sl)
{
    if (sl->max_items) {
        free(sl->items);
    }
}

/********************************************************************
 * strlist_append()
 ********************************************************************/
static void strlist_append(strlist *sl, char *str)
{
    /**** Realloc ****/
    if (sl->num_items >= sl->max_items) {
        if (sl->max_items) {
            sl->items = (char **)realloc(sl->items, 
                    sizeof(char *)*sl->max_items+(sizeof(char *)*16));
        } else {
            sl->items = (char **)malloc(sizeof(char *)*16);
        }
        sl->max_items += 16;
    }

    sl->items[sl->num_items++] = strdup(str);
}

/********************************************************************
 * get_dlname()
 *
 * Reads a .la file to discover the name of the shared-lib to load
 ********************************************************************/
static char *get_dlname(char *filename)
{
    FILE *fp;
    static char  buf[1024];
    char   *ret = 0;
    
    if (!(fp = fopen(filename, "r"))) {
        fprintf(stderr, "ERROR :: mk_pkg_index cannot open \"%s\" for"
                " reading!!\n", filename);
        exit(1);
    }

    while (fgets(buf, 1024, fp)) {
        /**** Found it... ****/
        if (!strncmp(buf, "dlname", 6)) {
            char *eptr = strchr(buf, '=');
            if (!eptr) {
                break;
            }

            eptr++;

            while (*eptr == '\'' || isspace(*eptr)) {
                eptr++;
            }
            
            ret = eptr;

            while (*eptr && !isspace(*eptr) && !(*eptr == '\'')) {
                eptr++;
            }

            *eptr = 0;
            break;
        }
    }

    if (!ret) {
        fprintf(stderr, "ERROR :: mk_pkg_index cannot locate dlname in "
                "%s\n", filename);
        exit(1);
    }

    fclose(fp);

    return ret;
}

/********************************************************************
 * get_init()
 ********************************************************************/
static char *get_init(char *libname)
{
    static char buf[512];
    char *ret, *ptr;

    strcpy(buf, libname);
    ret = buf;

    /**** Get rid of the 'lib' prefix ****/
    if (!strncmp(ret, "lib", 3)) {
        ret += 3;
    }

    /**** Now get rid of the suffix ****/
    if (!(ptr = strchr(ret, '.'))) {
        fprintf(stderr, "ERROR :: mk_pkg_index - problem locating init proc\n");
        exit(1);
    }
    *ptr = 0;

    if ((ptr = strrchr(ret, '-'))) {
        *ptr = 0;
    }

    return ret;
}

/********************************************************************
 * main()
 ********************************************************************/ 
int main (int argc, char **argv)
{
    char    *pkg_name = 0;
    strlist  tcl_files, la_files;
    int      i;
    int      static_pkg = 0;

    strlist_init(&tcl_files);
    strlist_init(&la_files);


    /****************************************************************
     * process command-line args
     ****************************************************************/
    for (i=1; i<argc; i++) {
        if (!strcmp(argv[i], "-pkg")) {
            i++;
            pkg_name = argv[i];
        } else if (!strcmp(argv[i], "-static")) {
            static_pkg = 1;
        } else {
            char *ext = strchr(argv[i], '.');
            if (ext && 
                    (!strcmp(++ext,  SLIB_EXT) || !strcmp(ext, "a"))) {
                strlist_append(&la_files, argv[i]);
            } else {
                char *last_sl;

                if ((last_sl = strrchr(argv[i], '/'))) {
                    last_sl++;
                } else {
                    last_sl = argv[i];
                }
                strlist_append(&tcl_files, last_sl);
            }
        }
    }

    /****************************************************************
     * sanity checks
     ****************************************************************/
    if (!pkg_name) {
        fprintf(stderr, "ERROR :: mk_pkg_index requires -pkg <pkg_name> be "
                "specified\n\tPlease set PKG_IDX_FLAGS in your Makefile\n");
        exit(1);
    }


/*   fprintf(stdout, "package ifneeded %s 1.0 ", pkg_name); */

    /****************************************************************
     * Dump the DLL's...
     ****************************************************************/
    for (i=0; i<la_files.num_items; i++) {
        char *dlname, *init;

        if (static_pkg) {
            char *p;

            dlname = la_files.items[i];
            if (!strncmp(dlname, "lib", 3)) {
                dlname += 3;
            }

            if ((p = strchr(dlname, '.'))) {
                *p = 0;
            }

            fprintf(stdout, "ivi_load_pkg %s\n", dlname);
        } else {
	    dlname = la_files.items[i];
            init   = get_init(dlname);
            fprintf(stdout, "load [file join $dir %s] %s\n", dlname, init);
        }
    }

    /****************************************************************
     * Dump the TCL files...
     ****************************************************************/
    for (i=0; i<tcl_files.num_items; i++) {
        fprintf(stdout, "source [file join $dir %s]\n", tcl_files.items[i]);
        /*
        if (la_files.num_items || ((i+1)<tcl_files.num_items)) {
            fprintf(stdout, "%s]\n", tcl_files.items[i]);
        } else {
            fprintf(stdout, "%s]]", tcl_files.items[i]);
        }
         */
        fflush(stdout);
    }


    return 0;
}

