/****************************************************************************
 *                            WaveWidgetDraw.cc
 *
 * Author: Matthew Ballance
 * Desc:   Trace-drawing code for the wave widget
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "WaveWidget.h"
#include "WidgetManager.h"
#include "WidgetCommon.h"
#include "SdbReader.h"
#include "SdbrCursor.h"
#include "CallbackMgr.h"
#include "PsPixmapObj.h"
#include "BitVector.h"
#include "BitVectorABVal.h"

#include "SigDB.h"
#include "SdbSignal.h"
#include "DFIO.h"
#include "DFIOTrace.h"

#include "LogInstance.h"
#include "LogDestination.h"
#include "LogMgr.h"

#include "TreeWidget.h"
#include "TreeWidgetNode.h"

#undef TRACE_WIN_REPAINT
#undef DEBUG_VECTOR_COMPACT

#define FP stderr

#ifdef TRACE_WIN_REPAINT
#define WIN_MSG(x) fprintf x
#else
#define WIN_MSG(x)
#endif

#ifdef DEBUG_VECTOR_COMPACT
#define VSC_MSG(x) fprintf x
#else
#define VSC_MSG(x)
#endif

#define BETWEEN(lb, v, ub) ((((lb) <= (v)) && ((ub) >= (v))))


/******************************************************************
 * DrawTimeScale
 *
 * - startTime
 * - endTime
 * - 
 ******************************************************************/
void WaveWidget::DrawTimeScale(
        PixmapObj    &pixmap,
        Uint32        startTime,
        Uint32        endTime,
        Uint32        pmStartOffset,
        Uint32        pixHeight,
        Uint32        drawTags)
{
    Uint32    delta;
    Uint32    preBar, firstBar, numBars, curBar, i, height, x, y;
    Int32     t_height, t_width;
    GCObj    *gcObjs = getGCs();
    GCObj    *gc  = &gcObjs[TC_ScaleColor];
    GCObj    *tgc = &gcObjs[TC_TimescaleTextColor];
    Tk_TextLayout  layout;
    char      buf[64];
    Uint32    ys;

    /**** Ensure that the GCs are attached to this pixmap... ****/
    pixmap.addGCs(gcObjs, TC_NumColors);

    if (!timeScale) {
        return;
    }

    delta = endTime-startTime;

    /**** Okay, find what the first time-bar is... Then find out
     **** what the one-time offset is to the second bar (or first ;-)
     ****
     **** First visible bar is one timeScale beyond the preBar.
     ****/
    preBar = ((startTime/timeScale)*timeScale);
    firstBar     = preBar+timeScale;

    numBars = (endTime-firstBar)/timeScale;

    for (i=0; i<=numBars; i++) {
        curBar = (firstBar + (timeScale*i));
        x = TimeToPixels(curBar-startTime);

        if (drawTags) {
            FormatTime(buf, curBar);

            layout = tgc->ComputeTextLayout(buf, -1,
                    TK_JUSTIFY_CENTER, &t_width, &t_height);

            /**** Draw the tag need the bottom of the pixmap:
             **** - (pixmapHeight-t_height) + (t_height/2)
             ****/
            y = (pixHeight-t_height) + (t_height/2);
            d_scaleTextHeight = t_height;
            tgc->DrawTextLayout(layout, (x-(t_width/2)), 
                    pmStartOffset+(pixHeight-t_height));

            gc->line(x, pmStartOffset, x, 
                    pmStartOffset+(pixHeight-t_height));
        } else {
            ys = pmStartOffset;
            gc->line(x, ys, x, pixHeight-pmStartOffset);
        }
    }
}


/******************************************************************
 * DrawScalarTrace
 *
 * All X coordinates in this function are to be in units of time
 * as an offset of the initial window time.
 ******************************************************************/
void WaveWidget::DrawScalarTrace(
    SdbSignal *signal, PixmapObj &pixmap, Uint32 xOff, Uint32 yOff, 
    Vector<DFIOValChg> *chgs, Uint32 scale, Uint32 startTime, Uint32 endTime)
{
    Uint32        i, y = yOff, xt = 0, len;
    Uint32        nxt, ny, base, dy, ot_offset = 0;
    DFIOValChg   *chg1, *chg2;
    Uint32        x, nx, traceMaxTime;
    GCObj        *gcObjs = getGCs();
    GCObj        *gc = &gcObjs[TC_Scalar0Color];
    GCObj        *cgc = &gcObjs[TC_ScalarChgColor];

    WIN_MSG((FP, "----> ::DrawScalarTrace()\n"));

    /**** Ensure that the GCs are attached to this pixmap... ****/
    pixmap.addGCs(gcObjs, TC_NumColors);

    log_high("pixmapStartTime = %d\n", startTime);

    dy = (d_config.traceHeight*7)/16;

    if (!(chg1 = chgs->idx(0))) {
        return;
    }

    traceMaxTime = signal->dfioTrace->getTraceMaxTime();

    if (chg1->changeTime > startTime) {
        WIN_MSG((FP, "\tinc xt by %d\n", chg1->changeTime-startTime));
        xt += chg1->changeTime-startTime;
    } else {
        WIN_MSG((FP, "\tot_offset = %d\n", startTime-chg1->changeTime));
        ot_offset = startTime-chg1->changeTime;
    }

    len = chgs->length();

    WIN_MSG((FP, "\tlen=%d\n", len));

    for (i=0; i<len; i++) {
        WIN_MSG((FP, "\t\tidx=%d ; changeTime=%d ; val=%c\n",
            i, chgs->idx(i)->changeTime, chgs->idx(i)->bitVal[0]));
    }

    chg1 = chgs->idx(0);
    for (i=0; i<len; i+=scale) {
        gc = 0;

        chg2 = chgs->idx(i+scale);
        if (!chg1) {
            break;
        }

        WIN_MSG((FP, "val=%c ; time=%d\n", chg1->bitVal[0], chg1->changeTime));

        switch(DFIOValChg_GetBit(chg1->bitVal, 0)) {
            case DFIOBitVal_0:
                ny = yOff + dy;
                gc = &gcObjs[TC_Scalar0Color];
            break;

            case DFIOBitVal_1:
                ny = yOff - dy;
                gc = &gcObjs[TC_Scalar1Color];
            break;

            case DFIOBitVal_Z:
                ny = yOff;
                gc = &gcObjs[TC_ScalarZColor];
            break;

            default:
                ny = yOff;
                gc = &gcObjs[TC_ScalarXColor];
            break;
        }

        if (chg2) {
            nxt = xt + (chg2->changeTime-chg1->changeTime) - ot_offset;
            /**** If either the trace-end time or the last val-change are 
             **** on screen, then draw a transition... Otherwise, bail...
             ****/
        } else if (BETWEEN(startTime, chg1->changeTime, endTime) 
                || (traceMaxTime >= startTime)) {
            Uint32       chgTime;

            if (chg1->changeTime < startTime) {
                chgTime = startTime;
            } else {
                chgTime = chg1->changeTime;
            }

            if (endTime > traceMaxTime) {
                nxt = xt + (traceMaxTime-chgTime);
                WIN_MSG((FP, "\tnxt = %d + (%d-%d) => %d\n", xt, traceMaxTime, 
                    chgTime, nxt));
            } else {
                WIN_MSG((FP, "\tnxt = %d + (%d-%d) => %d\n", xt, endTime, 
                    chgTime, nxt));
                nxt = xt + (endTime-chgTime);
            }
        } else {
            break;
        }

        x  = TimeToPixels(xt) + xOff;
        nx = TimeToPixels(nxt) + xOff;

        if (nx > pixmapWidth) {
            nx = pixmapWidth;
        }

        /*** NOTE: change-compression disabled for now ***/
        if (0 && ((nx-x) <= 1) && 
            (DFIOValChg_GetBit(chg1->bitVal, 0) != DFIOBitVal_Z) &&
                (DFIOValChg_GetBit(chg1->bitVal, 0) != DFIOBitVal_X)) {
            Uint32         cnx, cx, ci, cnxt, cxt=xt;
            DFIOValChg    *cchg1, *cchg2;
            Uint32         rect_x_start = x;
            Uint32         rect_x_end = x;

            cnx = nx;
            cx  = x;

            cchg1 = chgs->idx(i);

            for (ci=i; ci<(len-1); ci++) {
                DFIOBitVal   val;
                cchg2 = chgs->idx(ci+1);

                val = DFIOValChg_GetBit(cchg1->bitVal, 0);

                if (val == DFIOBitVal_Z || val == DFIOBitVal_X) {
                    break;
                }

                cnxt = cxt + (cchg2->changeTime-cchg1->changeTime - ot_offset);

                cx  = TimeToPixels(cxt)+xOff;
                cnx = TimeToPixels(cnxt)+xOff;

                if ((cnx-cx) > 1) {
                    if (ci) {
                        chg2 = cchg1;
                        ci--;
                    }
                    rect_x_end = cx;
                    nxt = cxt;
                    break;
                }

                cchg1 = cchg2;
                cxt   = cnxt;
                
                ot_offset = 0;
            }

            if (ci==(len-1)) {
                rect_x_end=cx;
            }

            cgc->fill_rect(rect_x_start, yOff-dy, 
                    (rect_x_end-rect_x_start), 2*dy);

            i = ci;
        } else {
            if (y != ny) {
                WIN_MSG((FP, "dy_line(%d, %d, %d, %d)\n", x, y, x, ny));
                cgc->line(x, y, x, ny);
            }

            WIN_MSG((FP, "line(%d, %d, %d, %d)\n", x, ny, nx, ny));
            gc->line(x, ny, nx, ny);
 
        }

        xt = nxt;
        y = ny;
        ot_offset = 0;

        chg1 = chg2;
    }

    WIN_MSG((FP, "<---- ::DrawScalarTrace()\n"));

www_exit:
    return;
}

/******************************************************************
 * DrawVectorTrace
 ******************************************************************/
void WaveWidget::DrawVectorTrace(
        SdbSignal            *signal, 
        PixmapObj            &pixmap, 
        Uint32                xOff, 
        Uint32                yOff, 
        Vector<DFIOValChg>   *chgs,
        Uint32                scale,
        Uint32                startTime,
        Uint32                endTime)
{
    Uint32         dy, i, draw_trans, xt = 0, y = yOff, nxt, ii;
    Uint32         x, nx, len;
    Uint32         ot_offset = 0;
    DFIOValChg    *chg1, *chg2;
    Tk_TextLayout  layout;
    int            text_width, text_height;
    GCObj         *gcObjs = getGCs();
    GCObj         *gc = &gcObjs[TC_VectBinColor];
    GCObj         *vcgc = &gcObjs[TC_VectChgColor];
    GCObj         *lgc = &gcObjs[TC_VectBinColor];
    Uint32         traceMaxTime;
    Display       *display = getDisplay();
    Uint32         flags;
    bool           lastXZ = true;
    bool           currXZ = false;

    VSC_MSG((FP, "DrawVectorTrace(pixmapStartTime=%d, "
            "displayStartTime=%d, start_pix=%d)\n", 
            pixmapStartTime, displayStartTime, 
            TimeToPixels(displayStartTime-pixmapStartTime)));

#ifdef DEBUG_VECTOR_COMPACT
    chg2 = 0;
    for (i=0; i<chgs->length(); i++) {
        chg1 = chgs->idx(i);

        fprintf(stderr, "\tidx %d => %d (%d) ", i, chg1->changeTime,
                TimeToPixels(chg1->changeTime-pixmapStartTime));
        if (chg2) {
            fprintf(stderr, "delta=%d\n", 
                    TimeToPixels(chg1->changeTime-chg2->changeTime));
        } else {
            fprintf(stderr, "\n");
        }
        chg2 = chg1;
    }
#endif

    dy = (d_config.traceHeight*7)/16;

    if (!(chg1 = chgs->idx(0))) {
        return;
    }

    traceMaxTime = signal->dfioTrace->parent->getMaxTime();

    if (chg1->changeTime > startTime) {
        xt += (chg1->changeTime-startTime);
    } else {
        ot_offset = (startTime-chg1->changeTime);
    }

    len = chgs->length();

    chg1 = chgs->idx(0);
    for (i=0; i<len; i+=scale) {
        draw_trans = 0;
        chg2 = chgs->idx(i+scale);

	if (!chg1) {
	    return;
	}
        if (chg2) {
            draw_trans = 1;
            nxt = xt + (chg2->changeTime-chg1->changeTime - ot_offset);
        } else if (BETWEEN(startTime, chg1->changeTime, endTime) 
                || (traceMaxTime >= startTime)) {
            Uint32       chgTime;
            draw_trans = 1;
            if (chg1->changeTime < startTime) {
                chgTime = startTime;
            } else {
                chgTime = chg1->changeTime;
            }

            if (endTime > traceMaxTime) {
                nxt = xt + (traceMaxTime-chgTime);
            } else {
                nxt = xt + (endTime-chgTime);
            }
        } else {
            break;
        }

        x = TimeToPixels(xt)+xOff;
        nx = TimeToPixels(nxt)+xOff;

        lastXZ = currXZ;
        signal->FormatValue(chg1, strVal, flags);
        if (flags & BitVectorABVal::FmtInfo_ExclZ) {
            gc = &gcObjs[TC_VectZColor];
            currXZ = true;
        } else if (flags & BitVectorABVal::FmtInfo_ExclX) {
            gc = &gcObjs[TC_VectXColor];
            currXZ = true;
        } else if (flags & BitVectorABVal::FmtInfo_HasXZ) {
            currXZ = false;
            gc = &gcObjs[TC_VectXColor];
        } else {
            currXZ = false;
            gc = &gcObjs[TC_VectBinColor];
        }

        /*** NOTE: value-compression disabled for now ***/
        if (0 && ((nx-x) <= 1) && (currXZ == false)) {
            Uint32         cnx, cx, ci, cnxt, cxt=xt;
            DFIOValChg    *cchg1, *cchg2;
            bool           ccurrXZ=currXZ, lastXZ=lastXZ;
            Uint32         rect_x_start = x;
            Uint32         rect_x_end = x;

            /*** Now, loop while the delta between any two elements is
             *** <= 3
             ***/

            cnx = nx;
            cx = x;

            VSC_MSG((FP, "\tBegin compression @ idx %d\n", i));

            cchg1 = chgs->idx(i);

            for (ci=i; ci<(len-1); ci++) {
                cchg2 = chgs->idx(ci+1);

                signal->FormatValue(cchg1, strVal, flags);

                /*** Cannot compress X/Z values... quit... ***/
                if ((flags & BitVectorABVal::FmtInfo_ExclZ) ||
                        (flags & BitVectorABVal::FmtInfo_ExclX) ||
                        (flags & BitVectorABVal::FmtInfo_HasXZ)) {
                    VSC_MSG((FP, "\t\tQuit @ %d due to X/Z\n", ci));
                    break;
                }

                cnxt = cxt + (cchg2->changeTime-cchg1->changeTime - ot_offset);

                cx = TimeToPixels(cxt)+xOff;
                cnx = TimeToPixels(cnxt)+xOff;

                VSC_MSG((FP, "\t\t@ idx %d - cnx-cx=%d (cnx=%d)\n", 
                        ci, (cnx-cx), cnx));
                /*** See if this delta still qualifies... ***/
                if ((cnx-cx) > 1) {
                    VSC_MSG((FP, "\t\tQuit @ %d due to delta=%d\n", ci, (cnx-cx)));
                    if (ci) {
                        chg2 = cchg1;
                        ci--;
                    }
                    rect_x_end = cx;
                    nxt = cxt;
                    break;
                }
                cchg1=cchg2;
                cxt = cnxt;
                ot_offset = 0;
            }

            if (ci==(len-1)) {
                VSC_MSG((FP, "\t\tQuit @ %d due to overflow...\n", ci));
                rect_x_end = cx;
            } else {

            }

            /*** When the loop ends, see how large of a rectangle to draw... 
             ***/
            gcObjs[TC_VectChgColor].fill_rect(
                    rect_x_start, yOff-dy, (rect_x_end-rect_x_start), 2*dy);

            VSC_MSG((FP, "\tDraw rect %d => %d (cxt=%d, cnxt=%d)\n", 
                    rect_x_start, rect_x_end, cxt, cnxt));

            i = ci;
        } else {
            if (((nx-x) > 10) &&
                ((flags & BitVectorABVal::FmtInfo_ExclX) == 0) &&
                    ((flags & BitVectorABVal::FmtInfo_ExclZ) == 0)) {
                layout = gc->ComputeTextLayout(strVal.value(), strVal.length(),
                    TK_JUSTIFY_CENTER, &text_width, &text_height);
                if ((text_width+4) < TimeToPixels(nxt-xt)) {
                    gc->DrawTextLayout(layout,
                        TimeToPixels(xt+((nxt-xt-PixelsToTime(text_width))/2)), 
                        y-dy);
                }
                gc->FreeTextLayout(layout);
            }

            /**** If (lastXZ && currXZ) {
             ****    - no transition
             **** } else if (currXZ) {
             ****    - draw \ transition
             ****           /
             **** } else if (lastXZ) {
             ****    - draw ____/ transition
             ****               \
             **** } else {
             ****    - normal \/
             ****             /\
             ****/
            if (lastXZ && currXZ) {
                gc->line(x, yOff, nx, yOff);
            } else if (currXZ) {
                vcgc->line(x, yOff+dy, x+1, yOff);
                vcgc->line(x, yOff-dy, x+1, yOff);

                gc->line(x+1, yOff, nx, yOff);
            } else if (lastXZ) {
                vcgc->line(x, yOff, x+1, yOff+dy);
                vcgc->line(x, yOff, x+1, yOff-dy);

                gc->line(x+1, yOff+dy, nx, yOff+dy);
                gc->line(x+1, yOff-dy, nx, yOff-dy);
            } else {
                vcgc->line(x, yOff+dy, x+2, yOff-dy);
                vcgc->line(x, yOff-dy, x+2, yOff+dy);

                gc->line(x+2, yOff+dy, nx, yOff+dy);
                gc->line(x+2, yOff-dy, nx, yOff-dy);
            }

            VSC_MSG((FP, "\tDraw segment %d -> %d (idx=%d)\n", x, nx, i));
        }
                    
        xt = nxt;
        ot_offset = 0;

        chg1 = chg2;
    }
}


/******************************************************************
 * DrawExpandedTrace()
 *
 *
 * Returns:
 *    Number of pixels taken. In other words, yOff + Ret = 
 *    Position of Last-trace center. 
 ******************************************************************/
Uint32 WaveWidget::DrawExpandedTrace(
    SdbSignal           *signal,
    PixmapObj           &pixmap,
    Uint32               xOff,
    Uint32               yOff,
    Uint64               traceSpec,
    Vector<DFIOValChg>  *chgs,
    Uint32               scale,
    Uint32               startTime,
    Uint32               endTime)
{
    Uint32        i, ii, start, x = xOff;
    Uint32        nx, ny, y, dy;
    Uint32        xt = 0, nxt;
    DFIOValChg   *valChg1, *valChg2, *chg1;
    Uint32        total = 0, ot_offset = 0;
    Uint32        traceMaxTime;
    GCObj        *gcObjs = getGCs();
    GCObj        *gc = &gcObjs[TC_VectExp0Color];
    Display      *display;
    Uint32        traceSep = getTraceSep();
    GCObj        *cgc = &gcObjs[TC_VectExpChgColor];
 
    if (!(chg1 = chgs->idx(0))) {
        goto wav_exit;
    }

    dy = (d_config.traceHeight*7)/16;

    /**** Ensure that the GCs are attached to this pixmap... ****/
    pixmap.addGCs(gcObjs, TC_NumColors);

    traceMaxTime = signal->dfioTrace->parent->getMaxTime();

    if (chg1->changeTime > startTime) {
        xt += (chg1->changeTime-startTime);
    } else {
        ot_offset = (startTime-chg1->changeTime);
    }

    if (traceSpec.high >= yTmpStorageSize) {
        delete yTmpStorage;
        yTmpStorage = new Uint32[traceSpec.high+1];
        yTmpStorageSize = traceSpec.high+1;
    }

    start = ((traceSpec.low)?traceSpec.low:1);
    for (i=start; i<=traceSpec.high; i++) {
        yTmpStorage[i] = yOff + (traceSep*(i-start));
    }

    valChg1 = chgs->idx(0);
    for (i=0; i<chgs->length(); i+=scale) {
        valChg2 = chgs->idx(i+scale);

        if (valChg2) {
            nxt = xt + (valChg2->changeTime-valChg1->changeTime) - ot_offset;
        } else if (BETWEEN(startTime, valChg1->changeTime, endTime) 
                || (traceMaxTime >= startTime)) {
            Uint32       chgTime;

            if (valChg1->changeTime < startTime) {
                chgTime = startTime;
            } else {
                chgTime = valChg1->changeTime;
            }

            if (endTime > traceMaxTime) {
                nxt = xt + (traceMaxTime - chgTime);
            } else {
                nxt = xt + (endTime - chgTime);
            }
        } else {
            break;
        }

        for (ii=start; ii<=traceSpec.high; ii++) {
            switch (DFIOValChg_GetBit(valChg1->bitVal, 
                        traceSpec.high-ii)) {
                case DFIOBitVal_0:
                    ny = (yOff + (ii-start)*traceSep) + dy; 
                    gc = &gcObjs[TC_VectExp0Color];
                break;

                case DFIOBitVal_1:
                    ny = (yOff + (ii-start)*traceSep) - dy; 
                    gc = &gcObjs[TC_VectExp1Color];
                break;

                case DFIOBitVal_Z:
                    ny = (yOff + (ii-start)*traceSep); 
                    gc = &gcObjs[TC_VectExpZColor];
                break;

                default:
                    ny = (yOff + (ii-start)*traceSep); 
                    gc = &gcObjs[TC_VectExpXColor];
                break;
            }

            x = TimeToPixels(xt) + xOff;
            nx = TimeToPixels(nxt) + xOff;

            if (nx > pixmapWidth) {
                nx = pixmapWidth;
            }

            if (yTmpStorage[ii] != ny) {
                cgc->line(x, yTmpStorage[ii], x, ny);
                yTmpStorage[ii] = ny;
            }

            gc->line(x, ny, nx, ny);
        }
        xt = nxt;
        ot_offset = 0;

        valChg1 = valChg2;
    }

wav_exit:

    return (traceSep * (traceSpec.high-start+1));
}

/******************************************************************
 * DrawVectorTraceMaster()
 ******************************************************************/
Uint32 WaveWidget::DrawVectorTraceMaster(
    SdbSignal             *signal,
    PixmapObj             &pixmap,
    Uint32                 xOff, 
    Uint32                 yOff,
    Uint64                 traceSpec,
    Vector<DFIOValChg>    *chgs,
    Uint32                 scale,
    Uint32                 startTime,
    Uint32                 endTime)
{
    Uint32       i;
    Uint32       total = 0;

    if (traceSpec.low == 0) {
        DrawVectorTrace(signal, pixmap, xOff, yOff, chgs, scale, 
                startTime, endTime);
        yOff += getTraceSep();
        total += getTraceSep();
    }

    if (traceSpec.low || (traceSpec.high > traceSpec.low)) {
        total += DrawExpandedTrace(signal, pixmap, xOff, yOff, traceSpec, 
                chgs, scale, startTime, endTime);
    }

    return total;
}

/******************************************************************
 * DrawWaveProper()
 *
 * Draws the main portion of the wave window. This routine may
 * be called by the printing routines or the screen management
 * code.
 ******************************************************************/
void WaveWidget::DrawWaveProper(
        PixmapObj        &pixmap,
        Uint32            startPixY,
        Uint32            endPixY,
        Uint32            startTime,
        Uint32            endTime,
        Double            pixPerTime
        )
{
    char                   tmpb[64];
    Uint64                 spec;
    Uint32                 yOff = startPixY + 
        (d_config.traceHeight/2+d_config.trace_pady);
    Double                 range, curr;
    Uint32                 tmpTime, i;
    Double                 pixPerTimeBak = pixelsPerTimeUnit;
    Uint64                 startTrace, endTrace;
    SdbSignal             *sig;
    Vector<DFIOValChg>    *chgs;
    Uint32                 scale = 1;
    Uint32                 startPixX = 0;
    GCObj                 *gcObjs = getGCs();

    pixmap.addGCs(gcObjs, TC_NumColors);

    CalcStartEndPos(&startTrace, &endTrace, &startPixY);

    /**** Timescale must cover the pixmap (?) Therefore, use the pixmap
     **** width for endTime
     ****/
    DrawTimeScale(pixmap, startTime, PixelsToTime(pixmap.width())+startTime,
            0, pixmap.height(), 1);
    range = (Double)(endTrace.high - startTrace.high);

    for (i=startTrace.high; i<=endTrace.high; i++) {
        sig = sdbr->d_signals.idx(i);

        if (!sig) {
            continue;
        }

        spec.low = 0;
        spec.high = sig->traceMax();

        if (i==startTrace.high) {
            spec.low = startTrace.low;
        }

        if (i==endTrace.high) {
            spec.high = endTrace.low;
        }

        if (!sig->isSeparator()) {
            chgs = sig->dfioTrace->getValue(startTime, endTime, 
                    DFIOTrace::GVF_Default);
        }

        if (!sig->isSeparator()) {
            if (sig->length() > 1) {
                yOff += DrawVectorTraceMaster(sig, pixmap,
                        startPixX, yOff, spec, chgs, scale, startTime,
                        endTime);
            } else {
                DrawScalarTrace(sig, pixmap, startPixX, yOff, chgs, scale,
                        startTime, endTime);
                yOff += getTraceSep();
            }
            delete chgs;
        } else {
            Uint32 mid = yOff;
            gcObjs[TC_SeparatorColor].line(0, mid,
                    TimeToPixels(endTime-startTime), mid);
            yOff += getTraceSep();
        }
    }

    pixelsPerTimeUnit = pixPerTimeBak;
}

/******************************************************************
 * UpdatePixmaps()
 *
 * Check that the current pixmap sizes are correct for the widget's
 * geometry.
 ******************************************************************/
void WaveWidget::UpdatePixmaps(void)
{
    Tk_Window    tkwin = getWin();
    if (!winValid()) {
        return;
    }

    if ((height() != widgetHeight) || (width() != widgetWidth)) {
        widgetHeight = height();
        widgetWidth  = width();

        d_displayPixmap.width_height(width(), height());
    }

    if ( (height() > pixmapYReallocThresh) ||
         (width() > pixmapXReallocThresh) ) {

        pixmapHeight = 2*widgetHeight;
        pixmapWidth  = 2*widgetWidth;

        pixmapYReallocThresh = (3*pixmapHeight)/4;
        pixmapXReallocThresh = (3*pixmapWidth)/4;

        d_backingPixmapValid = 0;

        log_med("UpdatePixmaps invalidates Pixmap1\n");

        d_backingPixmap.width_height(pixmapWidth, pixmapHeight);
    }
}

/******************************************************************
 * CalcStartEndPos()
 * 
 * Calculates the starting and ending trace/subtrace pairs. 
 * Routine also suggests a starting position.
 * 
 * Depends on pixmapStartPixPos and pixmapEndPixPos
 ******************************************************************/
void WaveWidget::CalcStartEndPos(
    Uint64       *startPos,
    Uint64       *endPos,
    Uint32       *startY)
{
    SdbSignal           *signal;
    Uint32               firstPrimTrace, lastPrimTrace;
    Uint32               traceTotal, i, x, tmp, len;
    Vector<SdbSignal>   *signals;
    Uint32               traceSep = getTraceSep();

    log_med.enter("CalcStartEndPos(startPos=(%d,%d), endPos=(%d,%d),"
            " startY=%d)\n", startPos->high, startPos->low, endPos->high,
            endPos->low, *startY);

    startPos->low = startPos->high = 0;
    endPos->low   = endPos->high   = 0;

   /**** First, based on the requested pixmapStartPixPos, calculate
    **** the offset in primitive sub-traces
    ****
    **** firstPrimTrace and lastPrimTrace are the number of traces that
    **** fit in the space leading up to their respective position pixels.
    **** For lastPrimTrace, we actually want one trace before...
    ****/
    firstPrimTrace = pixmapStartPixPos / traceSep;
    lastPrimTrace  = pixmapEndPixPos   / traceSep;
    if (lastPrimTrace) {
        lastPrimTrace -= 1;
    }

    log_med("pixmapStartPixPos = %d, pixmapEndPixPos = %d, traceSep = %d\n",
            pixmapStartPixPos, pixmapEndPixPos, traceSep);
    log_med("firstPrimTrace = %d ; lastPrimTrace = %d\n",
            firstPrimTrace, lastPrimTrace);

    *startY = (firstPrimTrace * traceSep);

   /**** Now, translate that to physical traces.
    ****/
    traceTotal = 0;

    /**** Bail if no SDBR is attached... ****/
    if (!sdbr) {
        return;
    }
    signals = &sdbr->d_signals;

    len     = signals->length();

    log_med.enter("Start Loop\n");
    for (i=0; i<len; i++) {
        signal = signals->idx(i);
        log_med("Signal %d - %d Vis Traces\n", i, signal->getVisTraces());
        traceTotal += signal->getVisTraces();
        if (traceTotal >= firstPrimTrace) {
            log_med("Stop at %d ; (tmp=%d) > (firstPrimTrace=%d)\n",
                    i, traceTotal, firstPrimTrace);
            startPos->high = i;
            startPos->low  = signal->getVisTraces(); 
            startPos->low -= (traceTotal-firstPrimTrace);
            break;
        }
    }
    log_med.leave("Start Loop\n");

    /**** If the current trace covers everything to be displayed, 
     **** adjust now...
     ****/
    if (traceTotal >= lastPrimTrace) {
        log_med("traceTotal (%d) >= lastPrimTrace (%d)\n",
                traceTotal, lastPrimTrace);
        endPos->high  = i;
        endPos->low   = signal->getVisTraces();
        endPos->low  -= traceTotal-lastPrimTrace;

        log_med("endPos = (%d,%d)\n", endPos->high, endPos->low);
    } else {
       /**** Find the last trace-descriptor...
        ****/
        tmp = traceTotal;
        log_med.enter("Stop Loop\n");
        for (x=i+1; x<len; x++) {
            signal = signals->idx(x);
            log_med("Signal %d - %d Vis Traces\n", 
                    x, signal->getVisTraces());
            tmp += signal->getVisTraces();

            if (tmp >= lastPrimTrace) {
                log_med("Stop at %d ; tmp = %d >= lastPrimTrace %d\n",
                        x, tmp, lastPrimTrace);
                endPos->high = x;
                endPos->low = signal->getVisTraces();
                endPos->low -= (tmp-lastPrimTrace);

                if (((int)endPos->low) < 0) {
                    log_low("ERROR :: Paused...\n");
                    endPos->low = 0;
                }
                break;
            }
        }
        log_med.leave("Stop Loop\n");
    }

    /**** Do some final processing if we are showing the last trace...
     ****/
    if (x == len) {
        if (len) {
            endPos->high = (len-1);
            endPos->low  = signals->idx(endPos->high)->traceMax();
        } else {
            endPos->high = 0;
            endPos->low  = 0;
        }
    }

    log_med.leave("CalcStartEndPos(startPos=(%d,%d), endPos=(%d,%d),"
            " startY=%d)\n", startPos->high, startPos->low, endPos->high,
            endPos->low, *startY);
}


/******************************************************************
 * Center()
 *
 * Calculates the boundary points for the backing pixmap based on 
 * the size of the backing pixmap and the start/size of the 
 * displayed area. Note that all units must be the same: everything
 * must be in pixels, or everything must be in time units, etc...
 * 
 * NOTE: it is assumed that (pixmapSize >= displaySize)
 ******************************************************************/
void WaveWidget::Center(
    Uint32        displayBegin,
    Uint32        displaySize,
    Uint32        pixmapSize,
    Uint32       *pixmapBegin,
    Uint32       *pixmapEnd)
{
    Uint32    half = (pixmapSize-displaySize)/2;

    if (half <= displayBegin) {
        *pixmapBegin = (!displayBegin)?0:(displayBegin - half);
    } else {
        *pixmapBegin = 0;
    }
    *pixmapEnd   = (*pixmapBegin+pixmapSize);
}

/****************************************************************** 
 * RepaintBackingPixmaps()
 *
 * Updates the content of the backing pixmaps. Leaves pixmap1 
 * valid.
 *
 * For redraw, need:
 *  - start-time of repaint
 *  - end-time of repaint
 *  - start-trace of repaint
 *  - end-trace of repaint
 *  - start-pixelX           (referenced to backing pixmap)
 *  - start-pixelY           (referenced to backing pixmap)
 * 
 *  - Require repaint?       (perhaps can just re-adjust the screen)
 *
 * 
 * Parameters:  (implicit)
 *   - newDisplayStartPos  (measured in pixels)
 *   - newDisplayStartTime (measured in time units)
 ******************************************************************/
void WaveWidget::RepaintBackingPixmaps(void)
{
    Uint32                 repaintStartTime  = pixmapStartTime;
    Uint32                 repaintEndTime    = pixmapEndTime;

    Uint32                 repaintStartPos   = 0;
    Uint32                 repaintEndPos     = 0;

    Uint32                 repaintStartTrace = 0;
    Uint32                 repaintEndTrace   = 0;

    Uint32                 repaintStartPixX = 0;
    Uint32                 repaintStartPixY = 0;
    Uint32                 needRedraw = 0, i, tmp;
    SdbSignal             *sig;
    Vector<DFIOValChg>    *chgs;

    Uint32    copyStartX, copyStartY;
    Uint32    displayLastTime = 0;
    Uint32    pixmapStartPix, pixmapEndPix;  /*** display start-pix referenced
                                              *** to the backing pixmap
                                              ***/
    Uint32    height;
    Uint32    scale = 1;

    log_high.enter("RepaintBackingPixmaps()\n");

    if (!d_backingPixmapValid) {
        /*** In this case, we'll want to update both X/Y coords...
         ***/
        needRedraw = 1;
        newDisplayStartTime = displayStartTime;
        newDisplayStartPos  = displayStartPos;
        SET_FLAG(RepaintScrollX|RepaintScrollY);
        log_high("NOTE: Redraw due to pixmap invalid\n");
    }

    if (pixelsPerTimeUnit <= 0.000000000001) {
        pixelsPerTimeUnit = 
            (((Double)(widgetWidth))/
             ((Double)(newDisplayEndTime-newDisplayStartTime)));
    }

    /**** First check the X direction...
     ****/
    if (IS_FLAG_SET(RepaintScrollX)) {
        newDisplayEndTime = PixelsToTime(widgetWidth)+newDisplayStartTime;
        pixmapEndTime     = PixelsToTime(pixmapWidth)+pixmapStartTime;

        if ((newDisplayStartTime < pixmapStartTime) ||
            (newDisplayEndTime   > pixmapEndTime) || (needRedraw & 1)) {
            needRedraw |= 2;
            Center(newDisplayStartTime, PixelsToTime(widgetWidth),
                    PixelsToTime(pixmapWidth), &repaintStartTime, 
                    &repaintEndTime);
            pixmapStartTime = repaintStartTime;
            pixmapEndTime   = repaintEndTime;
            log_high("NOTE: Setting pixmapStartTime = "
                        "%d EndTime = %d\n", pixmapStartTime, pixmapEndTime);
            log_high("NOTE: Redraw due to X scroll\n");
            log_high("\trepaintStartTime = %d ; pixmapStartTime = "
                        "%d\n", repaintStartTime, pixmapStartTime);
        } 
        displayStartTime = newDisplayStartTime;
    }

    if (IS_FLAG_SET(RepaintScrollY)) {

        if ((newDisplayStartPos < pixmapStartPixPos) ||
            ((newDisplayStartPos+widgetHeight) > 
             (pixmapStartPixPos+pixmapHeight)) || (needRedraw & 1)) {
            needRedraw |= 4;
            Center(newDisplayStartPos, widgetHeight, pixmapHeight,
                    &repaintStartPos, &repaintEndPos);
            pixmapStartPixPos = repaintStartPos;
            pixmapEndPixPos = pixmapStartPixPos + pixmapHeight;
            CalcStartEndPos(&d_startTrace, &d_endTrace, &pixmapStartPixPos);
            pixmapEndPixPos = pixmapStartPixPos + pixmapHeight;
            log_high("NOTE: Redraw due to Y scroll\n");
            log_high("\tnewDisplayStartPos = %d ; "
                    "pixmapStartPixPos = %d\n", newDisplayStartPos,
                    pixmapStartPixPos);
            log_high("\twidgetHeight = %d ; pixmapHeight = %d\n",
                    widgetHeight, pixmapHeight);
            log_high("\tstartTrace = {%d %d} ; endTrace = {%d %d}\n",
                    d_startTrace.high, d_startTrace.low, d_endTrace.high, 
                    d_endTrace.low);
            log_high("\trepaintStartTime = %d ; pixmapStartTime = %d\n",
                    repaintStartTime, pixmapStartTime);
        }
        displayStartPos = newDisplayStartPos;
    }

    displayEndTime   = displayStartTime + PixelsToTime(widgetWidth);

    if (!timeScaleValid) {
        timeScale = UpdateTimeScale(pixmapStartTime, pixmapEndTime, 
                pixmapWidth);
    }

    if (needRedraw && sdbr) {
        Char          tmpb[64];
        Uint64        spec;
        Uint32        yOff = repaintStartPixY + 
            (d_config.traceHeight/2+d_config.trace_pady) ;
        Double        range, curr;
        Uint32        tmpTime;

        log_high.enter("Pixmap Redraw\n");

        tmpTime = sdbr->getMaxTime();
        if (tmpTime > totalTime) {
            totalTime = tmpTime;
        }


        d_backingPixmap.blank();

        /*** Draw the timescale. This goes beneath the wave traces ***/
        DrawTimeScale(d_backingPixmap, pixmapStartTime, pixmapEndTime,
                0, d_backingPixmap.height(), 0);

        log_med("REDRAW: pixmapStartTime = %d ; EndTime = %d\n",
                pixmapStartTime, pixmapEndTime);

        range = (Double)(d_endTrace.high - d_startTrace.high);


        for (i=d_startTrace.high; i<=d_endTrace.high; i++) {

            if (d_config.progressCommand) {
                /*
                curr = (Double)(i-d_startTrace.high);
                sprintf(tmpb, " %f ", curr/range);

                if (Tcl_VarEval(d_interp, d_config.progressCommand, 
                    tmpb, 0) != TCL_OK) {
                    fprintf(stderr, "ERROR: VarEval failed: %s\n",
                            Tcl_GetStringResult(d_interp));
                }
                */
            }

            sig = sdbr->d_signals.idx(i);

            if (!sig) {
                continue;
            }

            spec.low   = 0;
            spec.high  = sig->traceMax();

            if (i==d_startTrace.high) {
                spec.low = d_startTrace.low;
            }

            if (i==d_endTrace.high) {
                spec.high = d_endTrace.low;
            }

            /**** If the signal isn't a separator, get values. Otherwise,
             **** there is no signal data and we'll fault!!
             ****/
            if (!sig->isSeparator()) {
                chgs = sig->dfioTrace->getValue(repaintStartTime, 
                        repaintEndTime, DFIOTrace::GVF_Default);

                if (!chgs) {
                    fprintf(stderr, "ERROR: chgs is NULL\n");
                }

                log_high("getValue(%d, %d) returned length "
                        "%d (%d - %d)\n",
                  repaintStartTime, repaintEndTime, chgs->length(),
                  (chgs->length())?chgs->idx(0)->changeTime:0xFFFFFFFF,
                  (chgs->length())?chgs->idx(chgs->length()-1)->changeTime:
                      0xFFFFFFFF);
            }

            log_high("NOTE: Trace %d has spec %d %d\n",
                    i, spec.high, spec.low);

            if (!sig->isSeparator()) {
                if (sig->length() > 1) {
                    yOff += DrawVectorTraceMaster(sig, d_backingPixmap, 
                            repaintStartPixX, yOff, spec, chgs, scale,
                            pixmapStartTime, pixmapEndTime);
                } else {
                    DrawScalarTrace(sig, d_backingPixmap, repaintStartPixX,
                            yOff, chgs, scale, pixmapStartTime, pixmapEndTime);
                    yOff += getTraceSep();
                }
                delete chgs;
            } else {
                Uint32 mid = yOff;
                getGCs()[TC_SeparatorColor].line(0, mid, 
                        TimeToPixels(repaintEndTime-repaintStartTime), mid);

                yOff += getTraceSep();
            }
        }

        log_high.leave("Pixmap Redraw\n");
    } else if (!sdbr) {
        d_backingPixmap.blank();
    }

    d_backingPixmapValid = 1;

    CLR_FLAG(REPAINT_FLAGS);

    log_high.leave("RepaintBackingPixmaps()\n");
}

/******************************************************************
 * DrawSelectionBox()
 *
 * Adds the selection-box overlay to the display pixmap if the 
 * selection-box is active.
 ******************************************************************/
void WaveWidget::DrawSelectionBox(PixmapObj &pixmap)
{
    GCObj *gcObjs = getGCs();

   /**************************************************************
    * Take care of displaying selection-box if the mouse-button
    * is active...
    **************************************************************/
    if (IS_FLAG_SET(RepaintB2Down) ||
       (IS_FLAG_SET(RepaintB1Down) && !strcmp(d_config.cursorMode, "zoom"))) {
        Int32    x, y;
        Uint32    width, height;

        pixmap.addGCs(gcObjs, TC_NumColors);

        if (sel_xi < sel_xf) {
            x = sel_xi;
            width = sel_xf - sel_xi;
        } else {
            x = sel_xf;
            width = sel_xi - sel_xf;
        }

        if (sel_yi < sel_yf) {
            y = sel_yi;
            height = sel_yf - sel_yi;
        } else {
            y = sel_yf;
            height = sel_yi - sel_yf;
        }

        if ((y+height) >= (this->height()-getCursorPaneHeight())) {
            height = (this->height()-getCursorPaneHeight()-y);
        }
      
        gcObjs[TC_SelBoxColor].rect(x, y, width, height);
    } 
}

/******************************************************************
 * getCursorPaneHeight()
 *
 * Returns the required height (in pixels) of the cursor pane.
 ******************************************************************/
Uint32 WaveWidget::getCursorPaneHeight()
{
    Uint32 h = getTraceSep();

    if (sdbr && sdbr->cursorMgr) {
        Vector<SdbrCursor>    *cursors = sdbr->cursorMgr->getCursors();

        h = (getTraceSep()*(cursors->length()+1));
    }

    d_config.cursorpane_height = h;

    return h;
}

/******************************************************************
 * FormatTime()
 ******************************************************************/
void WaveWidget::FormatTime(char *buf, Uint32 time)
{
    static char *res_strings[] = {
/*    0   1    2     3     4     5     6     7     8     9     10    11    12 */
     "s", "s", "ms", "ms", "us", "us", "us", "ns", "ns", "ns", "ps", "ps", "ps",
     ""
    };
//    const char *res = "ns";
    int   res_idx   = 9;

    if (sdbr) {
//        res = sdbr->get_resString();
        res_idx = -(sdbr->get_resolution());
    }

    /*** minimization sequence should be by 3 (9 6 3 0) ***/
    /*** Now, try maximizing the units... ***/
    while ((time > 0) && !(time % 1000)) {
        if (res_idx < 3) {
            break;
        } else {
            time /= 1000;
            res_idx -= 3;
        }
    }

    sprintf(buf, "%d%s", time, res_strings[res_idx]);
}

