/*-- 

 $Id: JAXPDOMAdapter.java,v 1.5 2002/01/08 09:17:10 jhunter Exp $

 Copyright (C) 2000 Brett McLaughlin & Jason Hunter.
 All rights reserved.
 
 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:
 
 1. Redistributions of source code must retain the above copyright
    notice, this list of conditions, and the following disclaimer.
 
 2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions, and the disclaimer that follows 
    these conditions in the documentation and/or other materials 
    provided with the distribution.

 3. The name "JDOM" must not be used to endorse or promote products
    derived from this software without prior written permission.  For
    written permission, please contact license@jdom.org.
 
 4. Products derived from this software may not be called "JDOM", nor
    may "JDOM" appear in their name, without prior written permission
    from the JDOM Project Management (pm@jdom.org).
 
 In addition, we request (but do not require) that you include in the 
 end-user documentation provided with the redistribution and/or in the 
 software itself an acknowledgement equivalent to the following:
     "This product includes software developed by the
      JDOM Project (http://www.jdom.org/)."
 Alternatively, the acknowledgment may be graphical using the logos 
 available at http://www.jdom.org/images/logos.

 THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED.  IN NO EVENT SHALL THE JDOM AUTHORS OR THE PROJECT
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 SUCH DAMAGE.

 This software consists of voluntary contributions made by many 
 individuals on behalf of the JDOM Project and was originally 
 created by Brett McLaughlin <brett@jdom.org> and 
 Jason Hunter <jhunter@jdom.org>.  For more information on the 
 JDOM Project, please see <http://www.jdom.org/>.
 
 */

package org.jdom.adapters;

import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXParseException;

import org.jdom.input.BuilderErrorHandler;

/**
 * <b><code>JAXPDOMAdapater</code></b>
 * <p>
 * This class defines wrapper behavior for obtaining a DOM
 *   <code>Document</code> object using the JAXP APIs.
 * </p>
 *
 * @author Jason Hunter
 * @version $Revision: 1.5 $, $Date: 2002/01/08 09:17:10 $
 */
public class JAXPDOMAdapter extends AbstractDOMAdapter {

    private static final String CVS_ID = 
      "@(#) $RCSfile: JAXPDOMAdapter.java,v $ $Revision: 1.5 $ $Date: 2002/01/08 09:17:10 $ $Name:  $";

    /**
     * <p>
     * This creates a new <code>{@link Document}</code> from an
     *   existing <code>InputStream</code> by letting a JAXP
     *   parser handle parsing using the supplied stream.
     * </p>
     *
     * @param in <code>InputStream</code> to parse.
     * @param validate <code>boolean</code> to indicate if validation 
     *        should occur.
     * @return <code>Document</code> - instance ready for use.
     */
    public Document getDocument(InputStream in, boolean validate)
        throws ClassNotFoundException, NoSuchMethodException, 
               IllegalAccessException, InvocationTargetException {

        // Try using JAXP...
        // Note we need DOM Level 2 and thus JAXP 1.1.
        Class.forName("javax.xml.transform.Transformer");

        // Try JAXP 1.1 calls to build the document
        Class factoryClass =
            Class.forName("javax.xml.parsers.DocumentBuilderFactory");

        // factory = DocumentBuilderFactory.newInstance();
        Method newParserInstance =
            factoryClass.getMethod("newInstance", null);
        Object factory = newParserInstance.invoke(null, null);

        // factory.setValidating(validate);
        Method setValidating =
            factoryClass.getMethod("setValidating",
                                   new Class[]{boolean.class});
        setValidating.invoke(factory,
                             new Object[]{new Boolean(validate)});

        // factory.setNamespaceAware(true);
        Method setNamespaceAware =
            factoryClass.getMethod("setNamespaceAware",
                                   new Class[]{boolean.class});
        setNamespaceAware.invoke(factory,
                             new Object[]{Boolean.TRUE});

        // jaxpParser = factory.newDocumentBuilder();
        Method newDocBuilder =
            factoryClass.getMethod("newDocumentBuilder", null);
        Object jaxpParser  = newDocBuilder.invoke(factory, null);

        // jaxpParser.setErrorHandler(null);
        Class parserClass = jaxpParser.getClass();
        Method setErrorHandler =
            parserClass.getMethod("setErrorHandler",
                             new Class[]{org.xml.sax.ErrorHandler.class});
        setErrorHandler.invoke(jaxpParser,
                             new Object[]{new BuilderErrorHandler()});

        // domDoc = jaxpParser.parse(in);
        Method parse = parserClass.getMethod(
            "parse", new Class[]{InputStream.class});
        org.w3c.dom.Document domDoc = (org.w3c.dom.Document)
            parse.invoke(jaxpParser, new Object[]{in});

        // System.out.println("Using jaxp " +
        //   domDoc.getClass().getName());
        return domDoc;

        // Allow all exceptions to pass through
    }

    /**
     * <p>
     * This creates an empty <code>Document</code> object based
     *   on a specific parser implementation.
     * </p>
     *
     * @return <code>Document</code> - created DOM Document.
     */
    public Document createDocument() 
        throws ClassNotFoundException, NoSuchMethodException, 
               IllegalAccessException, InvocationTargetException {

        // We need DOM Level 2 and thus JAXP 1.1.
        // If JAXP 1.0 is all that's available then we error out.
        Class.forName("javax.xml.transform.Transformer");

        // Try JAXP 1.1 calls to build the document
        Class factoryClass =
            Class.forName("javax.xml.parsers.DocumentBuilderFactory");

        // factory = DocumentBuilderFactory.newInstance();
        Method newParserInstance =
            factoryClass.getMethod("newInstance", null);
        Object factory = newParserInstance.invoke(null, null);

        // jaxpParser = factory.newDocumentBuilder();
        Method newDocBuilder =
            factoryClass.getMethod("newDocumentBuilder", null);
        Object jaxpParser  = newDocBuilder.invoke(factory, null);

        // domDoc = jaxpParser.newDocument();
        Class parserClass = jaxpParser.getClass();
        Method newDoc = parserClass.getMethod("newDocument", null);
        org.w3c.dom.Document domDoc =
            (org.w3c.dom.Document) newDoc.invoke(jaxpParser, null);

        return domDoc;
        // System.out.println("Using jaxp " +
        //   domDoc.getClass().getName());

        // Allow all exceptions to pass through
    }
}
