#include <jmp-config.h>

#ifdef JMPDEBUG_MONITOR

#include <stdio.h>
#include <string.h>

#include <jvmpi.h>

#include <jmp.h>
#include <jmp-debug.h>
#include <monitor.h>

struct monitor_mast {
  struct monitor *head;
};

static struct monitor_mast monitor_mast;

static void monitor_free (struct monitor *m) {
    struct monitor *mtmp;
    if (monitor_mast.head == m) {
        monitor_mast.head = monitor_mast.head->next;
    } else {
        for (mtmp = monitor_mast.head; mtmp != NULL; mtmp = mtmp->next) {
            if (mtmp->next == m) {
                mtmp->next = mtmp->next->next;
                break;
            }
        }
    }

    if (m->name != NULL) {
        free (m->name);
        m->name = NULL;
    }
    free (m);
}

static struct monitor *monitor_new (JVMPI_RawMonitor monitor_id, const char *name) {
    struct monitor *m;

    m = malloc (sizeof (*m));
    if (m == NULL)
	return NULL;

    if (name != NULL)
        m->name = strdup (name);
    else
        m->name = NULL;

    m->monitor_id = monitor_id;
    m->use[0] = 0;
    m->use[1] = 0;

    m->next = monitor_mast.head;
    monitor_mast.head = m;

    return m;
}

static struct monitor *monitor_find (const char *name) {
    struct monitor *m;
    for (m = monitor_mast.head; m != NULL; m = m->next) {
        if (strcmp(m->name, name) == 0)
            break;
    }
    return m;
}

static struct monitor *monitor_ptr (struct monitor *find) {
    struct monitor *m;
    for (m = monitor_mast.head; m != NULL; m = m->next) {
        if (find == m)
            break;
    }
    return m;
}

struct monitor *jmp_create_monitor (char *name) {
    struct monitor *m = monitor_find (name);
    if (m == NULL) {
        JVMPI_RawMonitor mon_id;
        mon_id = jvmpi_create_monitor (name);
        trace3 ("MONITOR  create:\"%s\" = %p\n", name, mon_id);
        m = monitor_new(mon_id, name);
        return m;
    }
    fprintf (stderr, "ERROR %s:%d:jmp_create_monitor(): already exists [%p, \"%s\"]\n", __FILE__, __LINE__, m->monitor_id, m->name);
    return NULL;	
}

void jmp_delete_monitor (struct monitor *m) {
    struct monitor *mon = monitor_ptr (m);
    if (mon == NULL) {
        fprintf (stderr, "ERROR %s:%d:jmp_delete_monitor(): does not exist [ptr=%p]\n", __FILE__, __LINE__, m);
        return;
    }
    trace3 ("MONITOR destory:%p \"%s\"\n", m->monitor_id, m->name);
    jvmpi_delete_monitor (m->monitor_id);
    monitor_free (m);
}

void jmp_lock_monitor (struct monitor *m, int which) {
    if (m->use[which] != 0)
       fprintf (stderr, "ERROR[%d] %s:%d:jmp_lock_monitor(): already locked [%p, \"%s\"]\n", which, __FILE__, __LINE__, m->monitor_id, m->name);
    trace3 ("MONITOR   enter:%p \"%s\"\n", m->monitor_id, m->name);
    jvmpi_lock_monitor (m->monitor_id);
    m->use[which]++;	/* FIXME: debug only option needs atomic increment */
}

void jmp_unlock_monitor (struct monitor *m, int which) {
    if (m->use[which] == 0)
       fprintf (stderr, "ERROR[%d] %s:%d:jmp_unlock_monitor(): not locked [%p, \"%s\"]\n", which, __FILE__, __LINE__, m->monitor_id, m->name);
    trace3 ("MONITOR    exit:%p \"%s\"\n", m->monitor_id, m->name);
    jvmpi_unlock_monitor (m->monitor_id);
    m->use[which]--;	/* FIXME: debug only option needs atomic decrement */
}

void jmp_monitor_wait (struct monitor *m, jlong milis) {
    struct monitor *mon = monitor_ptr (m);
    if (mon == NULL) {
        fprintf (stderr, "ERROR %s:%d:jmp_monitor_wait(): does not exist [ptr=%p]\n", __FILE__, __LINE__, m);
        return;
    }
    trace2 ("MONITOR    wait:%p\n", m->monitor_id);
    jvmpi_monitor_wait (m->monitor_id, milis);
}

void jmp_monitor_notify_all (struct monitor *m) {
    struct monitor *mon = monitor_ptr (m);
    if (mon == NULL) {
        fprintf (stderr, "ERROR %s:%d:jmp_monitor_notify_all(): does not exist [ptr=%p]\n", __FILE__, __LINE__, m);
        return;
    }
    trace2 ("MONITOR  notify:%p\n", m->monitor_id);
    jvmpi_monitor_notify_all (m->monitor_id);
}
#endif /* JMPDEBUG_MONITOR */

/* Emacs Local Variables: */
/* Emacs mode:C */
/* Emacs c-indentation-style:"gnu" */
/* Emacs c-hanging-braces-alist:((brace-list-open)(brace-entry-open)(defun-open after)(substatement-open after)(block-close . c-snug-do-while)(extern-lang-open after)) */
/* Emacs c-cleanup-list:(brace-else-brace brace-elseif-brace space-before-funcall) */
/* Emacs c-basic-offset:4 */
/* Emacs End: */
