
/*
 * utils.c -- written for Juice
 *  Copyright (C) 1999, 2000, 2001 Abraham vd Merwe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define _GNU_SOURCE		/* isblank() */

#include <string.h>		/* strcpy(), strlen(), strchr() */
#include <ctype.h>		/* isblank() */
#include <unistd.h>		/* read() stat(),
						 * struct stat,
						 * S_ISLNK(), S_ISREG(), S_ISDIR(), S_ISCHR(), S_ISBLK(), S_ISFIFO(), S_ISSOCK(),
						 * S_IRUSR, S_IWUSR, S_IXUSR, S_IRGRP, S_IWGRP, S_IXGRP, S_IROTH, S_IWOTH, S_IXOTH */
#include <sys/stat.h>	/* stat(),
						 * struct stat,
						 * S_ISLNK(), S_ISREG(), S_ISDIR(), S_ISCHR(), S_ISBLK(), S_ISFIFO(), S_ISSOCK(),
						 * S_IRUSR, S_IWUSR, S_IXUSR, S_IRGRP, S_IWGRP, S_IXGRP, S_IROTH, S_IWOTH, S_IXOTH */
#include <stdlib.h>		/* NULL */

#include "typedefs.h"	/* bool, TRUE, FALSE */

/*
 * Trim blanks from left and right sides of given string.
 */
void trim (char *str)
{
   register char *s = str;
   /* First we trim left side */
   while (isblank (*s)) s++;
   strcpy (str,s);
   /* Now for the right side */
   s = str + strlen (str) - 1;
   while ((s >= str) && isblank (*s)) *(s--) = '\0';
}

/*
 * Read at most count characters from fd and store it in buf. If a
 * '\n' is not found, -1 is returned, otherwise the number of
 * characters read is returned.
 *
 * UPDATE: I've changed the code to accept '\n' since some of my
 * friends have invalid descript.ion's that doesn't have a '\n'
 * after the last entry.
 */
int readline (int fd,char *buf,int count)
{
   register int i;
   buf[0] = '\0';
   i = 0;
   do
	 {
		if (read (fd,buf + i,1) <= 0)
		  {
			 if (i == 0) return -1; else break; /* EOF */
		  }
		if ((buf[i] == '\0') || (buf[i] == '\n')) break;
		i++;
	 }
   while (i < count);
   buf[i] = '\0';
   return i;
}

/*
 * Check if the given file is one of the types specified in
 * type, and that we have the necessary permissions to open
 * the file in the mode specified. Mode can be one or more of
 * the following characters: 'r' (readable), 'w' (writable),
 * 'x' (executable), e.g. "rw" would mean that we want read
 * and write permissions. Type can be one or more of the
 * following characters: 'l' (symbolic link), 'r' (regular
 * file), 'd' (directory), 'c' (character device), 'b' (block
 * device), 'f' (fifo), 's' (socket), e.g. "bsf" would mean
 * that we want a block device, socket, or a fifo. Both type
 * and mode must contain at least one character.
 */
bool fperm (const char *filename,const char *type,const char *mode)
{
   struct stat filestat;
   int i;
   /* Check parameters */
   if ((strlen (type) > 7) || (*type == '\0')) return FALSE;
   if ((strlen (mode) > 3) || (*mode == '\0')) return FALSE;
   for (i = 0; i < (int) strlen (type); i++)
	 if ((type[i] != 'l') &&
		 (type[i] != 'r') &&
		 (type[i] != 'd') &&
		 (type[i] != 'c') &&
		 (type[i] != 'b') &&
		 (type[i] != 'f') &&
		 (type[i] != 's')) return FALSE;
   for (i = 0; i < (int) strlen (mode); i++)
	 if ((mode[i] != 'r') &&
		 (mode[i] != 'w') &&
		 (mode[i] != 'x')) return FALSE;
   /* Get file permissions */
   if (stat (filename,&filestat) < 0) return FALSE;
   /* Check file types */
   if ((strchr (type,'l') != NULL) && (!S_ISLNK (filestat.st_mode))) return FALSE;
   if ((strchr (type,'r') != NULL) && (!S_ISREG (filestat.st_mode))) return FALSE;
   if ((strchr (type,'d') != NULL) && (!S_ISDIR (filestat.st_mode))) return FALSE;
   if ((strchr (type,'c') != NULL) && (!S_ISCHR (filestat.st_mode))) return FALSE;
   if ((strchr (type,'b') != NULL) && (!S_ISBLK (filestat.st_mode))) return FALSE;
   if ((strchr (type,'f') != NULL) && (!S_ISFIFO (filestat.st_mode))) return FALSE;
   if ((strchr (type,'s') != NULL) && (!S_ISSOCK (filestat.st_mode))) return FALSE;
   /* Are we the user? */
   if (filestat.st_uid == getuid ())
	 {
		/* Check file permissions for user */
		if ((strchr (mode,'r') != NULL) && (!(S_IRUSR & filestat.st_mode))) return FALSE;
		if ((strchr (mode,'w') != NULL) && (!(S_IWUSR & filestat.st_mode))) return FALSE;
		if ((strchr (mode,'x') != NULL) && (!(S_IXUSR & filestat.st_mode))) return FALSE;
	 }
   /* Are we in the group then? */
   else if (filestat.st_gid == getgid ())
	 {
		/* Check file permissions for group */
		if ((strchr (mode,'r') != NULL) && (!(S_IRGRP & filestat.st_mode))) return FALSE;
		if ((strchr (mode,'w') != NULL) && (!(S_IWGRP & filestat.st_mode))) return FALSE;
		if ((strchr (mode,'x') != NULL) && (!(S_IXGRP & filestat.st_mode))) return FALSE;
	 }
   /* We're a lowly other then */
   else
	 {
		/* Check file permissions for other */
		if ((strchr (mode,'r') != NULL) && (!(S_IROTH & filestat.st_mode))) return FALSE;
		if ((strchr (mode,'w') != NULL) && (!(S_IWOTH & filestat.st_mode))) return FALSE;
		if ((strchr (mode,'x') != NULL) && (!(S_IXOTH & filestat.st_mode))) return FALSE;
	 }
   return TRUE;
}

/*
 * Copy a string to a volatile string.
 */
void safe_strcpy (volatile char *dest,const char *src)
{
   register int i;
   for (i = 0; src[i] != '\0'; i++) dest[i] = src[i];
   dest[i] = src[i];
}

/*
 * Copy a volatile string to a string.
 */
void safe_strcpy2 (volatile char *dest,volatile const char *src)
{
   register int i;
   for (i = 0; src[i] != '\0'; i++) dest[i] = src[i];
   dest[i] = src[i];
}

