
/*
 * child.c -- written for Juice
 *	Copyright (C) 1999, 2000, 2001 Abraham vd Merwe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef PLAYER_CHILD_C
#define PLAYER_CHILD_C

/* If this is defined, the parent will wait for the child to execute before continuing */
#define SYNC_PTY_SIDES							 /* FIXME: Unnecessary? */

#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <ncurses.h>
#include <termios.h>

#include "typedefs.h"
#include "console.h"

/*
 * Forward declarations
 */

void flush_child_pty ();

/*
 * Definitions
 */

/* Specifies how often sigalrm_handler flushed the child's pty */
#define ALARM_DELAY_IN_SECONDS 1				 /* Don't make this value to big */

/* Initial length of the buffer for all I/O with the child */
#define PTY_BUFFER_SIZE 100						 /* Arbitrary; but keep it >= 80 */

/*
 * Global variables
 */

static int child_pty;							 /* File descriptor of the child's pty */
static char *pty_buffer;						 /* For reading/writing on the child's pty */
static pid_t child_pid;							 /* We save the child pid to kill is later */

static volatile bool child_alive_flag = FALSE;	 /* Flag to check if child is alive */

/*
 * Signal handlers
 */

#ifdef SYNC_PTY_SIDES
/* Handler for SIGUSR1, does nothing but accept the signal */
void sigusr1_handler (int sig)
{
}
#endif

/* Handler for SIGCHLD, waits for children so they don't become zombies and alert */
/* alert parent that child has terminated */
void sigchld_handler (int sig)
{
   int pid,status;
   while ((pid = wait3 (&status,WNOHANG,NULL)) > 0);
   child_alive_flag = FALSE;
}

/* Handler for SIGALRM, flushes the child's pty every now and then */
void sigalrm_handler (int sig)
{
   flush_child_pty ();
   signal (SIGALRM,sigalrm_handler);			 /* Set signal to this function */
   alarm (ALARM_DELAY_IN_SECONDS);				 /* Set alarm */
}

/*
 * Local functions
 */

int pty_open_master (char *pty_name)
{
   int pty_master;
   char *ptr1,*ptr2;
   strcpy (pty_name,"/dev/ptyXX");
   for (ptr1 = "pqrstuvwxyzPQRST"; *ptr1; ++ptr1)
	 {
		pty_name[8] = *ptr1;
		for (ptr2 = "0123456789abcdef"; *ptr2; ++ptr2)
		  {
			 pty_name[9] = *ptr2;
			 /* Try to open master */
			 if ((pty_master = open (pty_name,O_RDWR)) == -1)
			   {
				  if (errno == ENOENT)				 /* Different from EIO */
					return -1;						 /* Out of pty devices */
			   }
			 else continue;							 /* Try next pty device */
			 pty_name[5] = 't';						 /* Change "pty" to "tty" */
			 if (access (pty_name,R_OK | W_OK) == -1)
			   {
				  close (pty_master);
				  pty_name[5] = 'p';
				  continue;
			   }
			 return pty_master;
		  }
	 }
   return -1;		/* Ran out of pty devices */
}

int pty_open_slave (const char *pty_name)
{
   return open (pty_name,O_RDWR);
}

/*
 * All functions from here on starting with child_ is exported, all other is local
 */

bool child_create (const char *command,char *arguments[])
{
   int pty_slave;
#ifdef SYNC_PTY_SIDES
   /* Used to wait for a SIGUSR1 signal from the subprocess */
   sigset_t sigusr1_mask,old_sigusr1_mask;
#endif
   struct sigaction sigchld_act,old_sigchld_act; /* Used to catch SIGCHLD signals */
   static char pty_name[40];					 /* Remembered across calls to create_child */
   /* Open a pty for talking to the child */
   if ((child_pty = pty_open_master (pty_name)) == -1) return FALSE;
   if ((pty_slave = pty_open_slave (pty_name)) == -1) return FALSE;
   /* Initialize the pty's I/O buffer */
   pty_buffer = (char *) malloc (PTY_BUFFER_SIZE);
#ifdef SYNC_PTY_SIDES
   sigemptyset (&sigusr1_mask);
   sigaddset (&sigusr1_mask,SIGUSR1);
   sigprocmask (SIG_BLOCK,&sigusr1_mask,&old_sigusr1_mask);
   signal (SIGUSR1,sigusr1_handler);
#endif
   sigemptyset (&sigchld_act.sa_mask);
   sigchld_act.sa_flags = SA_NOCLDSTOP;			/* Only interested in terminations, */
												/* not stopped (e.g. user pressed Ctrl-Z) */
												/* processes */
   sigchld_act.sa_handler = sigchld_handler;
   sigaction (SIGCHLD,&sigchld_act,&old_sigchld_act);
   child_alive_flag = TRUE;
   /* Fork the child */
   switch ((child_pid = fork ()))
	 {
		/*
		 * Fork failed
		 */
	  case -1:
		return FALSE;
		/*
		 * Child process
		 */
	  case 0:
		if (setsid () == -1) _exit (1);			 /* Request a new session (job control) */
		pty_slave = pty_open_slave (pty_name);	 /* Open the slave side of the pty: again */
		close (child_pty);
		ioctl (pty_slave,TIOCSCTTY,0);			 /* Ensure it is our controlling terminal */
		dup2 (pty_slave,STDIN_FILENO);
		dup2 (pty_slave,STDOUT_FILENO);
		dup2 (pty_slave,STDERR_FILENO);			 /* Attach our file descriptors to the pty */
		close (pty_slave);						 /* These may be FD_CLOEXEC, but just in case... */
#ifdef SYNC_PTY_SIDES
		/* Give our parent process the go-ahead */
		kill (getppid (),SIGUSR1);
#endif
#ifdef WITH_SHELL_SUPPORT
//		setpgid (0,getpid ());
		execv (command,arguments);				 /* Execute child process */
#else
		execvp (command,arguments);				 /* Execute child process */
#endif
		_exit (1);								 /* execl() failed */
		/*
		 * Parent process
		 */
	  default:
#ifdef SYNC_PTY_SIDES
		sigsuspend (&old_sigusr1_mask);
		signal (SIGUSR1,SIG_DFL);
		sigprocmask (SIG_SETMASK,&old_sigusr1_mask,NULL);
#endif
		close (pty_slave);
		if (!child_alive_flag) return FALSE;			 /* Child died immediately */
#ifdef WITH_SHELL_SUPPORT
//		setpgid (child_pid,child_pid);
#endif
		return TRUE;
	 }
}

void feed_child (chtype hotkey)
{
   fd_set read_set;								 /* For select */
   int bytes;									 /* For the return value from read */
   int i;										 /* Loop counter */
   while (TRUE)
	 {
		if (!child_alive_flag) return;
		/* Prepare the file-descriptor set and call select */
		FD_ZERO (&read_set);
		FD_SET (child_pty,&read_set);
		FD_SET (STDIN_FILENO,&read_set);
		if (select (FD_SETSIZE,&read_set,NULL,NULL,NULL) == -1)
		  {
			 /* Despite using SA_RESTART, we still have to check for this */
			 if (errno == EINTR) continue;		 /* try all over again */
			 return; /* exit (1); */
		  }
		if (FD_ISSET (child_pty,&read_set))
	        {
			   bytes = read (child_pty,pty_buffer,PTY_BUFFER_SIZE);
			   if ((bytes == -1) && (errno != EIO)) return; /* exit (1); */
			   console_write (pty_buffer,bytes);
			   write (STDERR_FILENO,pty_buffer,bytes);
			}
		else if (FD_ISSET (STDIN_FILENO,&read_set))
		  {
			 bytes = read (STDIN_FILENO,pty_buffer,PTY_BUFFER_SIZE);
			 if (bytes == -1) return; /* exit (1); */
			 for (i = 0; i < bytes; ++i) if (pty_buffer[i] == (int) hotkey)
			   {
				  write (child_pty,pty_buffer,i);
				  return;
			   }
			 write (child_pty,pty_buffer,bytes);
		  }
		else return;
	 }
}

void child_invoke (chtype hotkey)
{
//   struct termios saved_mode;					 /* Backup of old terminal mode */
   struct termios raw_mode;						 /* Fiddle with terminal modes */
   tcgetattr (STDERR_FILENO,&raw_mode);
//   tcgetattr (STDERR_FILENO,&saved_mode);
   raw_mode.c_lflag &= ~ICANON;				 	 /* Disable line-editing chars, etc. */
   raw_mode.c_lflag &= ~ISIG;					 /* Disable intr, quit & suspend chars */
   raw_mode.c_lflag &= ~ECHO;					 /* Disable input echoing */
   raw_mode.c_iflag &= ~IXON;					 /* Pass ^S/^Q to child undisturbed */
   raw_mode.c_iflag &= ~ICRNL;					 /* Don't translate CRs into LFs */
   raw_mode.c_oflag &= ~OPOST;					 /* Don't postprocess output */
   raw_mode.c_cc[VTIME] = 0;					 /* IE: wait forever, and return as */
   raw_mode.c_cc[VMIN] = 1;					 	 /* soon as a character is available */
   tcsetattr (STDERR_FILENO,TCSANOW,&raw_mode);
   feed_child (hotkey);
//   tcsetattr (STDERR_FILENO,TCSANOW,&saved_mode);
}

void flush_child_pty ()
{
   fd_set read_set;								 /* For select */
   int bytes;									 /* For the return value from read */
   struct timeval tv;
   tv.tv_sec = 0;
   tv.tv_usec = 500;
   while (TRUE)
	 {
		if (!child_alive_flag) return;
		/* Prepare the file-descriptor set and call select */
		FD_ZERO (&read_set);
		FD_SET (child_pty,&read_set);
		if (select (FD_SETSIZE,&read_set,NULL,NULL,&tv) == -1)
		  {
			 /* Despite using SA_RESTART, we still have to check for this */
			 if (errno == EINTR) continue;		 /* try all over again */
			 return; /* exit (1); */
		  }
		if (FD_ISSET (child_pty,&read_set))
	        {
			   bytes = read (child_pty,pty_buffer,PTY_BUFFER_SIZE);
			   if ((bytes == -1) && (errno != EIO)) return; /* exit (1); */
			   console_write (pty_buffer,bytes);
			}
		else return;
	 }
}

void child_flush_start ()
{
   sigalrm_handler (0);
}

void child_flush_stop ()
{
   signal (SIGALRM,SIG_DFL);					 /* Restore default SIGALRM handler */
   alarm (0);									 /* Cancel any scheduled alarms */
}

void child_destroy ()
{
   kill (child_pid,SIGQUIT);					 /* Quit child process */
   while (child_alive_flag) ;					 /* Wait for child to die */
   signal (SIGCHLD,SIG_DFL);					 /* Restore old SIGCHLD handler */
   close (child_pty);							 /* Close child pty */
   /* FIXME: What about closing the slave_pty, etc? */
}

bool child_alive ()
{
   return child_alive_flag;
}

void child_pause ()
{
   kill (-child_pid,SIGSTOP);
}

void child_resume ()
{
   kill (-child_pid,SIGCONT);
}

void child_skip ()
{
   kill (-child_pid,SIGINT);
}

#endif
