#ifndef K3DSDK_IDOCUMENT_H
#define K3DSDK_IDOCUMENT_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares idocument, an abstract interface for an open K-3D document
		\author Tim Shead (tshead@k-3d.com)
*/

#include "iunknown.h"
#include "iobject.h"

#include "custom_marshallers.h"
#include "signal_system.h"

#include <string>

// Forward declarations
namespace boost { namespace filesystem { class path; } }

namespace k3d
{

// Forward declarations
class idag;
class igeometry_read_format;
class igeometry_write_format;
class imouse_event_observer;
class iobject_collection;
class iselection;
class istate_recorder;

/// Abstract interface for an open K-3D document
class idocument :
	public virtual iunknown
{
public:
	/// Returns the collection of K-3D objects within this document
	virtual k3d::iobject_collection& objects() = 0;
	/// Returns the current selection for this document
	virtual k3d::iselection& selection() = 0;
	/// Returns the directed acyclic graph of dependencies between object properties within this document
	virtual k3d::idag& dag() = 0;
	/// Returns the k3d::istate_recorder interface for this document
	virtual k3d::istate_recorder& state_recorder() = 0;

	/// Returns the full filepath to the document (could be empty)
	virtual const boost::filesystem::path path() = 0;
	/// Returns the document title (could be empty string)
	virtual const std::string title() = 0;
	/// Saves the document to the given file
	virtual bool save(const boost::filesystem::path& File) = 0;

	/// Sets the current mouse input focus (NULL is allowed)
	virtual void set_mouse_focus(imouse_event_observer* const Observer) = 0;
	/// Returns the current mouse input focus (could return NULL)
	virtual imouse_event_observer* mouse_focus() = 0;

	// Signals

	/// Defines a signal emitted before the document is closed - observers can return false to cancel closing (e.g. if a resource is unsaved), true otherwise
	typedef SigC::Signal0<bool, marshaller::cancelable> safe_to_close_signal_t;
	virtual safe_to_close_signal_t& safe_to_close_signal() = 0;

	/// Defines a signal emitted when the document closes
	typedef SigC::Signal0<void> close_signal_t;
	virtual close_signal_t& close_signal() = 0;

	/// Defines a signal emitted when the document title changes
	typedef SigC::Signal0<void> title_signal_t;
	virtual title_signal_t& title_signal() = 0;

	/// Defines a signal emitted when the mouse focus changes (note: the observer could be NULL)
	typedef SigC::Signal0<void> mouse_focus_signal_t;
	virtual mouse_focus_signal_t& mouse_focus_signal() = 0;

protected:
	idocument() {}
	idocument(const idocument& RHS) {}
	idocument& operator = (const idocument& RHS) { return *this; }
	virtual ~idocument() {}
};

} // namespace k3d

#endif // K3DSDK_IDOCUMENT_H


