#ifndef K3DUI_COLOR_CHOOSER_H
#define K3DUI_COLOR_CHOOSER_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include "k3dcontrol.h"

#include <k3dsdk/color.h>
#include <k3dsdk/iproperty.h>
#include <k3dsdk/iwritable_property.h>

#include <sdpgtk/sdpgtkopengldrawingarea.h>

// Forward declarations
class k3d::istate_recorder;
class sdpGtkIObjectContainer;
class sdpxml::Element;

namespace k3d
{

namespace color_chooser
{

/////////////////////////////////////////////////////////////////////////////
// idata_proxy

/// Abstract interface for an object that proxies a data source for a color chooser control (i.e. the "model" in model-view-controller)
class idata_proxy
{
public:
	virtual ~idata_proxy() {}

	/// Called to return the underlying data value
	virtual k3d::color value() = 0;
	/// Called to set a new data value
	virtual void set_value(const k3d::color Value) = 0;
	/// Signal emitted if the underlying data changes
	typedef SigC::Signal0<void> changed_signal_t;
	/// Signal emitted if the underlying data changes
	virtual changed_signal_t& changed_signal() = 0;

protected:
	idata_proxy() {}
	idata_proxy(const idata_proxy& RHS) {}
	idata_proxy& operator=(const idata_proxy& RHS) { return *this; }
};

/////////////////////////////////////////////////////////////////////////////
// control

/// Provides a UI for manipulating colors (i.e. the view and the controller from model-view-controller)
class control :
	public k3dControl
{
	typedef k3dControl base;

public:
	control(k3d::iunknown* const CommandNodeParent, const std::string CommandNodeName);
	~control();

	/// Called by the framework when instantiating the button from a GTKML (XML) document
	bool Create(sdpGtkIObjectContainer* const ObjectContainer, sdpxml::Document& Document, sdpxml::Element& Element);

	/// Attaches the button to the data it fronts for
	bool attach(std::auto_ptr<idata_proxy> Data, k3d::istate_recorder* const StateRecorder, const std::string StateChangeName);

	const std::string CustomType() const;

	bool execute_command(const std::string& Command, const std::string& Arguments);

private:
	/// Called whenever the the button state needs to be updated
	void update();

	/// Called to handle SDPGTK events
	void OnEvent(sdpGtkEvent* Event);
	/// Called to draw the current value
	void on_draw_value();
	/// Calle to modify the current value
	void on_edit_value();
	/// Called when the GTK+ widgets are about to disappear
	void on_destroy();
	/// Called when the user toggles the button
	void on_toggle();

	/// Stores a reference to the underlying data object
	std::auto_ptr<idata_proxy> m_data;
	/// OpenGL widget
	sdpGtkOpenGLDrawingArea m_value_widget;
};

/// Provides an implementation of k3d::color_chooser::idata_proxy that supports any data source that supports the value(), set_value(), and changed_signal() concepts
template<typename data_t>
class data_proxy :
	public idata_proxy
{
public:
	data_proxy(data_t& Data) :
		m_data(Data)
	{
	}

	k3d::color value()
	{
		return m_data.value();
	}

	void set_value(const k3d::color Value)
	{
		m_data.set_value(Value);
	}

	changed_signal_t& changed_signal()
	{
		return m_data.changed_signal();
	}

private:
	data_proxy(const data_proxy& RHS);
	data_proxy& operator=(const data_proxy& RHS);
	~data_proxy() {}

	data_t& m_data;
};

/// Specialization of k3d::color_chooser::data_proxy for use with k3d::iproperty objects
template<>
class data_proxy<iproperty> :
	public idata_proxy
{
public:
	typedef iproperty data_t;
	
	data_proxy(data_t& Data) :
		m_readable_data(Data),
		m_writable_data(dynamic_cast<iwritable_property*>(&Data))
	{
	}

	k3d::color value()
	{
		return boost::any_cast<k3d::color>(m_readable_data.value());
	}

	void set_value(const k3d::color Value)
	{
		return_if_fail(m_writable_data);
		m_writable_data->set_value(Value);
	}

	changed_signal_t& changed_signal()
	{
		return m_readable_data.changed_signal();
	}

private:
	data_proxy(const data_proxy& RHS);
	data_proxy& operator=(const data_proxy& RHS);
	~data_proxy() {}

	data_t& m_readable_data;
	iwritable_property* const m_writable_data;
};

/// Convenience factory function for creating color_chooser::idata_proxy objects
template<typename data_t>
std::auto_ptr<idata_proxy> proxy(data_t& Data)
{
	return std::auto_ptr<idata_proxy>(new data_proxy<data_t>(Data));
}

} // namespace color_chooser

} // namespace k3d

#endif // K3DUI_COLOR_CHOOSER_H




