#ifndef K3DUI_GTKML_H
#define K3DUI_GTKML_H

#include <k3dsdk/application.h>
#include <k3dsdk/result.h>

#include <sdpxml/sdpxml.h>

#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/path.hpp>

namespace k3d
{

/// Sets the root path to GTKML templates / external objects (such as pixmaps) ... note: must be called once-and-only-once, prior to any calls to dialog_template_path()
void set_dialog_template_path(const boost::filesystem::path& Path);
/// Returns the root path to GTKML templates / external objects (such as pixmaps) ... note: set_dialog_template_path() must be called prior to calling this function
const boost::filesystem::path dialog_template_path();

/// Given the path to a GTKML template, returns the path to a template written in the user's preferred language, if available.  If not, returns the input path.
const boost::filesystem::path preferred_language_template(const boost::filesystem::path& Template);

/// Given the path to a GTKML template and an object deriving from sdpGtkObjectContainer, loads the template.  Returns true on success, false on failure.
template<typename container_t>
bool load_gtkml(const boost::filesystem::path& Template, container_t& Target)
{
	boost::filesystem::ifstream stream(Template);
	return load_gtkml(stream, Template.native_file_string(), Target);
}

/// Given an input stream containing a GTKML template and an object deriving from sdpGtkObjectContainer, loads the template.  Returns true on success, false on failure.
template<typename container_t>
bool load_gtkml(std::istream& Template, const std::string& TemplateName, container_t& Target)
{
	sdpxml::Document gtkml("empty");
	return_val_if_fail(gtkml.Load(Template, TemplateName), false);
	return load_gtkml(gtkml, TemplateName, Target);
}

template<typename container_t>
bool load_gtkml(sdpxml::Document& Document, const std::string& TemplateName, container_t& Target)
{
	return_val_if_fail(Target.Load(Document, dialog_template_path()), false);
	return true;
}

} // namespace k3d

#endif // !K3DUI_GTKML_H

