// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/auto_ptr.h>
#include <k3dsdk/bitmap_source.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/file_filter.h>
#include <k3dsdk/ibitmap_read_format.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/module.h>
#include <k3dsdk/object.h>
#include <k3dsdk/path_data.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>
#include <k3dsdk/xml_utility.h>

#include <boost/filesystem/operations.hpp>

#include <iterator>

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_file_in_implementation

class bitmap_file_in_implementation :
	public k3d::bitmap_source<k3d::persistent<k3d::object> >
{
	typedef k3d::bitmap_source<k3d::persistent<k3d::object> > base;

public:
	bitmap_file_in_implementation(k3d::idocument& Document) :
		base(Document),
		m_file(k3d::init_name("file") + k3d::init_description("File [string]") + k3d::init_document(Document) + k3d::init_value(boost::filesystem::path()))
	{
		enable_serialization(k3d::persistence::proxy(m_file));
		
		register_property(m_file);

		m_file.changed_signal().connect(SigC::slot(*this, &bitmap_file_in_implementation::on_reset_bitmap));
		m_output.need_data_signal().connect(SigC::slot(*this, &bitmap_file_in_implementation::on_create_bitmap));
	}

	void on_reset_bitmap()
	{
		m_output.reset();
	}
	
	k3d::bitmap* on_create_bitmap()
	{
		std::auto_ptr<k3d::bitmap> bitmap(new k3d::bitmap());

		const boost::filesystem::path file = m_file.property_value();
		if(!boost::filesystem::exists(file))
			return 0;

		k3d::auto_ptr<k3d::ibitmap_read_format> filter(k3d::auto_file_filter<k3d::ibitmap_read_format>(file));
		return_val_if_fail(filter.get(), 0);

		return_val_if_fail(filter->read_file(file, *bitmap), 0);

		return bitmap.release();
	}
	
	void load(sdpxml::Document& Document, sdpxml::Element& Element)
	{
		base::load(Document, Element);

		// Look for a stored relative path ...
		const boost::filesystem::path document_path(document().path().branch_path());
		const boost::filesystem::path absolute_path(m_file.value());
		const boost::filesystem::path relative_path(sdpxml::GetAttribute(k3d::xml::safe_element(k3d::xml::safe_element(Element, "variables"), sdpxml::Element("variable", "", sdpxml::Attribute("name", "relative_path"))), "value", std::string()), boost::filesystem::native);
		const boost::filesystem::path calculated_path(document_path / relative_path);

/*
std::cerr << debug << document_path.native_file_string() << std::endl;
std::cerr << debug << relative_path.native_file_string() << std::endl;
std::cerr << debug << calculated_path.native_file_string() << std::endl;
*/
		
		if(relative_path.empty())
			{
				std::cerr << info << "Using absolute path " << absolute_path.native_file_string() << " to load bitmap" << std::endl;
				return;
			}
			
		if(boost::filesystem::exists(calculated_path))
			{
				std::cerr << info << "Using calculated path " << calculated_path.native_file_string() << " to load bitmap" << std::endl;
				m_file.set_value(calculated_path);
			}
		else
			{
				std::cerr << error << "Couldn't locate calculated path " << calculated_path.native_file_string() << ", falling-back on absolute path " << absolute_path.native_file_string() << std::endl;
			}
	}

	void save(sdpxml::Element& Element, k3d::idependencies& Dependencies)
	{
		base::save(Element, Dependencies);

		const boost::filesystem::path document_path(document().path().branch_path());
		const boost::filesystem::path absolute_path(m_file.value());
		
		boost::filesystem::path relative_path;
		
		boost::filesystem::path::iterator a = document_path.begin();
		boost::filesystem::path::iterator b = absolute_path.begin();
		while(a != document_path.end() && b != absolute_path.end() && *a == *b)
			{
				++a;
				++b;
			}	
			
		for(; a != document_path.end(); ++a)
			relative_path /= "..";
			
		for(; b != absolute_path.end(); ++b)
			relative_path /= *b;

/*
std::cerr << debug << document_path.native_file_string() << std::endl;
std::cerr << debug << absolute_path.native_file_string() << std::endl;
std::cerr << debug << relative_path.native_file_string() << std::endl;
*/

		k3d::xml::safe_element(Element, "variables").Append(sdpxml::Element("variable", "", sdpxml::Attribute("name", "relative_path"), sdpxml::Attribute("value", relative_path.native_file_string())));
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_file_in_implementation>,
			k3d::interface_list<k3d::ibitmap_source> > factory(
				k3d::classes::FileBitmap(),
				"BitmapFileIn",
				"Loads a bitmap from the filesystem",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data_property(boost::filesystem::path, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_file;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_file_in_factory

k3d::iplugin_factory& bitmap_file_in_factory()
{
	return bitmap_file_in_implementation::get_factory();
}

} // namespace libk3dbitmap

