// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief An object that take two doubles as input and produce their sum as output
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/object.h>
#include <k3dsdk/computed_property.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/module.h>

namespace libk3dcore
{

class scalar_sine_implementation : 
	public k3d::persistent<k3d::object> 
{
	typedef k3d::persistent<k3d::object>  base;
public:
	scalar_sine_implementation(k3d::idocument& Document) : 
		base(Document),
		m_input(k3d::init_name("input") + k3d::init_description("Input [double]") + k3d::init_value(0.0) + k3d::init_document(Document)),
		m_output("output", "Sine wave [double]", k3d::method_call(*this, &scalar_sine_implementation::get_value))
	{
		register_property(m_input);
		register_property(m_output);

		enable_serialization(k3d::persistence::proxy(m_input));

		m_input.changed_signal().connect(m_output.changed_signal().slot());
	}
	
	double get_value()
	{
		return std::sin(m_input.property_value());
	}
	
	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<scalar_sine_implementation> > factory(
			k3d::uuid(0xb00248d6, 0x9d6843d5, 0xab5a24bc, 0x852519e5),
			"ScalarSine", 
			"Applies a sine function to its input", 
			"Objects", 
			k3d::iplugin_factory::STABLE);
		return factory;
	}
	
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}
	
private:
	k3d_data_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_input;
	k3d::computed_property<double, k3d::method_call_t<scalar_sine_implementation, double> > m_output;
	
};

k3d::iplugin_factory& scalar_sine_factory()
{
	return scalar_sine_implementation::get_factory();
}

} //namespace libk3dcore
