// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/axis.h>
#include <k3dsdk/bounding_box.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/mouse_event_observer.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace libk3ddeformation
{

/////////////////////////////////////////////////////////////////////////////
// center_points_implementation

class center_points_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	center_points_implementation(k3d::idocument& Document) :
		base(Document),
		m_center_x(k3d::init_name("center_x") + k3d::init_description("Center X [boolean]") + k3d::init_value(true) + k3d::init_document(Document)),
		m_center_y(k3d::init_name("center_y") + k3d::init_description("Center Y [boolean]") + k3d::init_value(true) + k3d::init_document(Document)),
		m_center_z(k3d::init_name("center_z") + k3d::init_description("Center Z [boolean]") + k3d::init_value(true) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_center_x));
		enable_serialization(k3d::persistence::proxy(m_center_y));
		enable_serialization(k3d::persistence::proxy(m_center_z));

		register_property(m_center_x);
		register_property(m_center_y);
		register_property(m_center_z);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &center_points_implementation::on_reset_geometry));

		m_center_x.changed_signal().connect(SigC::slot(*this, &center_points_implementation::on_reshape_geometry));
		m_center_y.changed_signal().connect(SigC::slot(*this, &center_points_implementation::on_reshape_geometry));
		m_center_z.changed_signal().connect(SigC::slot(*this, &center_points_implementation::on_reshape_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &center_points_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	void on_reshape_geometry()
	{
		if(m_output_mesh.empty())
			return;

		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return;

		k3d::mesh* const output = const_cast<k3d::mesh*>(m_output_mesh.value());
		if(!output)
			return;

		reshape_geometry(*input, *output);

		m_output_mesh.changed_signal().emit();
	}

	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Otherwise, we make a copy of the input mesh and modify the copy ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		reshape_geometry(*input, *output);

		return output;
	}

	void reshape_geometry(const k3d::mesh& Source, k3d::mesh& Target)
	{
		// Sanity checks ...
		assert(Source.points.size() == Target.points.size());

		const k3d::bounding_box bounds = k3d::bounds(Source);
		if(bounds.empty())
			return;

		const bool ignore_selection = !k3d::contains_selection(Source);
		const bool center_x = m_center_x.property_value();
		const bool center_y = m_center_y.property_value();
		const bool center_z = m_center_z.property_value();

		const k3d::vector3 offset(
			center_x ? -0.5 * (bounds.px + bounds.nx) : 0,
			center_y ? -0.5 * (bounds.py + bounds.ny) : 0,
			center_z ? -0.5 * (bounds.pz + bounds.nz) : 0);

		for(unsigned long i = 0; i != Target.points.size(); ++i)
			{
				if(ignore_selection || Source.points[i]->selected)
					{
						Target.points[i]->position = Source.points[i]->position + offset;
					}
				else
					{
						Target.points[i]->position = Source.points[i]->position;
					}
			}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<center_points_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0xc030994f, 0x97ed4165, 0x9ac0b7f0, 0xa777abbc),
				"CenterPoints",
				"Centers mesh points around the origin",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_center_x;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_center_y;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_center_z;
};

/////////////////////////////////////////////////////////////////////////////
// center_points_factory

k3d::iplugin_factory& center_points_factory()
{
	return center_points_implementation::get_factory();
}

} // namespace libk3ddeformation


