// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace libk3dmesh
{

namespace detail
{

/// std::pair equivalent that maintains the order of its members
template<typename T1, typename T2>
class ordered_pair;

template<typename T1, typename T2>
bool operator<(const ordered_pair<T1,T2>& lhs, const ordered_pair<T1,T2>& rhs);

template<typename T1, typename T2>
class ordered_pair :
	public std::pair<T1, T2>
{
public:
	typedef T1 first_type;
	typedef T2 second_type;

	T1 first;
	T2 second;

	explicit ordered_pair()
	{
	}

	explicit ordered_pair(const T1& First, const T2& Second) :
		first(First < Second ? First : Second),
		second(First < Second ? Second : First)
	{
	}

	explicit ordered_pair(const k3d::split_edge* Edge) :
		first(Edge->vertex < Edge->face_clockwise->vertex ? Edge->vertex : Edge->face_clockwise->vertex),
		second(Edge->vertex < Edge->face_clockwise->vertex ? Edge->face_clockwise->vertex : Edge->vertex)
	{
	}

	friend bool operator< <>(const ordered_pair& lhs, const ordered_pair& rhs);
};

template<typename T1, typename T2>
bool operator<(const ordered_pair<T1,T2>& lhs, const ordered_pair<T1,T2>& rhs)
{
	if(lhs.first != rhs.first)
		return lhs.first < rhs.first;

	return lhs.second < rhs.second;
}

typedef ordered_pair<k3d::point*, k3d::point*> ordered_edge_t;
typedef std::set<ordered_edge_t> ordered_edges_t;

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// edges_to_blobby_implementation

class edges_to_blobby_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	edges_to_blobby_implementation(k3d::idocument& Document) :
		base(Document),
		m_radius(k3d::init_name("radius") + k3d::init_description("Segments radius [number]") + k3d::init_document(Document) + k3d::init_value(1.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_type(k3d::init_name("type") + k3d::init_description("Operation [enumeration]") + k3d::init_enumeration(operation_values()) + k3d::init_value(MAX) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_radius));
		enable_serialization(k3d::persistence::proxy(m_type));

		register_property(m_radius);
		register_property(m_type);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &edges_to_blobby_implementation::on_reset_geometry));

		m_radius.changed_signal().connect(SigC::slot(*this, &edges_to_blobby_implementation::on_reset_geometry));
		m_type.changed_signal().connect(SigC::slot(*this, &edges_to_blobby_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &edges_to_blobby_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.property_value();
		return_val_if_fail(input, 0);

		// Otherwise, we convert input mesh polyhedra to blobbies ...
		k3d::mesh* const output = new k3d::mesh();
		update_geometry(*input, *output);

		return output;
	}

	void update_geometry(const k3d::mesh& Input, k3d::mesh& Output)
	{
		std::map<k3d::point*, k3d::point*> point_map;
		for(k3d::mesh::points_t::const_iterator point = Input.points.begin(); point != Input.points.end(); ++point)
			{
				k3d::point* const new_point = new k3d::point(**point);
				Output.points.push_back(new_point);
				point_map[*point] = new_point;
			}

		// Collect edges ...
		detail::ordered_edges_t edges;
		for(k3d::mesh::polyhedra_t::const_iterator polyhedron_iterator = Input.polyhedra.begin(); polyhedron_iterator != Input.polyhedra.end(); ++polyhedron_iterator)
			for(k3d::polyhedron::edges_t::const_iterator edge = (*polyhedron_iterator)->edges.begin(); edge != (*polyhedron_iterator)->edges.end(); edge++)
				{
					k3d::point* p1 = (*edge)->vertex;
					if((*edge)->face_clockwise)
						{
							k3d::point* p2 = (*edge)->face_clockwise->vertex;
							if(p1 && p2)
								edges.insert(detail::ordered_edge_t(point_map[p1], point_map[p2]));
						}
				}

		const double radius = m_radius.property_value();
		k3d::blobby::variable_operands* new_blobby = 0;
		switch(m_type.property_value())
			{
				case ADD:
					new_blobby = new k3d::blobby::add();
					break;
				case MULT:
					new_blobby = new k3d::blobby::multiply();
					break;
				case MIN:
					new_blobby = new k3d::blobby::min();
					break;
				default:
					new_blobby = new k3d::blobby::max();
					break;
			}

		for(detail::ordered_edges_t::const_iterator edge = edges.begin(); edge != edges.end(); ++edge)
			new_blobby->add_operand(new k3d::blobby::segment(edge->first, edge->second, radius, k3d::identity3D()));

		Output.blobbies.push_back(new k3d::blobby(new_blobby));
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<edges_to_blobby_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink> > > factory(
				k3d::uuid(0xc6a00316, 0x72a54b1a, 0xb9ac478e, 0x00fdfc6c),
				"EdgesToBlobby",
				"Converts input edges to segment blobbies",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	typedef enum
	{
		ADD,
		MULT,
		MIN,
		MAX
	} operation_t;

	friend const k3d::ienumeration_property::values_t& operation_values()
	{
		static k3d::ienumeration_property::values_t values;
		if(values.empty())
			{
				values.push_back(k3d::ienumeration_property::value_t("Addition", "addition", "Combine blobby segments with BlobbyAdd"));
				values.push_back(k3d::ienumeration_property::value_t("Multiplication", "multiplication", "Combine blobby segments with BlobbyMult"));
				values.push_back(k3d::ienumeration_property::value_t("Minimum", "minimum", "Combine blobby segments with BlobbyMin"));
				values.push_back(k3d::ienumeration_property::value_t("Maximum", "maximum", "Combine blobby segments with BlobbyMax"));
			}

		return values;
	}

	friend std::ostream& operator << (std::ostream& Stream, const operation_t& Value)
	{
		switch(Value)
			{
				case ADD:
					Stream << "addition";
					break;
				case MULT:
					Stream << "multiplication";
					break;
				case MIN:
					Stream << "minimum";
					break;
				case MAX:
					Stream << "maximum";
					break;
			}

		return Stream;
	}

	friend std::istream& operator >> (std::istream& Stream, operation_t& Value)
	{
		std::string text;
		Stream >> text;

		if(text == "addition")
			Value = ADD;
		else if(text == "multiplication")
			Value = MULT;
		else if(text == "minimum")
			Value = MIN;
		else if(text == "maximum")
			Value = MAX;
		else
			std::cerr << __PRETTY_FUNCTION__ << ": unknown enumeration [" << text << "]"<< std::endl;

		return Stream;
	}

	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_radius;
	k3d_enumeration_property(operation_t, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_type;
};

/////////////////////////////////////////////////////////////////////////////
// edges_to_blobby_factory

k3d::iplugin_factory& edges_to_blobby_factory()
{
	return edges_to_blobby_implementation::get_factory();
}

} // namespace libk3dmesh


