// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/basic_math.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/renderman.h>
#include <k3dsdk/transform.h>

#include <boost/random/lagged_fibonacci.hpp>
#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// lawn_implementation

class lawn_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	lawn_implementation(k3d::idocument& Document) :
		base(Document),
		m_width(k3d::init_name("width") + k3d::init_description("Lawn width [number]") + k3d::init_document(Document) + k3d::init_value(10) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_depth(k3d::init_name("depth") + k3d::init_description("Lawn depth [number]") + k3d::init_document(Document) + k3d::init_value(10) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_blade_count(k3d::init_name("blade_count") + k3d::init_description("Blade count [unsigned integer]") + k3d::init_value(100) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(0L)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_blade_width(k3d::init_name("blade_width") + k3d::init_description("Blade width [number]") + k3d::init_document(Document) + k3d::init_value(0.2) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_blade_height(k3d::init_name("blade_height") + k3d::init_description("Blade height [number]") + k3d::init_document(Document) + k3d::init_value(2) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::distance)))
	{
		enable_serialization(k3d::persistence::proxy(m_width));
		enable_serialization(k3d::persistence::proxy(m_depth));
		enable_serialization(k3d::persistence::proxy(m_blade_count));
		enable_serialization(k3d::persistence::proxy(m_blade_width));
		enable_serialization(k3d::persistence::proxy(m_blade_height));

		register_property(m_width);
		register_property(m_depth);
		register_property(m_blade_count);
		register_property(m_blade_width);
		register_property(m_blade_height);
		
		m_material.changed_signal().connect(SigC::slot(*this, &lawn_implementation::on_reset_geometry));
	
		m_width.changed_signal().connect(SigC::slot(*this, &lawn_implementation::on_reset_geometry));
		m_depth.changed_signal().connect(SigC::slot(*this, &lawn_implementation::on_reset_geometry));
		m_blade_count.changed_signal().connect(SigC::slot(*this, &lawn_implementation::on_reset_geometry));
		m_blade_width.changed_signal().connect(SigC::slot(*this, &lawn_implementation::on_reset_geometry));
		m_blade_height.changed_signal().connect(SigC::slot(*this, &lawn_implementation::on_reset_geometry));
		
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &lawn_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		const double lawn_width = m_width.property_value();
		const double lawn_depth = m_depth.property_value();
		const unsigned long blade_count = m_blade_count.property_value();
		const double blade_width = m_blade_width.property_value();
		const double blade_height = m_blade_height.property_value();

		boost::lagged_fibonacci607 random;

		k3d::mesh* const mesh = new k3d::mesh();

		k3d::cubic_curve_group* const group = new k3d::cubic_curve_group();
		group->material = m_material.interface();

		for(unsigned long i = 0; i != blade_count; ++i)
			{
				k3d::cubic_curve* const curve = new k3d::cubic_curve();

				const k3d::vector3 base = k3d::vector3((random()-0.5) * lawn_width, 0, (random()-0.5) * lawn_depth);
				const k3d::vector3 n1 = base + k3d::vector3(0, blade_height * 0.5, 0);
				const k3d::vector3 n2 = base + k3d::vector3(0, blade_height, 0);
				const k3d::vector3 tip = base + k3d::vector3((random()-0.5) * blade_height, blade_height, (random()-0.5) * blade_height);
			
				k3d::point* const base_p = new k3d::point(base);
				k3d::point* const n1_p = new k3d::point(n1);
				k3d::point* const n2_p = new k3d::point(n2);
				k3d::point* const tip_p = new k3d::point(tip);
			
				mesh->points.push_back(base_p);
				mesh->points.push_back(n1_p);
				mesh->points.push_back(n2_p);
				mesh->points.push_back(tip_p);

				curve->control_points.push_back(base_p);
				curve->control_points.push_back(n1_p);
				curve->control_points.push_back(n2_p);
				curve->control_points.push_back(tip_p);

				curve->varying_data.resize(2);
				
				curve->varying_data[0]["width"] = blade_width;
				curve->varying_data[1]["width"] = 0.0;

				curve->varying_data[0]["N"] = k3d::ri::normal(k3d::vector3(random()-0.5, 0, random()-0.5));
				curve->varying_data[1]["N"] = k3d::ri::normal(k3d::vector3(random()-0.5, 0, random()-0.5));

				group->curves.push_back(curve);
			}

		mesh->cubic_curve_groups.push_back(group);

		return mesh;
	}
	
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<lawn_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
			k3d::uuid(0xc114dc24, 0xa659409b, 0x8909fa57, 0x1e228231),
			"Lawn",
			"Creates a simple grass lawn using cubic curves",
			"Objects",
			k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_width;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_depth;
	k3d_measurement_property(long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_blade_count;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_blade_width;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_blade_height;
};

/////////////////////////////////////////////////////////////////////////////
// lawn_factory

k3d::iplugin_factory& lawn_factory()
{
	return lawn_implementation::get_factory();
}

} // namespace libk3dmesh
