// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/algebra.h>
#include <k3dsdk/computed_property.h>
#include <k3dsdk/itime_sink.h>
#include <k3dsdk/itransform_sink.h>
#include <k3dsdk/itransform_source.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/module.h>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// make_path_implementation

class make_path_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >,
	public k3d::itime_sink,
	public k3d::itransform_source,
	public k3d::itransform_sink
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	make_path_implementation(k3d::idocument& Document) :
		base(Document),
		m_input(k3d::init_name("input_matrix") + k3d::init_description("Input matrix [matrix4]") + k3d::init_value(k3d::identity3D()) + k3d::init_document(Document)),
		m_start_time(k3d::init_name("start_time") + k3d::init_description("Start Time [number]") + k3d::init_document(Document) + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::time))),
		m_time(k3d::init_name("time") + k3d::init_description("Time [number]") + k3d::init_document(Document) + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::time))),
		m_end_time(k3d::init_name("end_time") + k3d::init_description("End Time [number]") + k3d::init_document(Document) + k3d::init_value(1.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::time))),
		m_output("output_matrix", "Output matrix [matrix4]", k3d::method_call(*this, &make_path_implementation::output_value))
	{
		enable_serialization(k3d::persistence::proxy(m_start_time));
		enable_serialization(k3d::persistence::proxy(m_time));
		enable_serialization(k3d::persistence::proxy(m_end_time));

		register_property(m_input);
		register_property(m_start_time);
		register_property(m_time);
		register_property(m_end_time);
		register_property(m_output);
		
		m_input.changed_signal().connect(m_output.changed_signal().slot());
		m_start_time.changed_signal().connect(m_output.changed_signal().slot());
		m_time.changed_signal().connect(m_output.changed_signal().slot());
		m_end_time.changed_signal().connect(m_output.changed_signal().slot());
	}

	k3d::iproperty& time_sink_input()
	{
		return m_time;
	}

	k3d::iproperty& transform_source_output()
	{
		return m_output;
	}
	
	k3d::iproperty& transform_sink_input()
	{
		return m_input;
	}

	std::vector<double> get_segment_lengths(const k3d::linear_curve& Curve)
	{
		std::vector<double> results;

		const unsigned long point_count = Curve.control_points.size();		
		for(unsigned long i = 0; i != point_count; ++i)
			results.push_back((Curve.control_points[(i+1)%point_count]->position - Curve.control_points[i]->position).Length());

		return results;
	}

	k3d::matrix4 get_transformation(const k3d::linear_curve& Curve, const bool Wrap, const double Position)
	{
		const std::vector<double> segment_lengths = get_segment_lengths(Curve);
		if(segment_lengths.empty())
			return k3d::identity3D();

		const double total_length = std::accumulate(segment_lengths.begin(), segment_lengths.end(), 0.0) - (Wrap ? 0.0 : segment_lengths.back());
		double length = total_length * k3d::clamp(Position, 0.0, 1.0);
		
		const unsigned long point_count = Curve.control_points.size();
		for(unsigned long i = 0; i != segment_lengths.size(); ++i)
			{
				if(length < segment_lengths[i])
					return k3d::translation3D(k3d::mix(Curve.control_points[i]->position, Curve.control_points[(i+1)%point_count]->position, length / segment_lengths[i]));
					
				length -= segment_lengths[i];
			}

		return k3d::translation3D(Wrap ? Curve.control_points.front()->position : Curve.control_points.back()->position);
	}

	k3d::matrix4 get_transformation(const k3d::linear_curve_group& CurveGroup, const double Position)
	{
		if(CurveGroup.curves.empty())
			return k3d::identity3D();
		
		return get_transformation(**CurveGroup.curves.begin(), CurveGroup.wrap, Position);
	}

	k3d::matrix4 get_transformation(const k3d::point_group& PointGroup, const double Position)
	{
		if(PointGroup.points.empty())
			return k3d::identity3D();

		const unsigned long point_index = static_cast<unsigned long>(k3d::clamp(Position, 0.0, 1.0) * (PointGroup.points.size() - 1));

		return k3d::translation3D(PointGroup.points[point_index]->position);
	}

	k3d::matrix4 output_value()
	{
		k3d::matrix4 result = m_input.property_value();
		
		k3d::mesh* const mesh = m_input_mesh.property_value();
		if(mesh)
			{
				const double start_time = m_start_time.property_value();
				const double time = m_time.property_value();
				const double end_time = m_end_time.property_value();
				const double delta_time = end_time - start_time;
				const double position = (0 != delta_time) ? (k3d::clamp(time, start_time, end_time) - start_time) / delta_time : 0;

				// Choose the geometry that we're going to use ...
				if(mesh->point_groups.size())
					result = result * get_transformation(**mesh->point_groups.begin(), position);
				else if(mesh->linear_curve_groups.size())
					result = result * get_transformation(**mesh->linear_curve_groups.begin(), position);
			}
		
		return result;
	}
			
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<make_path_implementation>,
			k3d::interface_list<k3d::itransform_source,
			k3d::interface_list<k3d::itransform_sink > > > factory(
			k3d::uuid(0xe28ff816, 0xe1334197, 0xab91b6cb, 0x1f0d0def),
			"MakePath",
			"Converts mesh geometry into an animated path",
			"Objects",
			k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

	k3d_data_property(k3d::matrix4, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_input;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_start_time;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_time;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_end_time;
	k3d::computed_property<k3d::matrix4, k3d::method_call_t<make_path_implementation, k3d::matrix4> > m_output;
};

/////////////////////////////////////////////////////////////////////////////
// make_path_factory

k3d::iplugin_factory& make_path_factory()
{
	return make_path_implementation::get_factory();
}

} // namespace libk3dmesh




