// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
	\author Romain Behar (romainbehar@yahoo.com)
	\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace libk3dmesh
{

namespace detail
{

typedef std::vector<unsigned long> CushionPath;
typedef std::vector<CushionPath> CushionPaths;

void AddCushionTriangle(const unsigned long v1, const unsigned long v2, const unsigned long v3, CushionPaths& Paths)
{
	CushionPath path;
	path.push_back(v1);
	path.push_back(v2);
	path.push_back(v3);

	Paths.push_back(path);
}

void AddCushionQuad(const unsigned long v1, const unsigned long v2, const unsigned long v3, const unsigned long v4, CushionPaths& Paths)
{
	CushionPath path;
	path.push_back(v1);
	path.push_back(v2);
	path.push_back(v3);
	path.push_back(v4);

	Paths.push_back(path);
}

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// poly_cushion_implementation

class poly_cushion_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	poly_cushion_implementation(k3d::idocument& Document) :
		base(Document),
		m_length_segments(k3d::init_name("length_segments") + k3d::init_description("") + k3d::init_value(5) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_document(Document) + k3d::init_precision(0) + k3d::init_step_increment(1.0) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_radial_segments(k3d::init_name("radial_segments") + k3d::init_description("") + k3d::init_value(5) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_document(Document) + k3d::init_precision(0) + k3d::init_step_increment(1.0) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_diameter(k3d::init_name("diameter") + k3d::init_description("") + k3d::init_value(4.0) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_width(k3d::init_name("width") + k3d::init_description("") + k3d::init_value(8.0) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_height(k3d::init_name("height") + k3d::init_description("") + k3d::init_value(8.0) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_depth(k3d::init_name("depth") + k3d::init_description("") + k3d::init_value(8.0) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance)))
	{
		enable_serialization(k3d::persistence::proxy(m_length_segments));
		enable_serialization(k3d::persistence::proxy(m_radial_segments));
		enable_serialization(k3d::persistence::proxy(m_diameter));
		enable_serialization(k3d::persistence::proxy(m_width));
		enable_serialization(k3d::persistence::proxy(m_height));
		enable_serialization(k3d::persistence::proxy(m_depth));

		register_property(m_length_segments);
		register_property(m_radial_segments);
		register_property(m_diameter);
		register_property(m_width);
		register_property(m_height);
		register_property(m_depth);

		m_material.changed_signal().connect(SigC::slot(*this, &poly_cushion_implementation::on_reset_geometry));
		
		m_length_segments.changed_signal().connect(SigC::slot(*this, &poly_cushion_implementation::on_reset_geometry));
		m_radial_segments.changed_signal().connect(SigC::slot(*this, &poly_cushion_implementation::on_reset_geometry));
		m_diameter.changed_signal().connect(SigC::slot(*this, &poly_cushion_implementation::on_reset_geometry));
		m_width.changed_signal().connect(SigC::slot(*this, &poly_cushion_implementation::on_reset_geometry));
		m_height.changed_signal().connect(SigC::slot(*this, &poly_cushion_implementation::on_reset_geometry));
		m_depth.changed_signal().connect(SigC::slot(*this, &poly_cushion_implementation::on_reset_geometry));
	
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &poly_cushion_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}
	
	k3d::mesh* on_create_geometry()
	{
		std::auto_ptr<k3d::mesh> mesh(new k3d::mesh());
		
		mesh->polyhedra.push_back(new k3d::polyhedron());
		k3d::polyhedron& polyhedron = *mesh->polyhedra.back();
		polyhedron.material = m_material.interface();

		const unsigned long length_segments = m_length_segments.property_value();
		const unsigned long radial_segments = m_radial_segments.property_value();
		const double diameter = m_diameter.property_value();
		const double width = m_width.value();
		const double height = m_height.value();
		const double depth = m_depth.value();

		// Calculate points on the surface of a sphere ...
		typedef std::vector<k3d::vector3> SpherePoints;
		SpherePoints spherepoints;

		const unsigned long N = length_segments*4;
		const unsigned long R = radial_segments*2-1;

		unsigned long i, j, n, c, d, e, f;

		for(j = 0; j < N; j++)
			for(i = 0; i < R; i++)
				{
					//double  l = diameter/2 - (double)(i+1)*diameter/((double)R+1);
					double l = sin(M_PI/2 -(double)(i+1)*M_PI / (double)(R+1))*diameter/2;
					double r = sqrt(diameter*diameter/4 - l*l);

					spherepoints.push_back(k3d::vector3( r*sin((double)j*2*M_PI / (double)N), l, r*cos((double)j*2*M_PI / (double)N)));
				}
		spherepoints.push_back(k3d::vector3(0, diameter/2, 0));
		spherepoints.push_back(k3d::vector3(0, -diameter/2, 0));

		// Cut sphere into 8 parts using distances ...

		// Calculate individual cushion points ...
		typedef std::vector<k3d::vector3> CushionPoints;
		CushionPoints cushionpoints;

		c = (R+1)/2;

		k3d::vector3 tP = k3d::vector3(width / 2.0, height / 2.0, depth / 2.0);
		for(j = 0; j < N / 4+1; j++)
			for(i = 0; i < c; i++)
				cushionpoints.push_back(spherepoints[R == 1 ? j : j*R+i] + tP);

		cushionpoints.push_back(spherepoints[N*R] + tP);

		tP = k3d::vector3(width / 2.0, -height / 2.0, depth / 2.0);
		for(j = 0; j < N / 4+1; j++)
			for(i = 0; i < c; i++)
				cushionpoints.push_back(spherepoints[R / 2 +(R == 1 ? j : j*R+i)] + tP);

		cushionpoints.push_back(spherepoints[N*R+1] + tP);

		tP = k3d::vector3(width / 2.0, height / 2.0, -depth / 2.0);
		for(j = N / 4; j < N / 2+1; j++)
			for(i = 0; i < c; i++)
				cushionpoints.push_back(spherepoints[R == 1 ? j : j*R+i] + tP);

		cushionpoints.push_back(spherepoints[N*R] + tP);

		tP = k3d::vector3(width / 2.0, -height / 2.0, -depth / 2.0);
		for(j = N / 4; j < N / 2+1; j++)
			for(i = 0; i < c; i++)
				cushionpoints.push_back(spherepoints[R / 2 +(R == 1 ? j : j*R+i)] + tP);

		cushionpoints.push_back(spherepoints[N*R+1] + tP);

		tP = k3d::vector3(-width / 2.0, height / 2.0, -depth / 2.0);
		for(j = N / 2; j < 3*N / 4+1; j++)
			for(i = 0; i < c; i++)
				cushionpoints.push_back(spherepoints[R == 1 ? j : j*R+i] + tP);

		cushionpoints.push_back(spherepoints[N*R] + tP);

		tP = k3d::vector3(-width / 2.0, -height / 2.0, -depth / 2.0);
		for(j = N / 2; j < 3*N / 4+1; j++)
			for(i = 0; i < c; i++)
				cushionpoints.push_back(spherepoints[R / 2 +(R == 1 ? j : j*R+i)] + tP);

		cushionpoints.push_back(spherepoints[N*R+1] + tP);

		tP = k3d::vector3(-width / 2.0, height / 2.0, depth / 2.0);
		for(j = 3*N / 4; j <= N; j++)
			for(i = 0; i < c; i++)
				cushionpoints.push_back(spherepoints[R == 1 ? j % N :(j % N)*R+i] + tP);

		cushionpoints.push_back(spherepoints[N*R] + tP);

		tP = k3d::vector3(-width / 2.0, -height / 2.0, depth / 2.0);
		for(j = 3*N / 4; j <= N; j++)
			for(i = 0; i < c; i++)
				cushionpoints.push_back(spherepoints[R / 2 +(R == 1 ? j % N :(j % N)*R+i)] + tP);

		cushionpoints.push_back(spherepoints[N*R+1] + tP);

		// Calculate individual cushion paths
		detail::CushionPaths cushionpaths;

		// Make sphere parts faces ...
		d = c*(N/4+1)+1;

		for(n = 0; n < 8; n++)
			{
				for(j = 0; j < N / 4; j++)
					for(i = 0; i < c-1; i++)
						detail::AddCushionQuad(n*d + c*j + i, n*d + c*j + i+1, n*d + c*(j+1) + i+1, n*d + c*(j+1) + i, cushionpaths);

				for(j = 0; j < N / 4; j ++)
					if(n % 2 == 0)
						detail::AddCushionTriangle(n*d + c*j, n*d + c*(j+1), (n+1) * d - 1, cushionpaths);
					else
						detail::AddCushionTriangle(n*d + c*(j+1) - 1, (n+1) * d - 1, n*d + c*(j+2) - 1, cushionpaths);
			}

		// Link parts ...
		for(n = 0; n < 8; n++)
			{
				e = (n+1)*d - c-1;
				f = ((n+2) % 8)*d;

				for(j = 0; j < c-1; j++)
					detail::AddCushionQuad(e+j, e+j+1, f+j+1, f+j, cushionpaths);
			}

		detail::AddCushionQuad(0, d - 1, 7*d - 1, 7*d - c-1, cushionpaths);
		detail::AddCushionQuad(d - c-1, 2*d, 3*d - 1, d - 1, cushionpaths);
		detail::AddCushionQuad(3*d - c-1, 4*d, 5*d - 1, 3*d - 1, cushionpaths);
		detail::AddCushionQuad(5*d - c-1, 6*d, 7*d - 1, 5*d - 1, cushionpaths);

		detail::AddCushionQuad(d + c-1, 8*d - 2, 8*d - 1, 2*d - 1, cushionpaths);
		detail::AddCushionQuad(2*d - 2, 2*d - 1, 4*d - 1, 3*d + c-1, cushionpaths);
		detail::AddCushionQuad(4*d - 2, 4*d - 1, 6*d - 1, 5*d + c-1, cushionpaths);
		detail::AddCushionQuad(6*d - 2, 6*d - 1, 8*d - 1, 7*d + c-1, cushionpaths);


		for(n = 0; n < 4; n++)
			{
				e = d*(n*2) + c-1;
				f = d*(n*2+1);

				for(j = 0; j < N / 4; j++)
					detail::AddCushionQuad(e+j*c, f+j*c, f+(j+1)*c, e+(j+1)*c, cushionpaths);
			}

		detail::AddCushionQuad(c-1, d*7 - 2, d*8 - c-1, d, cushionpaths);
		detail::AddCushionQuad(d-2, d*2 - c-1, d*3, d*2 + c-1, cushionpaths);
		detail::AddCushionQuad(d*3 - 2, d*4 - c-1, d*5, d*4 + c-1, cushionpaths);
		detail::AddCushionQuad(d*5 - 2, d*6 - c-1, d*7, d*6 + c-1, cushionpaths);

		detail::AddCushionQuad(d-1, d*3 - 1, d*5 - 1, d*7 - 1, cushionpaths);
		detail::AddCushionQuad(d*2 - 1, d*8 - 1, d*6 - 1, d*4 - 1, cushionpaths);

		// Add geometry to the mesh ...
		for(unsigned long i = 0; i != cushionpoints.size(); ++i)
			mesh->points.push_back(new k3d::point(cushionpoints[i]));
			
		for(unsigned long i = 0; i != cushionpaths.size(); ++i)
			{
				k3d::polyhedron::edges_t edges;
				for(unsigned long j = 0; j != cushionpaths[i].size(); ++j)
					edges.push_back(new k3d::split_edge(mesh->points[cushionpaths[i][j]]));
				for(unsigned long j = 0; j != cushionpaths[i].size(); ++j)
					edges[j]->face_clockwise = edges[(j+1)%cushionpaths[i].size()];
					
				polyhedron.edges.insert(polyhedron.edges.end(), edges.begin(), edges.end());
				polyhedron.faces.push_back(new k3d::face(edges.front()));
			}

		return_val_if_fail(is_valid(polyhedron), 0);
//		assert_warning(is_solid(polyhedron));
																																				
		return mesh.release();
	}
	
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<poly_cushion_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
			k3d::uuid(0xc11b963d, 0x108d471c, 0xa3826195, 0x821116b0),
			"PolyCushion",
			"Generates a polygonal cushion (a cube with rounded edges)",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_length_segments;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_radial_segments;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_diameter;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_width;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_height;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_depth;
};

/////////////////////////////////////////////////////////////////////////////
// poly_cushion_factory

k3d::iplugin_factory& poly_cushion_factory()
{
	return poly_cushion_implementation::get_factory();
}

} // namespace libk3dmesh

