// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/imaterial.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace libk3dnurbs
{

/////////////////////////////////////////////////////////////////////////////
// nurbs_grid_implementation

class nurbs_grid_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	nurbs_grid_implementation(k3d::idocument& Document) :
		base(Document),
		m_u_order(k3d::init_name("u_order") + k3d::init_description("U Order [integer]") + k3d::init_value(4) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(2UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_v_order(k3d::init_name("v_order") + k3d::init_description("V Order [integer]") + k3d::init_value(4) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(2UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_u_segments(k3d::init_name("u_segments") + k3d::init_description("Columns [integer]") + k3d::init_value(1) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_v_segments(k3d::init_name("v_segments") + k3d::init_description("Rows [integer]") + k3d::init_value(1) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_width(k3d::init_name("width") + k3d::init_description("Width [number]") + k3d::init_document(Document) + k3d::init_value(10.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_height(k3d::init_name("height") + k3d::init_description("Height [number]") + k3d::init_document(Document) + k3d::init_value(10.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance)))
	{
		enable_serialization(k3d::persistence::proxy(m_u_order));
		enable_serialization(k3d::persistence::proxy(m_v_order));
		enable_serialization(k3d::persistence::proxy(m_u_segments));
		enable_serialization(k3d::persistence::proxy(m_v_segments));
		enable_serialization(k3d::persistence::proxy(m_width));
		enable_serialization(k3d::persistence::proxy(m_height));

		register_property(m_u_order);
		register_property(m_v_order);
		register_property(m_u_segments);
		register_property(m_v_segments);
		register_property(m_width);
		register_property(m_height);


		m_material.changed_signal().connect(SigC::slot(*this, &nurbs_grid_implementation::on_reset_geometry));

		m_u_order.changed_signal().connect(SigC::slot(*this, &nurbs_grid_implementation::on_reset_geometry));
		m_v_order.changed_signal().connect(SigC::slot(*this, &nurbs_grid_implementation::on_reset_geometry));
		m_u_segments.changed_signal().connect(SigC::slot(*this, &nurbs_grid_implementation::on_reset_geometry));
		m_v_segments.changed_signal().connect(SigC::slot(*this, &nurbs_grid_implementation::on_reset_geometry));
		m_width.changed_signal().connect(SigC::slot(*this, &nurbs_grid_implementation::on_reset_geometry));
		m_height.changed_signal().connect(SigC::slot(*this, &nurbs_grid_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &nurbs_grid_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		k3d::mesh* const mesh = new k3d::mesh();

		k3d::nupatch* const patch = new k3d::nupatch();
		mesh->nupatches.push_back(patch);

		patch->material = m_material.interface();

		const unsigned long u_order = m_u_order.property_value();
		const unsigned long v_order = m_v_order.property_value();
		const unsigned long u_segments = m_u_segments.property_value();
		const unsigned long v_segments = m_v_segments.property_value();
		const double width = m_width.property_value();
		const double height = m_height.property_value();

		const unsigned long u_degree = u_order - 1;
		const unsigned long v_degree = v_order - 1;
		const unsigned long u_points = (u_segments * u_degree) + 1;
		const unsigned long v_points = (v_segments * v_degree) + 1;

		patch->u_order = u_order;
		patch->v_order = v_order;

		patch->u_knots.insert(patch->u_knots.end(), u_order, 0);
		for(unsigned long i = 1; i != u_segments; ++i)
			patch->u_knots.insert(patch->u_knots.end(), u_order-1, i);
		patch->u_knots.insert(patch->u_knots.end(), u_order, u_segments);

		patch->v_knots.insert(patch->v_knots.end(), v_order, 0);
		for(unsigned long i = 1; i != v_segments; ++i)
			patch->v_knots.insert(patch->v_knots.end(), v_order-1, i);
		patch->v_knots.insert(patch->v_knots.end(), v_order, v_segments);

		for(unsigned long row = 0; row != v_points; ++row)
			{
				const double row_percent = 0.5 - (static_cast<double>(row) / static_cast<double>(v_points-1));

				for(unsigned long column = 0; column != u_points; ++column)
					{
						const double column_percent = (static_cast<double>(column) / static_cast<double>(u_points-1)) - 0.5;

						k3d::point* point = new k3d::point(width * column_percent, height * row_percent, 0);
						mesh->points.push_back(point);
						patch->control_points.push_back(k3d::nupatch::control_point(point, 1));
					}
			}

		// Do some sanity checking just for the heck of it ...
		assert_warning(k3d::is_valid(*patch));

		return mesh;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<nurbs_grid_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
			k3d::uuid(0x5aac4e72, 0xf9b04b61, 0xf8b1bdbc, 0x851cf62e),
			"NurbsGrid",
			"Generates a NURBS grid",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_u_order;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_v_order;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_u_segments;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_v_segments;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_width;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_height;
};

/////////////////////////////////////////////////////////////////////////////
// nurbs_grid_factory

k3d::iplugin_factory& nurbs_grid_factory()
{
	return nurbs_grid_implementation::get_factory();
}

} // namespace libk3dnurbs


