// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the RenderManBackgroundSphere K-3D object, which renders a world-aligned sphere, centered at the camera's position, with radius equal to the camera farplane
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/imaterial.h>
#include <k3dsdk/iprojection.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/module.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/property.h>
#include <k3dsdk/renderman.h>
#include <k3dsdk/transform.h>
#include <k3dsdk/vectors.h>

#ifdef	WIN32
#ifdef	near
#undef	near
#endif	//near
#ifdef	far
#undef	far
#endif	//far
#endif	//WIN32

namespace
{

/////////////////////////////////////////////////////////////////////////////
// background_sphere_implementation

class background_sphere_implementation :
	public k3d::material_collection<k3d::persistent<k3d::object> >,
	public k3d::ri::irenderable
{
	typedef k3d::material_collection<k3d::persistent<k3d::object> > base;

public:
	background_sphere_implementation(k3d::idocument& Document) :
		base(Document),
		m_distance(k3d::init_name("distance") + k3d::init_description("Distance [scalar]") + k3d::init_value(0.99) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(0.0, k3d::constraint::maximum(1.0)))),
		m_render_final(k3d::init_name("render_final") + k3d::init_description("Visible in the final rendered image [boolean]") + k3d::init_value(true) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_distance));
		enable_serialization(k3d::persistence::proxy(m_render_final));
		
		register_property(m_distance);
		register_property(m_render_final);
	}

	void renderman_pre_render(const k3d::ri::render_state& State)
	{
	}

	void renderman_render(const k3d::ri::render_state& State)
	{
		if(!m_render_final.property_value())
			return;
			
		// We never generate shadows ...
		if(k3d::ri::SHADOW_MAP == State.render_context)
			return;
		// We only generate output for the last sample ...
		if(!k3d::ri::last_sample(State))
			return;

		k3d::iperspective* const perspective = dynamic_cast<k3d::iperspective*>(&State.projection);
		k3d::iorthographic* const orthographic = dynamic_cast<k3d::iorthographic*>(&State.projection);
		if(!perspective && !orthographic)
			{
				std::cerr << error << k3d_file_reference() << " unknown projection type" << std::endl;
				return;
			}
	
		double radius = 0;
		if(perspective)
			{
				const double near = boost::any_cast<double>(k3d::get_property_value(document().dag(), perspective->near()));
				const double far = boost::any_cast<double>(k3d::get_property_value(document().dag(), perspective->far()));
				radius = k3d::mix(near, far, m_distance.property_value());
			}

		if(orthographic)
			{
				const double near = boost::any_cast<double>(k3d::get_property_value(document().dag(), orthographic->near()));
				const double far = boost::any_cast<double>(k3d::get_property_value(document().dag(), orthographic->far()));
				radius = k3d::mix(near, far, m_distance.property_value());
			}

		// Get data from the camera ...
		const k3d::vector3 camera_coords = State.camera_matrix * k3d::vector3(0, 0, 0);

		State.engine.RiAttributeBegin();
		State.engine.RiAttributeV("identifier", k3d::ri::parameter_list(1, k3d::ri::parameter("name", k3d::ri::UNIFORM, name())));
		State.engine.RiIdentity();
		State.engine.RiTranslate(camera_coords[0], camera_coords[1], camera_coords[2]);
		State.engine.RiRotate(90.0, 1.0, 0.0, 0.0);

		k3d::ri::setup_material(m_material.object(), State);
		
		State.engine.RiSphereV(radius, -radius, radius, 360.0);
		State.engine.RiAttributeEnd();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<background_sphere_implementation> >factory(
			k3d::uuid(0x00000001, 0x00000000, 0x00000000, 0x00000019),
			"RenderManBackgroundSphere",
			"",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_distance;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_render_final;
};

} // namespace

namespace libk3drenderman
{

k3d::iplugin_factory& background_sphere_factory()
{
	return background_sphere_implementation::get_factory();
}

} // namespace libk3drenderman



