// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the CSG Operator K-3D object, which allows geometry to be rendered with boolean effects
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/module.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/renderman.h>

#include <iostream>

namespace libk3drenderman
{

/////////////////////////////////////////////////////////////////////////////
// csg_operator_implementation

/// Allows geometry to be rendered with boolean effects
class csg_operator_implementation :
	public k3d::persistent<k3d::object> ,
	public k3d::ri::irenderable
{
	typedef k3d::persistent<k3d::object>  base;

public:
	csg_operator_implementation(k3d::idocument& Document) :
		base(Document),
		m_type(k3d::init_name("type") + k3d::init_description("Operation [enumeration]") + k3d::init_enumeration(boolean_values()) + k3d::init_value(BOOLEAN_INTERSECTION) + k3d::init_document(Document)),
		m_instance1(k3d::init_name("instance1") + k3d::init_description("Instance1 [object]") + k3d::init_object_value(0) + k3d::init_document(Document)),
		m_instance2(k3d::init_name("instance2") + k3d::init_description("Instance2 [object]") + k3d::init_object_value(0) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_type));
		enable_serialization(k3d::persistence::object_proxy(m_instance1));
		enable_serialization(k3d::persistence::object_proxy(m_instance2));
		
		register_property(m_type);
		register_property(m_instance1);
		register_property(m_instance2);
	}

	void renderman_pre_render(const k3d::ri::render_state&)
	{
	}

	void renderman_render(const k3d::ri::render_state& State)
	{
		// We only generate RIB for the last sample ...
		if(!k3d::ri::last_sample(State))
			return;

		k3d::ri::irenderable* const renderable1 = m_instance1.interface();
		k3d::ri::irenderable* const renderable2 = m_instance2.interface();
		if(!renderable1 || !renderable2)
			return;

		// Make sure we don't enter an infinite loop trying to render ourself ...
		if(renderable1 == this || renderable2 == this)
			{
				std::cerr << error << factory().name() << " [" << name() << "] cannot instance itself" << std::endl;
				return;
			}

		// Record the type of boolean ...
		k3d::ri::render_state state(State);
		state.render_context = k3d::ri::CSG_SOLID;

		switch(m_type.property_value())
			{
				case BOOLEAN_INTERSECTION:
					State.engine.RiSolidBegin(k3d::ri::RI_INTERSECTION());
					renderable1->renderman_render(state);
					renderable2->renderman_render(state);
					break;

				case BOOLEAN_UNION:
					State.engine.RiSolidBegin(k3d::ri::RI_UNION());
					renderable1->renderman_render(state);
					renderable2->renderman_render(state);
					break;

				case BOOLEAN_DIFFERENCE:
					State.engine.RiSolidBegin(k3d::ri::RI_DIFFERENCE());
					renderable1->renderman_render(state);
					renderable2->renderman_render(state);
					break;

				case BOOLEAN_REVERSE_DIFFERENCE:
					State.engine.RiSolidBegin(k3d::ri::RI_DIFFERENCE());
					renderable2->renderman_render(state);
					renderable1->renderman_render(state);
					break;
			}

		State.engine.RiSolidEnd();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<csg_operator_implementation> >factory(
			k3d::uuid(0x00000001, 0x00000000, 0x00000000, 0x0000000a),
			"RenderManCSGOperator",
			"",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	typedef enum
	{
		BOOLEAN_INTERSECTION,
		BOOLEAN_UNION,
		BOOLEAN_DIFFERENCE,
		BOOLEAN_REVERSE_DIFFERENCE
	} boolean_t;

	friend const k3d::ienumeration_property::values_t& boolean_values()
	{
		static k3d::ienumeration_property::values_t values;
		if(values.empty())
			{
				values.push_back(k3d::ienumeration_property::value_t("Intersection", "intersection", "Render intersecting volumes"));
				values.push_back(k3d::ienumeration_property::value_t("Union", "union", "Render the union of two volumnes"));
				values.push_back(k3d::ienumeration_property::value_t("Difference", "difference", "Render the difference of two volumes"));
				values.push_back(k3d::ienumeration_property::value_t("Reverse Difference", "reverse_difference", "Render the difference of two volumes"));
			}
		
		return values;
	}
	
	friend std::ostream& operator<<(std::ostream& Stream, const boolean_t& Value)
	{
		switch(Value)
			{
				case BOOLEAN_UNION:
					Stream << "union";
					break;
				case BOOLEAN_INTERSECTION:
					Stream << "intersection";
					break;
				case BOOLEAN_DIFFERENCE:
					Stream << "difference";
					break;
				case BOOLEAN_REVERSE_DIFFERENCE:
					Stream << "reverse_difference";
					break;
			}

		return Stream;
	}

	friend std::istream& operator>>(std::istream& Stream, boolean_t& Value)
	{
		std::string text;
		Stream >> text;

		if(text == "union")
			Value = BOOLEAN_UNION;
		else if(text == "intersection")
			Value = BOOLEAN_INTERSECTION;
		else if(text == "difference")
			Value = BOOLEAN_DIFFERENCE;
		else if(text == "reverse_difference")
			Value = BOOLEAN_REVERSE_DIFFERENCE;
		else
			std::cerr << __PRETTY_FUNCTION__ << ": unknown enumeration [" << text << "]"<< std::endl;

		return Stream;
	}

	k3d_enumeration_property(boolean_t, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_type;
	k3d_object_property(k3d::ri::irenderable, k3d::immutable_name, k3d::with_undo, k3d::local_storage) m_instance1;
	k3d_object_property(k3d::ri::irenderable, k3d::immutable_name, k3d::with_undo, k3d::local_storage) m_instance2;
};

k3d::iplugin_factory& csg_operator_factory()
{
	return csg_operator_implementation::get_factory();
}

} // namespace libk3drenderman



