// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead <tshead@k-3d.com>
*/

#include "imaterial.h"

#include <k3dsdk/color.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/module.h>
#include <k3dsdk/vectors.h>

#include <sdpgl/sdpgl.h>

namespace libk3dyafray
{

/////////////////////////////////////////////////////////////////////////////
// material

class material :
	public k3d::persistent<k3d::object> ,
	public libk3dyafray::imaterial,
	public k3d::imaterial
{
	typedef k3d::persistent<k3d::object>  base;

public:
	material(k3d::idocument& Document) :
		base(Document),
		m_color(k3d::init_name("color") + k3d::init_description("Color") + k3d::init_value(k3d::color(1, 1, 1)) + k3d::init_document(Document)),
		m_specular_color(k3d::init_name("specular_color") + k3d::init_description("Specular Color") + k3d::init_value(k3d::color(1, 1, 1)) + k3d::init_document(Document)),
		m_reflected_color(k3d::init_name("reflected_color") + k3d::init_description("Reflected Color") + k3d::init_value(k3d::color(0, 0, 0)) + k3d::init_document(Document)),
		m_transmitted_color(k3d::init_name("transmitted_color") + k3d::init_description("Transmitted Color") + k3d::init_value(k3d::color(0, 0, 0)) + k3d::init_document(Document)),
		m_hardness(k3d::init_name("hardness") + k3d::init_description("Hardness") + k3d::init_value(10.0) + k3d::init_document(Document)),
		m_index_of_refraction(k3d::init_name("index_of_refraction") + k3d::init_description("Index of Refraction") + k3d::init_value(1.0) + k3d::init_document(Document)),
		m_minimum_reflection(k3d::init_name("minimum_reflection") + k3d::init_description("Minimum Reflection") + k3d::init_value(0.0) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_color));
		enable_serialization(k3d::persistence::proxy(m_specular_color));
		enable_serialization(k3d::persistence::proxy(m_reflected_color));
		enable_serialization(k3d::persistence::proxy(m_transmitted_color));
		enable_serialization(k3d::persistence::proxy(m_hardness));
		enable_serialization(k3d::persistence::proxy(m_index_of_refraction));
		enable_serialization(k3d::persistence::proxy(m_minimum_reflection));

		register_property(m_color);
		register_property(m_specular_color);
		register_property(m_reflected_color);
		register_property(m_transmitted_color);
		register_property(m_hardness);
		register_property(m_index_of_refraction);
		register_property(m_minimum_reflection);
	}

	void setup_material(std::ostream& Stream)
	{
		const k3d::color color = m_color.property_value();
		const k3d::color specular_color = m_specular_color.property_value();
		const k3d::color reflected_color = m_reflected_color.property_value();
		const k3d::color transmitted_color = m_transmitted_color.property_value();

		const double hardness = m_hardness.property_value();
		const double index_of_refraction = m_index_of_refraction.property_value();
		const double minimum_reflection = m_minimum_reflection.property_value();

		Stream << "<shader type=\"generic\" name=\"" << name() << "\">" << std::endl;
		Stream << "	<attributes>" << std::endl;
		Stream << "		<color r=\"" << color.red << "\" g=\"" << color.green << "\" b=\"" << color.blue << "\"/>" << std::endl;
		Stream << "		<specular r=\"" << specular_color.red << "\" g=\"" << specular_color.green << "\" b=\"" << specular_color.blue << "\"/>" << std::endl;
		Stream << "		<reflected r=\"" << reflected_color.red << "\" g=\"" << reflected_color.green << "\" b=\"" << reflected_color.blue << "\"/>" << std::endl;
		Stream << "		<transmitted r=\"" << transmitted_color.red << "\" g=\"" << transmitted_color.green << "\" b=\"" << transmitted_color.blue << "\"/>" << std::endl;
		Stream << "		<hard value=\"" << hardness << "\"/>" << std::endl;
		Stream << "		<IOR value=\"" << index_of_refraction << "\"/>" << std::endl;
		Stream << "		<min_refle value=\"" << minimum_reflection << "\"/>" << std::endl;
		Stream << "	</attributes>" << std::endl;
		Stream << "</shader>" << std::endl;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<material> > factory(
			k3d::uuid(0x4b767ac5, 0x19ec4182, 0x9883cc81, 0x3f091dea),
			"YafRayMaterial",
			"YafRay Material",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data_property(k3d::color, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_color;
	k3d_data_property(k3d::color, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_specular_color;
	k3d_data_property(k3d::color, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_reflected_color;
	k3d_data_property(k3d::color, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_transmitted_color;

	k3d_data_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_hardness;
	k3d_data_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_index_of_refraction;
	k3d_data_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_minimum_reflection;
};

k3d::iplugin_factory& material_factory()
{
	return material::get_factory();
}

} // namespace libk3dyafray



