// K-3D
// Copyright (c) 2004-2006, Romain Behar
//
// Contact: romainbehar@yahoo.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/algebra.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/color.h>
#include <k3dsdk/frames.h>
#include <k3dsdk/fstream.h>
#include <k3dsdk/gl.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/icamera.h>
#include <k3dsdk/icamera_animation_render_engine.h>
#include <k3dsdk/icamera_preview_render_engine.h>
#include <k3dsdk/icamera_still_render_engine.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/imaterial_client.h>
#include <k3dsdk/imesh_source.h>
#include <k3dsdk/iprojection.h>
#include <k3dsdk/irender_farm.h>
#include <k3dsdk/itransform_source.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/property.h>
#include <k3dsdk/render_farm.h>
#include <k3dsdk/resolutions.h>
#include <k3dsdk/time_source.h>
#include <k3dsdk/transform.h>

#include <boost/filesystem/path.hpp>

#include <iomanip>
#include <iterator>

namespace k3d
{

inline bool operator < (const k3d::color& a, const k3d::color& b)
{
	if(a.red < b.red)
		return true;
	if(a.red > b.red)
		return false;
	if(a.green < b.green)
		return true;
	if(a.green > b.green)
		return false;
	if(a.blue < b.blue)
		return true;

	return false;
}

} // namespace k3d

namespace libk3dpov
{

void pov_point(const k3d::point3& point, std::ostream& stream)
{
	stream << "<" << point[0] << ", " << point[1] << ", " << point[2] << ">";
}

/////////////////////////////////////////////////////////////////////////////
// render_engine

class render_engine :
	public k3d::persistent<k3d::node>,
	public k3d::icamera_preview_render_engine,
	public k3d::icamera_still_render_engine,
	public k3d::icamera_animation_render_engine
{
	typedef k3d::persistent<k3d::node> base;

public:
	render_engine(k3d::idocument& Document) :
		base(Document),
		m_resolution(init_owner(*this) + init_name("resolution") + init_label(_("Resolution")) + init_description(_("Choose a predefined image resolution")) + init_enumeration(k3d::resolution_values()) + init_value(std::string(""))),
		m_pixel_width(init_owner(*this) + init_name("pixel_width") + init_label(_("Pixel width")) + init_description(_("Output pixel width")) + init_value(320) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar)) + init_constraint(constraint::minimum(1L))),
		m_pixel_height(init_owner(*this) + init_name("pixel_height") + init_label(_("Pixel height")) + init_description(_("Output pixel height")) + init_value(240) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar)) + init_constraint(constraint::minimum(1L)))
	{
		m_resolution.changed_signal().connect(sigc::mem_fun(*this, &render_engine::on_resolution_changed));
	}

	void on_resolution_changed(k3d::iunknown*)
	{
		const std::string new_resolution = m_resolution.value();

		const k3d::resolutions_t& resolutions = k3d::resolutions();
		for(k3d::resolutions_t::const_iterator resolution = resolutions.begin(); resolution != resolutions.end(); ++resolution)
		{
			if(resolution->name != new_resolution)
				continue;

			m_pixel_width.set_value(resolution->width);
			m_pixel_height.set_value(resolution->height);
			return;
		}

		assert_not_reached();
	}

	bool render_camera_preview(k3d::icamera& Camera)
	{
		// Start a new render job ...
		k3d::irender_job& job = k3d::render_farm().create_job("k3d-preview");

		// Add a single render frame to the job ...
		k3d::irender_frame& frame = job.create_frame("frame");

		// Create an output image path ...
		const boost::filesystem::path outputimagepath = frame.add_output_file("world.png");
		return_val_if_fail(!outputimagepath.empty(), false);

		// View the output image when it's done ...
		frame.add_view_operation(outputimagepath);

		// Render it (visible rendering) ...
		return_val_if_fail(render(Camera, frame, outputimagepath, true), false);

		// Start the job running ...
		k3d::render_farm().start_job(job);

		return true;
	}

	bool render_camera_frame(k3d::icamera& Camera, const boost::filesystem::path& OutputImage, const bool ViewImage)
	{
		// Sanity checks ...
		return_val_if_fail(!OutputImage.empty(), false);

		// Start a new render job ...
		k3d::irender_job& job = k3d::render_farm().create_job("k3d-render-frame");

		// Add a single render frame to the job ...
		k3d::irender_frame& frame = job.create_frame("frame");

		// Create an output image path ...
		const boost::filesystem::path outputimagepath = frame.add_output_file("world.png");
		return_val_if_fail(!outputimagepath.empty(), false);

		// Copy the output image to its requested destination ...
		frame.add_copy_operation(outputimagepath, OutputImage);

		// View the output image when it's done ...
		if(ViewImage)
			frame.add_view_operation(OutputImage);

		// Render it (hidden rendering) ...
		return_val_if_fail(render(Camera, frame, outputimagepath, false), false);

		// Start the job running ...
		k3d::render_farm().start_job(job);

		return true;
	}

	bool render_camera_animation(k3d::icamera& Camera, const boost::filesystem::path& OutputImages, const bool ViewCompletedImages)
	{
		// Sanity checks ...
		return_val_if_fail(!OutputImages.empty(), false);

		// Ensure that the document has animation capabilities, first ...
		k3d::iproperty* const start_time_property = k3d::get_start_time(document());
		k3d::iproperty* const end_time_property = k3d::get_end_time(document());
		k3d::iproperty* const frame_rate_property = k3d::get_frame_rate(document());
		k3d::iwritable_property* const time_property = dynamic_cast<k3d::iwritable_property*>(k3d::get_time(document()));
		return_val_if_fail(start_time_property && end_time_property && frame_rate_property && time_property, false);

		// Test the output images filepath to make sure it can hold all the frames we're going to generate ...
		const double start_time = boost::any_cast<double>(k3d::get_value(document().dag(), *start_time_property));
		const double end_time = boost::any_cast<double>(k3d::get_value(document().dag(), *end_time_property));
		const double frame_rate = boost::any_cast<double>(k3d::get_value(document().dag(), *frame_rate_property));

		const long start_frame = static_cast<long>(k3d::round(frame_rate * start_time));
		const long end_frame = static_cast<long>(k3d::round(frame_rate * end_time));

		k3d::frames frames(OutputImages, start_frame, end_frame);
		return_val_if_fail(frames.max_frame() >= end_frame, false);

		// Start a new render job ...
		k3d::irender_job& job = k3d::render_farm().create_job("k3d-render-animation");

		// For each frame to be rendered ...
		for(long view_frame = start_frame; view_frame < end_frame; ++view_frame)
		{
			// Set the frame time ...
			time_property->property_set_value(view_frame / frame_rate);

			// Redraw everything ...
			k3d::gl::redraw_all(document(), k3d::gl::irender_engine::SYNCHRONOUS);

			// Add a render frame to the job ...
			std::stringstream buffer;
			buffer << "frame-" << std::setw(frames.frame_digits()) << std::setfill('0') << view_frame;
			k3d::irender_frame& frame = job.create_frame(buffer.str());

			// Create an output image path ...
			const boost::filesystem::path outputimagepath = frame.add_output_file("world.png");
			return_val_if_fail(!outputimagepath.empty(), false);

			// Copy the output image to its requested destination ...
			boost::filesystem::path destination;
			frames.frame(view_frame, destination);
			frame.add_copy_operation(outputimagepath, destination);

			// View the output image when it's done ...
			if(ViewCompletedImages)
				frame.add_view_operation(destination);

			// Render it (hidden rendering) ...
			return_val_if_fail(render(Camera, frame, outputimagepath, false), false);
		}

		// Start the job running ...
		k3d::render_farm().start_job(job);

		return true;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<render_engine>,
			k3d::interface_list<k3d::icamera_preview_render_engine,
			k3d::interface_list<k3d::icamera_still_render_engine,
			k3d::interface_list<k3d::icamera_animation_render_engine> > > > factory(
				k3d::uuid(0x7982ee23, 0x854a43e7, 0x874c8ad7, 0x5d0b95d2),
				"POVEngine",
				"POV Render Engine",
				"POV RenderEngines",
				k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
/*
	const std::string shader_name(k3d::iunknown& Object)
	{
		k3d::imaterial_collection* const material_collection = dynamic_cast<k3d::imaterial_collection*>(&Object);
		if(material_collection)
		{
			libk3dpov::imaterial* const material = dynamic_cast<libk3dpov::imaterial*>(material_collection->material());
			if(material)
			{
				return dynamic_cast<k3d::inode*>(material)->name();
			}
		}

		return "k3d_default_shader";
	}
*/

	bool render(k3d::icamera& Camera, k3d::irender_frame& Frame, const boost::filesystem::path& OutputImagePath, const bool VisibleRender)
	{
		// Sanity checks ...
		return_val_if_fail(!OutputImagePath.empty(), false);

		// Start our POV file ...
		const boost::filesystem::path filepath = Frame.add_input_file("world.pov");
		return_val_if_fail(!filepath.empty(), false);

		// Open the POV file stream ...
		k3d::filesystem::ofstream stream(filepath);
		return_val_if_fail(stream.good(), false);

		// Setup the frame for POV rendering ...
		Frame.add_render_operation("pov", "povray", filepath, VisibleRender);

		// Setup a POV scene description ...
		stream << "// Persistence Of Vision raytracer file" << std::endl;
		stream << "#include \"colors.inc\"" << std::endl;
		stream << "#include \"shapes.inc\"" << std::endl;
		stream << "#include \"textures.inc\"" << std::endl;
		stream << std::endl;

		// Get the document contents ...
		const k3d::nodes_t objects = document().nodes().collection();

		// Setup materials  ...
/*
		for(k3d::nodes_t::const_iterator object = objects.begin(); object != objects.end(); ++object)
		{
			libk3dpov::imaterial* const material = dynamic_cast<libk3dpov::imaterial*>(*object);
			if(material)
				material->setup_material(stream);
		}
*/

		// Setup lights ...
		bool found_light = false;
/*
		for(k3d::nodes_t::const_iterator object = objects.begin(); object != objects.end(); ++object)
		{
			libk3dpov::ilight* const light = dynamic_cast<libk3dpov::ilight*>(*object);
			if(light)
			{
				light->setup_light(stream);
			}
		}
*/

		// Setup geometry
		for(k3d::nodes_t::const_iterator object = objects.begin(); object != objects.end(); ++object)
		{
			// Render sphere objects ...
/*
			if((*object)->factory().class_id() == k3d::classes::Sphere())
			{
				const k3d::point3 sphere_center = k3d::node_to_world_matrix(**object) * k3d::point3(0, 0, 0);
				const boost::any sphere_radius(k3d::get_value(**object, "radius"));
				if(typeid(double) == sphere_radius.type())
				{
					stream << "<object name=\"" << (*object)->name() << "\" shader_name=\"" << shader_name(**object) << "\">" << std::endl;
					stream << "	<attributes>" << std::endl;
					stream << "	</attributes>" << std::endl;
					stream << "	<sphere radius=\"" << boost::any_cast<double>(sphere_radius) << "\">" << std::endl;
					stream << "		<center x=\"" << std::fixed << -sphere_center[0] << "\" y=\"" << std::fixed << sphere_center[1] << "\" z=\"" << std::fixed << sphere_center[2] << "\"/>" << std::endl;
					stream << "	</sphere>" << std::endl;
					stream << "</object>" << std::endl;
				}

				continue;
			}
*/

			// Render mesh objects ...
			if((*object)->factory().class_id() == k3d::classes::MeshInstance())
			{
				k3d::imesh_source* const mesh_source = dynamic_cast<k3d::imesh_source*>(*object);
				if(!mesh_source)
					continue;

				k3d::mesh* const mesh = boost::any_cast<k3d::mesh*>(k3d::get_value(document().dag(), mesh_source->mesh_source_output()));
				if(!mesh)
					continue;

				// Output bicubic patches
				for(k3d::mesh::bicubic_patches_t::const_iterator patch = mesh->bicubic_patches.begin(); patch != mesh->bicubic_patches.end(); patch++)
				{
					stream << "bicubic_patch { type 0 flatness 0.01  u_steps 4  v_steps 4" << std::endl;
					for(unsigned long n = 0; n < 16; ++n)
					{
						k3d::point* point = (*patch)->control_points[n];
						pov_point(point->position, stream);

						if(n < 15)
							stream << ",";

						if(n % 4 == 0)
							stream << std::endl;
					}

					stream << "texture {" << std::endl;
					stream << "pigment {Blue}" << std::endl;
					stream << "finish {ambient 0.1 diffuse 0.9 phong 1}" << std::endl;
					stream << "}" << std::endl;

					stream << "}" << std::endl;
				}

				// Output polyhedra as triangles
				k3d::polyhedron::faces_t new_faces;
				k3d::mesh::points_t new_points;

				for(k3d::mesh::polyhedra_t::const_iterator polyhedron = mesh->polyhedra.begin(); polyhedron != mesh->polyhedra.end(); ++polyhedron)
					k3d::triangulate((*polyhedron)->faces, new_faces, new_points);

				k3d::mesh::points_t all_points;
				all_points.insert(all_points.end(), mesh->points.begin(), mesh->points.end());
				all_points.insert(all_points.end(), new_points.begin(), new_points.end());

				stream << "mesh2 {" << std::endl;
				stream << " vertex_vectors {" << std::endl;
				stream << "  " << all_points.size() << "," << std::endl;

				std::map<k3d::point*, unsigned long> point_map;
				const k3d::matrix4 mesh_matrix = k3d::node_to_world_matrix(**object);
				for(k3d::mesh::points_t::const_iterator point = all_points.begin(); point != all_points.end(); ++point)
				{
					point_map.insert(std::make_pair(*point, point_map.size()));
					const k3d::point3 position = mesh_matrix * (*point)->position;
					stream << "  "; pov_point(position, stream); stream << "," << std::endl;
				}
				stream << " }" << std::endl;

				// Get face color list
				typedef std::map<k3d::color, unsigned long> color_map_t;
				color_map_t color_map;
				for(k3d::polyhedron::faces_t::const_iterator face = new_faces.begin(); face != new_faces.end(); ++face)
				{
					k3d::parameters_t::const_iterator color = (*face)->uniform_data.find("Cs");
					if(color != (*face)->uniform_data.end())
					{
						k3d::color face_color = boost::any_cast<k3d::color>(color->second);
						if(color_map.find(face_color) == color_map.end())
						{
							// Save color
							unsigned long index = color_map.size();
							color_map[face_color] = index;
						}
					}
				}

				// Set textures
				if(color_map.size())
				{
					stream << " texture_list {" << std::endl;
					stream << "  " << color_map.size() << "," << std::endl;
					for(unsigned long n = 0; n < color_map.size(); n++)
					{
						color_map_t::const_iterator color;
						for(color = color_map.begin(); color != color_map.end(); color++)
							if(color->second == n)
							{
								k3d::color c = color->first;
								stream << "  texture { pigment { rgb <" << c.red << ", " << c.green << ", " << c.blue << "> } }," << std::endl;
								break;
							}
					}
					stream << " }" << std::endl;
				}

				stream << " face_indices {" << std::endl;
				stream << "  " << new_faces.size() << "," << std::endl;

				for(k3d::polyhedron::faces_t::const_iterator face = new_faces.begin(); face != new_faces.end(); ++face)
				{
					k3d::split_edge* const e0 = (*face)->first_edge;
					k3d::split_edge* const e1 = e0 ? e0->face_clockwise : 0;
					k3d::split_edge* const e2 = e1 ? e1->face_clockwise : 0;

					stream << "  <" << point_map[e0->vertex] << ", " << point_map[e1->vertex] << ", " << point_map[e2->vertex] << ">," << std::endl;

					// Texture index
					k3d::parameters_t::const_iterator color = (*face)->uniform_data.find("Cs");
					if(color != (*face)->uniform_data.end())
					{
						k3d::color face_color = boost::any_cast<k3d::color>(color->second);
						unsigned long index = color_map[face_color];
						stream << " " << index << "," << std::endl;
					}
				}
				stream << " }" << std::endl;

				stream << "}" << std::endl;

				std::for_each(new_faces.begin(), new_faces.end(), k3d::delete_object());
				std::for_each(new_points.begin(), new_points.end(), k3d::delete_object());
			}
		}

		// Setup the camera ...
		const k3d::matrix4 transform_matrix = boost::any_cast<k3d::matrix4>(get_value(document().dag(), Camera.transformation().transform_source_output()));
		const k3d::euler_angles camera_angles = k3d::euler_angles(transform_matrix, k3d::euler_angles::XYZstatic);
		const k3d::point3 camera_position(k3d::position(transform_matrix));

		const double pixel_width = static_cast<double>(m_pixel_width.value());
		const double pixel_height = static_cast<double>(m_pixel_height.value());

		stream << "camera {" << std::endl;
		stream << " location <0, 0, 0>" << std::endl;
		stream << " direction <0, 0, 1>" << std::endl;
		stream << " up <0, 1, 0>" << std::endl;
		stream << " right <" << (pixel_width / pixel_height) << ", 0, 0>" << std::endl;
		stream << " rotate <" << k3d::degrees(camera_angles.n[0]) << ", " << k3d::degrees(camera_angles[1]) << ", " << k3d::degrees(camera_angles[2]) << ">" << std::endl;
		stream << " translate "; pov_point(camera_position, stream); stream << std::endl;
		stream << "}" << std::endl;

		// Finish the scene ...
		if(!found_light)
		{
			stream << "light_source { <25, 25, 25> colour White }" << std::endl;
			stream << "light_source { <-25, 25, 25> colour White }" << std::endl;
			stream << "light_source { <25, -25, 25> colour White }" << std::endl;
			stream << "light_source { <25, 25, -25> colour White }" << std::endl;
			stream << "light_source { <-25, -25, -25> colour White }" << std::endl;
		}

		return true;
	}

	k3d_data(std::string, immutable_name, change_signal, with_undo, local_storage, no_constraint, enumeration_property, with_serialization) m_resolution;
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, with_constraint, measurement_property, with_serialization) m_pixel_width;
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, with_constraint, measurement_property, with_serialization) m_pixel_height;
};

k3d::iplugin_factory& render_engine_factory()
{
	return render_engine::get_factory();
}

} // namespace libk3dpov

