// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the k3d::bitmap_preview namespace, which provides a MVC UI for boolean data sources
		\author Tim Shead (tshead@k-3d.com)
*/

#include "bitmap_preview.h"

#include <k3dsdk/property.h>

#include <gdkmm/pixbuf.h>
#include <gtkmm/image.h>

namespace libk3dngui
{

namespace bitmap_preview
{

/// Specialization of bitmap_preview::data_proxy for use with iproperty objects
class property_proxy :
	public idata_proxy
{
public:
	property_proxy(k3d::idag& Graph, k3d::iproperty& Data) :
		m_graph(Graph),
		m_readable_data(Data)
	{
	}

	k3d::bitmap* value()
	{
		// As a special-case, if the internal property value is NULL, return the connected property (if any)
		k3d::bitmap* result = boost::any_cast<k3d::bitmap*>(m_readable_data.property_value());
		if(!result)
			result = boost::any_cast<k3d::bitmap*>(k3d::get_value(m_graph, m_readable_data));

		return result;
	}

	changed_signal_t& changed_signal()
	{
		return m_readable_data.property_changed_signal();
	}

private:
	property_proxy(const property_proxy& RHS);
	property_proxy& operator=(const property_proxy& RHS);

	k3d::idag& m_graph;
	k3d::iproperty& m_readable_data;
};

std::auto_ptr<idata_proxy> proxy(k3d::idag& Graph, k3d::iproperty& Data)
{
	return std::auto_ptr<idata_proxy>(new property_proxy(Graph, Data));
}

/////////////////////////////////////////////////////////////////////////////
// control

control::control(k3d::icommand_node& Parent, const std::string& Name, std::auto_ptr<idata_proxy> Data) :
	base(Gtk::BUTTONBOX_START, 2),
	ui_component(Name, &Parent),
	m_image_buffer(64, 64),
	m_alpha_buffer(64, 64),
	m_image(new Gtk::Image()),
	m_alpha(new Gtk::Image()),
	m_data(Data)
{
	pack_start(*manage(m_image), Gtk::PACK_SHRINK);
	pack_start(*manage(m_alpha), Gtk::PACK_SHRINK);

	data_changed(0);

	if(m_data.get())
		m_data->changed_signal().connect(sigc::mem_fun(*this, &control::data_changed));
}

void control::data_changed(k3d::iunknown*)
{
	return_if_fail(m_data.get());

	k3d::bitmap* const data = m_data->value();
	if(data && data->width() && data->height() && data->data())
	{
		k3d::scale_bitmap(*data, m_image_buffer);

		k3d::basic_bitmap<k3d::basic_alpha<boost::uint8_t> > temp(64, 64);
		k3d::scale_bitmap(*data, temp);
		m_alpha_buffer = temp;
	}
	else
	{
		k3d::checkerboard_fill(
			m_image_buffer,
			8,
			8,
			k3d::basic_rgb<boost::uint8_t>(32, 32, 32),
			k3d::basic_rgb<boost::uint8_t>(210, 210, 210));

		std::fill(m_alpha_buffer.begin(), m_alpha_buffer.end(), k3d::basic_rgb<boost::uint8_t>(255, 255, 255));
	}

	m_image->set(Gdk::Pixbuf::create_from_data(reinterpret_cast<guint8*>(m_image_buffer.data()),
		Gdk::COLORSPACE_RGB,
		false,
		8,
		m_image_buffer.width(),
		m_image_buffer.height(),
		m_image_buffer.width() * 3));

	m_alpha->set(Gdk::Pixbuf::create_from_data(reinterpret_cast<guint8*>(m_alpha_buffer.data()),
		Gdk::COLORSPACE_RGB,
		false,
		8,
		m_alpha_buffer.width(),
		m_alpha_buffer.height(),
		m_alpha_buffer.width() * 3));
}

} // namespace bitmap_preview

} // namespace k3d

