// K-3D
// Copyright (c) 1995-2006, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the renderframe application, which renders frames scheduled as part of a job with the virtual render farm
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/log.h>
#include <k3dsdk/logbufs.h>
#include <k3dsdk/options.h>
#include <k3dsdk/options_policy.h>
#include <k3dsdk/system.h>
#include <k3dsdk/utility.h>
#include <k3dsdk/version.h>

#include <k3dsdk/xml.h>
using namespace k3d::xml;

#ifdef K3D_PLATFORM_WIN32

	#include <direct.h>
	#define chdir _chdir

#endif // K3D_PLATFORM_WIN32

#include <k3dsdk/fstream.h>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/format.hpp>

#include <cassert>
#include <ctime>
#include <iostream>
#include <vector>

namespace
{

typedef std::vector<std::string> string_array;

typedef std::vector<std::streambuf*> logbufs_t;

bool g_show_timestamps = false;
bool g_show_process = true;
bool g_syslog = false;
bool g_color_level = true;
k3d::log_level_t g_minimum_log_level = k3d::DEBUG;

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// input_file

bool input_file(const element& XMLOperation)
{
	// Sanity checks ...
	assert(XMLOperation.name == "inputfile");

	return true;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// output_file

bool output_file(const element& XMLOperation)
{
	// Sanity checks ...
	assert(XMLOperation.name == "outputfile");

	return true;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// render_operation

bool render_operation(const element& XMLOperation)
{
	// Sanity checks ...
	assert(XMLOperation.name == "renderoperation");

	// Load paths for use during rendering ...


	// Load render data ...
	const std::string type = attribute_text(XMLOperation, "type");
	const std::string name = attribute_text(XMLOperation, "name");
	const std::string sourcepath = attribute_text(XMLOperation, "sourcepath");
	const std::string shaderspath = attribute_text(XMLOperation, "shaderspath");
	const std::string sharepath = attribute_text(XMLOperation, "sharepath");

	// Poke through global options and look for a render engine that matches ...
	std::string command_line;
	const k3d::options::render_engines_t render_engines = k3d::options::render_engines();
	for(k3d::options::render_engines_t::const_iterator render_engine = render_engines.begin(); render_engine != render_engines.end(); ++render_engine)
    {
        if(type != render_engine->type)
            continue;

        if(name != render_engine->name)
            continue;

        command_line = render_engine->render_command;
        break;
    }

	if(command_line.empty())
    {
        k3d::log() << error << "Could not find requested render engine [" << type << "] [" << name << "]" << std::endl;
        return false;
    }

/*
	// Log the render output ...
#ifndef K3D_PLATFORM_WIN32
	command_line += " 2>&1 | tee " + std::string("engine.log");
#else // !K3D_PLATFORM_WIN32
	command_line += " > " + std::string("engine.log");
#endif // K3D_PLATFORM_WIN32
*/

	// Substitute the input source file, shaders, and share paths ...
	boost::format command_line2(command_line);
	command_line2.exceptions(boost::io::all_error_bits ^ boost::io::too_many_args_bit);

	command_line2 % sourcepath;
	command_line2 % shaderspath;
	command_line2 % sharepath;

	// Execute the command ...
	k3d::system::spawn_sync(command_line2.str());

	return true;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// copy_operation

bool copy_operation(const element& XMLOperation)
{
	// Sanity checks ...
	assert(XMLOperation.name == "copyoperation");

	// Extract copy data ...
	const std::string from = attribute_text(XMLOperation, "from");
	const std::string to = attribute_text(XMLOperation, "to");

	// Copy the file ...
#ifndef K3D_PLATFORM_WIN32
	std::string command_line = "cp " + from + " " + to;
#else // !K3D_PLATFORM_WIN32
	std::string command_line = "copy " + from + " " + to;
#endif // K3D_PLATFORM_WIN32

	// Execute the command ...
	k3d::system::spawn_sync(command_line);

	return true;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// view_operation

bool view_operation(const element& XMLOperation)
{
	// Sanity checks ...
	assert(XMLOperation.name == "viewoperation");

	const std::string path = attribute_text(XMLOperation, "path");

#ifndef K3D_PLATFORM_WIN32

	// View the image ...
	boost::format command_line(k3d::options::get_command(k3d::options::command::bitmap_viewer()));
	command_line % path;

	// Execute the command ...
	k3d::system::spawn_async(command_line.str());

#else // !K3D_PLATFORM_WIN32

    ShellExecute(0, "open", path.c_str(), 0, 0, SW_SHOWDEFAULT);

#endif // K3D_PLATFORM_WIN32

	return true;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// frame_operation

/// Handles a single operation during processing for the frame ...
bool frame_operation(const element& XMLOperation)
{
	// Sanity checks ...
	const std::string operation = XMLOperation.name;

	if(operation == "inputfile")
		return input_file(XMLOperation);
	else if(operation == "outputfile")
		return output_file(XMLOperation);
	else if(operation == "renderoperation")
		return render_operation(XMLOperation);
	else if(operation == "copyoperation")
		return copy_operation(XMLOperation);
	else if(operation == "viewoperation")
		return view_operation(XMLOperation);

	return false;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// render_frame

/// Handles all processing for the given frame
bool render_frame(const boost::filesystem::path& FrameDirectory)
{
	// Sanity checks ...
	if(!boost::filesystem::exists(FrameDirectory))
    {
        k3d::log() << error << "Frame directory " << FrameDirectory.native_file_string() << " does not exist" << std::endl;
        return false;
    }

	if(!boost::filesystem::is_directory(FrameDirectory))
    {
        k3d::log() << error << "Frame directory " << FrameDirectory.native_file_string() << " is not a directory" << std::endl;
        return false;
    }

	// Skip the frame if it's complete ...
	if(boost::filesystem::exists(FrameDirectory / "complete"))
		return true;

	// Skip the frame if it errored out ...
	if(boost::filesystem::exists(FrameDirectory / "error"))
		return true;

	// Skip the frame if it's running ...
	if(boost::filesystem::exists(FrameDirectory / "running"))
		return true;

	// Make sure the frame is ready ...
	if(!boost::filesystem::exists(FrameDirectory / "ready"))
    {
        k3d::log() << error << "Frame " << FrameDirectory.native_file_string() << " is not ready" << std::endl;
        return false;
    }

	// Standard logging ...
	k3d::log() << info << "Starting Frame " << FrameDirectory.native_file_string() << std::endl;

	// Switch the frame status to running ...
	boost::filesystem::rename(FrameDirectory / "ready", FrameDirectory / "running");

	// Load the frame options file ...
	element xml_frame_options("empty");
	const boost::filesystem::path control_file_path = FrameDirectory / "control.k3d";
	try
    {
        k3d::filesystem::ifstream stream(control_file_path);
        hide_progress progress;
        parse(xml_frame_options, stream, control_file_path.native_file_string(), progress);
    }
	catch(std::exception& e)
    {
        k3d::log() << error << "Frame " << FrameDirectory.native_file_string() << " error parsing control file " << control_file_path.native_file_string() << " " << e.what() << std::endl;
        return false;
    }

	// Get the frame data ...
	element* const xml_frame = find_element(xml_frame_options, "frame");
	if(!xml_frame)
    {
        k3d::log() << error << "Missing <frame> data in control file " << control_file_path.native_file_string() << std::endl;
        return false;
    }

	// Setup our execution environment ...
	chdir(FrameDirectory.native_file_string().c_str());

	for(element::elements_t::iterator xml_operation = xml_frame->children.begin(); xml_operation != xml_frame->children.end(); ++xml_operation)
		frame_operation(*xml_operation);

	// Switch the frame status to complete ...
	boost::filesystem::rename(FrameDirectory / "running", FrameDirectory / "complete");

	// Standard logging ...
	k3d::log() << info << "Completed Frame " << FrameDirectory.native_file_string() << std::endl;

	return true;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// usage

/// Prints usage info
void usage(const std::string& Name, std::ostream& Stream)
{
	Stream << "usage: " << Name << " [options]" << std::endl;
	Stream << "       " << Name << " [optionspath] [directory ...]" << std::endl;
	Stream << std::endl;
	Stream << "  -h, --help               prints this help information and exits" << std::endl;
	Stream << "      --version            prints program version information and exits" << std::endl;
	Stream << std::endl;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// version

/// Prints version info
void print_version(std::ostream& Stream)
{
	Stream << "K-3D version " << K3D_VERSION << std::endl;
	Stream << "Copyright (c) 1995-2006, Timothy M. Shead.  See the AUTHORS file for contributors." << std::endl;
	Stream << "Licensed by the GNU General Public License.  See the COPYING file for details." << std::endl;
	Stream << "K-3D Home Page: http://www.k-3d.org" << std::endl;
	Stream << std::endl;
}

/////////////////////////////////////////////////////////////////////////////
// setup_logging

/// Sets-up options for logging our output
void setup_logging(const std::string& ProcessName, logbufs_t& LogBufs)
{
	LogBufs.push_back(new k3d::reset_level_buf(k3d::log()));

	if(g_show_timestamps)
		LogBufs.push_back(new k3d::timestamp_buf(k3d::log()));

	if(g_show_process)
		LogBufs.push_back(new k3d::tag_buf("[" + ProcessName + "]", k3d::log()));

	if(g_color_level)
    {
#ifdef K3D_PLATFORM_WIN32
        LogBufs.push_back(new k3d::ansi_to_win32_buf(k3d::log()));
#endif // K3D_PLATFORM_WIN32
        LogBufs.push_back(new k3d::color_level_buf(k3d::log()));
    }

	LogBufs.push_back(new k3d::show_level_buf(k3d::log()));

#ifndef	K3D_PLATFORM_WIN32
	if(g_syslog)
		LogBufs.push_back(new k3d::syslog_buf(k3d::log()));
#endif	// !K3D_PLATFORM_WIN32

	LogBufs.push_back(new k3d::filter_by_level_buf(g_minimum_log_level, k3d::log()));
}

} // namespace

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// main

/// Program main
int main(int argc, char* argv[])
{
	const std::string program_name = boost::filesystem::path(argv[0], boost::filesystem::native).leaf();

	// Put our arguments in a more useable form ...
	string_array options(&argv[1], &argv[argc]);

	// Print a "help" message ...
	if(std::count(options.begin(), options.end(), "-h") || std::count(options.begin(), options.end(), "--help"))
    {
        usage(program_name, std::cout);
        return 0;
    }

	// Print version data ...
	if(options.end() != std::find(options.begin(), options.end(), "--version"))
    {
        print_version(std::cout);
        return 0;
    }

	// Otherwise we should have a minimum of two arguments ...
	if(options.size() < 2)
    {
        usage(program_name, k3d::log());
        return 1;
    }

	// Setup logging right away ...
	logbufs_t logbufs;
	setup_logging(program_name, logbufs);

	// Open the global options file ...
	const boost::filesystem::path options_path(options[0], boost::filesystem::native);
	if(!boost::filesystem::exists(options_path))
    {
        k3d::log() << error << "User options file [" << options_path.native_file_string() << "] does not exist" << std::endl;
        return 1;
    }

	k3d::options::file_storage user_options(options_path);
	k3d::options::set_storage(user_options);

	// Each remaining argument should be a frame path to render ...
	int result = 0;
	for(unsigned long j = 1; j < options.size(); j++)
    {
        if(!render_frame(boost::filesystem::path(options[j], boost::filesystem::native)))
            result = 1;
    }

	// Note: we don't cleanup our logbufs because it causes a crash on gcc >= 3.4 when other code writes to std::cerr

	return result;
}
