#include <qdragobject.h>

#include "config_file.h"
#include "hot_key.h"
#include "icons_manager.h"
#include "tabwidget.h"
#include "userbox.h"

TabWidget::TabWidget()
{
	tabbar = new TabBar(this);
	setTabBar(tabbar);
	//akceptujemy dnd
	setAcceptDrops(true);
	connect(tabbar, SIGNAL(contextMenu(int, const QPoint&)),
			SLOT(onContextMenu(int, const QPoint&)));
	connect(tabbar, SIGNAL(deleteTab(int)),
			SLOT(onDeleteTab(int)));
	connect(tabbar, SIGNAL(wheelEventSignal(QWheelEvent*)),
			SLOT(wheelEvent(QWheelEvent*)));
	connect(tabbar,SIGNAL(mouseDoubleClickEventSignal(QMouseEvent*)),
			SLOT(mouseDoubleClickEvent(QMouseEvent*)));
	//przycisk otwarcia nowej karty pokazywany w lewym górnym rogu
	openChatButton = new QToolButton(this);
   	openChatButton->setIconSet(icons_manager->loadIcon("OpenChat"));
	setCornerWidget(openChatButton, Qt::TopLeft);
	connect(openChatButton, SIGNAL(clicked()),SLOT(newChat()));
	openChatButton->setAutoRaise(true);
	//przycisk zamknięcia aktywnej karty znajdujący się w prawym górnym rogu
	closeChatButton = new QToolButton(this);
   	closeChatButton->setIconSet(icons_manager->loadIcon("TabsRemove"));
	setCornerWidget(closeChatButton, Qt::TopRight);
	connect(closeChatButton, SIGNAL(clicked()),SLOT(deleteTab()));
	closeChatButton->setAutoRaise(true);
	openChatWithWindow=NULL;
}

void TabWidget::closeChatWidget(ChatWidget *chat)
{
	if (chat)
		delete chat;
}

void TabWidget::closeEvent(QCloseEvent* e)
{
	//w zależności od opcji w konfiguracji zamykamy wszystkie karty, lub tylko aktywną	
	if(config_oldStyleClosing)
	{
		QWidget* current = currentPage();
		delete current;

	}
	else	
		//dopóki są jeszcze karty zamykamy aktywną
		while(count())
		{
			QWidget* current = currentPage();
			delete current;
		}
	if (count() > 0)
		e->ignore();
	else
		e->accept();
}

void TabWidget::chatKeyPressed(QKeyEvent* e, CustomInput* k, bool &handled)
{
	handled = true;
	// obsługa skrótów klawiszowych
	if (HotKey::shortCut(e, "ShortCuts", "MoveTabLeft"))
		moveTabLeft();
	else if(HotKey::shortCut(e, "ShortCuts", "MoveTabRight"))
		moveTabRight();
	else if(HotKey::shortCut(e, "ShortCuts", "SwitchTabLeft"))
		switchTabLeft();
	else if(HotKey::shortCut(e, "ShortCuts", "SwitchTabRight"))
		switchTabRight();
	else
		// skrót nie został znaleziony i wykonany. Przekazujemy zdarzenie dalej
		handled = false;
}

void TabWidget::onContextMenu(int id, const QPoint& pos)
{
	emit contextMenu(page(id), pos);
}

void TabWidget::moveTab(int from, int to)
{
	kdebugf();
	QString tablabel=label(from);
	QWidget *w=page(from);
	QIconSet tabiconset = tabIconSet(w);
	QString tabtooltip = tabToolTip(w);
	bool current=(w==currentPage());
	blockSignals(true);
	removePage(w);

	insertTab(w, tabiconset, tablabel, to);

	if(current)
		setCurrentPage(to);

	blockSignals(false);
}

void TabWidget::onDeleteTab(int id)
{
	QWidget* chat = page(id);
	delete chat;
}

void TabWidget::wheelEvent(QWheelEvent* e)
{
	if (e->delta() > 0)
		switchTabLeft();
	else
		switchTabRight();
}

void TabWidget::switchTabLeft()
{
	if (currentPageIndex() == 0)
		setCurrentPage(count() - 1);
	else
		setCurrentPage(currentPageIndex()-1);
}

void TabWidget::switchTabRight()
{
	if (currentPageIndex()==(count()-1))
		setCurrentPage(0);
	else
		setCurrentPage(currentPageIndex()+1);
}

void TabWidget::moveTabLeft()
{
	if (currentPageIndex() == 0)
		moveTab(0, count() - 1);
	else
		moveTab(currentPageIndex(), currentPageIndex() - 1);
}

void TabWidget::moveTabRight()
{
	if (currentPageIndex() == (count() - 1))
		moveTab(count() - 1, 0);
	else
		moveTab(currentPageIndex(), currentPageIndex() + 1);
}

void TabWidget::dragEnterEvent(QDragEnterEvent* e)
{
	kdebugf();
	// Akceptujemu dnd jeśli pochodzi on z UserBox'a lub paska kart
	e->accept((UlesDrag::canDecode(e) && (dynamic_cast<UserBox*>(e->source())) || 
		(QTextDrag::canDecode(e) && dynamic_cast<TabBar*>(e->source()))));
	kdebugf2();
}

void TabWidget::dropEvent(QDropEvent* e)
{
	kdebugf();
	QStringList ules;
	QString altnicks;
	// Jeśli dnd pochodził z userboxa próbujemy dodać nową kartę
	if (dynamic_cast<UserBox*>(e->source()) && UlesDrag::decode(e, ules))
	{
		if (tabbar->selectTab(e->pos()))
		// Jeśli w miejscu upuszczenia jest karta, dodajemy na jej pozycji
			emit openTab(ules, tabbar->indexOf(tabbar->selectTab(e->pos())->identifier()));
		else
		// Jeśli nie na końcu tabbara
			emit openTab(ules, -1);
	}
	// Jeśli dnd pochodził z tabbara, zmieniamy pozycję kart
	else if(dynamic_cast<TabBar*>(e->source()) && QTextDrag::decode(e, altnicks))
	{
		// karta źródłowa
		int movingTabId = altnicks.toUInt(); 
		// karta docelowa
		int destinationTabId;
		if (tabbar->selectTab(e->pos())) 
		// jeśli w miejscu upuszczenia jest inna karta staje się ona kartą docelową
			destinationTabId = tabbar->indexOf(tabbar->selectTab(e->pos())->identifier());
		else
		// Jeśli nie to kartą docelową jest ostatnia karta
			destinationTabId = count() - 1;
		if(movingTabId != -1 && destinationTabId != movingTabId)
                         // zamieniamy miejcami kartę źródłową z docelową
			moveTab(movingTabId, destinationTabId);
	}
	kdebugf2();
}

void TabWidget::windowActivationChange(bool oldActive)
{
	kdebugf();
	ChatWidget *chat = dynamic_cast<ChatWidget*>(currentPage());
	if (isActiveWindow() && !oldActive && chat)
	{
		chat->markAllMessagesRead(); 
		emit chatWidgetActivated(chat);
	}
	kdebugf2();
}

void TabWidget::mouseDoubleClickEvent(QMouseEvent* e)
{
	kdebugf();
	// jeśli dwuklik nastąpil lewym przyciskiem myszy pokazujemy okno openchatwith
	if(e->button() == LeftButton)
		newChat();
	kdebugf2();
}

void TabWidget::newChat()
{
	// jeśli okno openchatwith nie istnieje tworzymy nowe
	if (!openChatWithWindow)
	{
		openChatWithWindow = new OpenChatWith(this, "open_chat_with");
   		connect(openChatWithWindow, SIGNAL(destroyed()), this, SLOT(openChatWithWindowClose()));
		// zapisujemy geometrię okna, aby ją później przywrócić (tabsy nie będą psuły pozycji okna z akcji kadu)
		// TO DO: jet to obejście problemu. W przyszłości zrobić to ładniej ;) (piszę jak Vogel ;p)
		openTabWithGeometry = openChatWithWindow->frameGeometry();
		// pokazujemy je w miejscu w którym nastąpiło wywołanie metody (obecnie znajduje się kursor)
		openChatWithWindow->setGeometry(QCursor::pos().x(), QCursor::pos().y(), openChatWithWindow->width(), openChatWithWindow->height());// jak wykonuje się to po pokazaniu okienkoa występuje denerwujące miganie
		openChatWithWindow->show();
	}
	// jeśli istnieje przywracamy je na pierwszy plan w miejscu cursora
	else
	{
		openChatWithWindow->setGeometry(QCursor::pos().x(), QCursor::pos().y(), openChatWithWindow->width(), openChatWithWindow->height());// jak wykonuje się to po pokazaniu okienka występuje denerwujące miganie
		openChatWithWindow->setWindowState(openChatWithWindow->windowState() & ~WindowMinimized);
		openChatWithWindow->raise();
	}
}

void TabWidget::deleteTab()
{
	// zamykamy bieżącą kartę
	QWidget* current = currentPage();
	delete current;
}

void TabWidget::openChatWithWindowClose()
{
	// ponownie zapisujemy poprawną pozycję okna openTabWith
	config_file.writeEntry("General", "OpenChatWith", openTabWithGeometry);
	/* 
	po zamknięciu okna openchatwith przypisujemy NULL do wskaźnika do niego, 
	wykonujemy to aby mieć pewność że nie odwołamy się do nieistniejącego elementu, bądź nie przypiszemy adresu
	nowego obiektu do już zajętego wskaźnika [wyciek pamięci](w niektórych książkach zalecają używanie tego zawsze
	po użyciu delete obiekt ;) A w jednym zdaniu zabezpieczamy się przed wysypem modułu i mamy pewność czy okno jest
	otwarte czy nie)
	*/	
	openChatWithWindow=NULL;
}

void TabWidget::configurationUpdated()
{
	// odświeżenie ikon
   	openChatButton->setIconSet(icons_manager->loadIcon("OpenChat"));
   	closeChatButton->setIconSet(icons_manager->loadIcon("TabsRemove"));

	// uaktualniamy zmienne konfiguracyjne
	closeChatButton->setShown(config_file.readBoolEntry("Tabs", "CloseButton"));
	openChatButton->setShown(config_file.readBoolEntry("Tabs", "OpenChatButton"));
	config_oldStyleClosing = config_file.readBoolEntry("Tabs", "OldStyleClosing");
	tabbar->setShowCloseButton(config_file.readBoolEntry("Tabs", "CloseButtonOnTab"));
}

TabBar::TabBar(QWidget *parent, char *name)
	: QTabBar(parent, name)
{
	// dzięki temu sygnał mouseMoveEvent jest emitowany także w momencie "zwykłego" poruszania się kursora myszy nad tabbarem
	setMouseTracking(true);

	// sposób działania przycisku na każdej karcie bazuje na na rozwiazaniu z QDevelop (http://qdevelop.org)
	// Tworzony jest przycisk zamknięcia karty pokazywany na aktywnej karcie
	crossButton = new QToolButton(this);
   	crossButton->hide();
   	connect(crossButton, SIGNAL(clicked()), this, SLOT(closeTab()) );
   	crossButton->setGeometry(0,0,15,15);
   	crossButton->setIconSet(icons_manager->loadIcon("TabsClose"));
	// staje się "podświetlany"
	crossButton->setAutoRaise(true);
	// domyśnie żadna karta nie jest "naciśnięta" (nie znajduje się na niej przycisk zamknięcia)
	clickedItem=-1;
	// Podłączamy się do sygnału emitowanego po jakiejkolwiek zmianie wyglądu tabbara. 
	// Dzięki temu mozemy uaktualniać pozycję przycisku zamknięcia na kartach np. w chwili zamknięcia karty.
   	connect(this, SIGNAL(layoutChanged()), SLOT(replaceCross()));
}

void TabBar::mousePressEvent(QMouseEvent* e)
{
	QTab* selectedtab = selectTab(e->pos());

	if (selectedtab && e->button() == RightButton)
		emit contextMenu(indexOf(selectedtab->identifier()), mapToGlobal(e->pos()));
	MouseStart = e->pos();

	QTabBar::mousePressEvent(e);
}

void TabBar::mouseReleaseEvent(QMouseEvent* e)
{
	QTab* selectedtab = selectTab(e->pos());

	if (selectedtab && e->button() == MidButton)
		emit deleteTab(indexOf(selectedtab->identifier()));
	QTabBar::mouseReleaseEvent(e);
}

void TabBar::wheelEvent(QWheelEvent *e)
{
	emit wheelEventSignal(e);
}

void TabBar::mouseMoveEvent(QMouseEvent* e)
{
	kdebugf();
	// Jeśli zaznaczona jest karta i wciśnięty lewy przycisk myszy, inicjujemy dnd
	if ((e->state() & LeftButton) && (selectTab(MouseStart)) && ((MouseStart - e->pos()).manhattanLength() >= 15))
	{
		// "przenoszonym" tekstem jest numer aktywowanej karty
		QString drag_text = QString::number(indexOf(selectTab(MouseStart)->identifier()));
		// Bezpośrednia inicjacja dnd
		QDragObject* d = new QTextDrag(drag_text, this);
		d->dragCopy();
	}
	// pomysł rozwiazania zaczęrpnięty z QDevelop
	// Jeśli aktywna jest opcja pokazywania przycisku zamknięcia na kartach i w obrębie zdarzenia jest karta
	else if (selectTab(e->pos()) && showCloseButton)
	{
		// pokazujemy przycik na odpowiedniej pozycji
		clickedItem = indexOf(selectTab(e->pos())->identifier()); 
		// pierwszy parametr ustawia przycisk 5 pixeli od prawej krawędzi karty.         
		crossButton->setGeometry(tabAt(clickedItem)->rect().x()+tabAt(clickedItem)->rect().width()-crossButton->width()-5, 6, crossButton->width(), crossButton->height());
                crossButton->show();
	}
	else
		QTabBar::mouseMoveEvent(e);
	kdebugf2();
}

void TabBar::leaveEvent(QEvent* e)
{
	// w chwili gdy kursor opuszcza tabbar chowamy przycisk zamknięcia
       	crossButton->hide();
}

void TabBar::closeTab()
{
	emit deleteTab(clickedItem);
}

void TabBar::mouseDoubleClickEvent(QMouseEvent* e)
{
	kdebugf();
	// w celu ułatwienia sobie zadania przekazujemy zdarzenie dalej- tu klasie tabdialog
	emit mouseDoubleClickEventSignal(e);
	kdebugf2();
}

void TabBar::setShowCloseButton(bool show)
{
	// odświeżenie ikonki
   	crossButton->setIconSet(icons_manager->loadIcon("TabsClose"));	

	showCloseButton = show;
	// w zaleźności czy w konfiguracji włączone jest pokazywanie przycisku zamykania na kartach ukrywamy go lub pokazujemy
	if(!showCloseButton)
		crossButton->hide();
}

void TabBar::replaceCross()
{
	// jeśli przycisk zamknięciakarty jest widoczny na karcie i w miejscu kursora znajuje się karta
	if(crossButton->isShown() && selectTab(mapFromGlobal(QCursor::pos())))
	{
		// uaktualniamy pozycję przycisku w prawym rogu karty
		clickedItem = indexOf(selectTab(mapFromGlobal(QCursor::pos()))->identifier()); 
		// pierwszy parametr ustawia przycisk 5 pixeli od prawej krawędzi karty.
		crossButton->setGeometry(tabAt(clickedItem)->rect().x()+tabAt(clickedItem)->rect().width()-crossButton->width()-5, 6, crossButton->width(), crossButton->height());
                crossButton->show();
	}
}
