/***************************************************************************

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

***************************************************************************

Copyright (C) 2007 by
Tomasz 'Dorregaray' Rostanski
rozteck <at> interia.pl

 ***************************************************************************/

#include <qapplication.h>
#include <qspinbox.h>
#include <qtooltip.h>

#include "color_button.h"
#include "config_file.h"
#include "debug.h"
#include "select_font.h"
#include "kadu.h"

#include "osd_configuration_widget.h"

OSDConfigurationWidget::OSDConfigurationWidget(QWidget *parent, char *name)
	: NotifierConfigurationWidget(parent, name), currentNotifyEvent("")
{
	warning = new QLabel("<b>" + tr("Uncheck 'Set to all' in 'OSD Hints' page to edit these values") + "</b>", this);
	preview = new QLabel(tr("<b>Preview</b> text"), this);
	font = new SelectFont(this);
	foregroundColor = new ColorButton(this);
	backgroundColor = new ColorButton(this);
	borderColor = new ColorButton(this);
	timeout = new QSpinBox(this);
	timeout->setSuffix(" s");
	timeout->setSpecialValueText(tr("Dont hide"));
	mask = new QComboBox(this);
	syntax = new QLineEdit(this);
	QToolTip::add(syntax, tr("Notification syntax: %&m - notification text (eg. New message), %&d - details (eg. message quotation),\n%&i - notification icon\n") + qApp->translate("@default", Kadu::SyntaxTextNotify));

	connect(font, SIGNAL(fontChanged(QFont)), this,
			   SLOT(fontChanged(QFont)));
	connect(foregroundColor, SIGNAL(changed(const QColor &)), this,
			   SLOT(foregroundColorChanged(const QColor &)));
	connect(backgroundColor, SIGNAL(changed(const QColor &)), this,
			   SLOT(backgroundColorChanged(const QColor &)));
	connect(borderColor, SIGNAL(changed(const QColor &)), this,
			   SLOT(borderColorChanged(const QColor &)));
	connect(timeout, SIGNAL(valueChanged(int)),this,
			  SLOT(timeoutChanged(int)));
	connect(mask, SIGNAL(activated(int)),this,
			  SLOT(maskChanged(int)));
	connect(syntax, SIGNAL(textChanged(const QString &)), this,
			   SLOT(syntaxChanged(const QString &)));

	QGridLayout *gridLayout = new QGridLayout(this, 0, 0, 0, 5);
	gridLayout->addMultiCellWidget(warning, 0, 0, 0, 1);
	gridLayout->addMultiCellWidget(preview, 1, 1, 0, 1);
	gridLayout->addWidget(
					new QLabel(tr("Font") + ":", this), 2, 0, Qt::AlignRight);
	gridLayout->addWidget(font, 2, 1);
	gridLayout->addWidget(
					new QLabel(tr("Font color") + ":", this), 3, 0, Qt::AlignRight);
	gridLayout->addWidget(foregroundColor, 3, 1);
	gridLayout->addWidget(
					new QLabel(tr("Background color") + ":", this), 4, 0, Qt::AlignRight);
	gridLayout->addWidget(backgroundColor, 4, 1);
	gridLayout->addWidget(
					new QLabel(tr("Border color") + ":", this), 5, 0, Qt::AlignRight);
	gridLayout->addWidget(borderColor, 5, 1);
	gridLayout->addWidget(
					new QLabel(tr("Timeout") + ":", this), 6, 0, Qt::AlignRight);
	gridLayout->addWidget(timeout, 6, 1);
	gridLayout->addWidget(
					new QLabel(tr("Mask Effect") + ":", this), 7, 0, Qt::AlignRight);
	gridLayout->addWidget(mask, 7, 1);
	gridLayout->addWidget(
					new QLabel(tr("Syntax") + ":", this), 8, 0, Qt::AlignRight);
	gridLayout->addWidget(syntax, 8, 1);

	mask->insertItem(tr("Plain"));
	mask->insertItem(tr("Dissolve"));
}

void OSDConfigurationWidget::saveNotifyConfigurations()
{
	kdebugf();

	if (currentNotifyEvent != "")
		hintProperties[currentNotifyEvent] = currentProperties;

	CONST_FOREACH(hintProperty, hintProperties)
	{
		const QString &event = (*hintProperty).eventName;

		config_file.writeEntry("osdhints", event + "_font",
									  (*hintProperty).font);
		config_file.writeEntry("osdhints", event + "_fgcolor",
									  (*hintProperty).foregroundColor);
		config_file.writeEntry("osdhints", event + "_bgcolor",
									  (*hintProperty).backgroundColor);
		config_file.writeEntry("osdhints", event + "_bordercolor",
									  (*hintProperty).borderColor);
		config_file.writeEntry("osdhints", event + "_timeout",
									  (int)(*hintProperty).timeout);
		config_file.writeEntry("osdhints", event + "_syntax",
									  (*hintProperty).syntax);
		config_file.writeEntry("osdhints", event + "_maskeffect",
									  (*hintProperty).maskEffect);
	}
}

void OSDConfigurationWidget::switchToEvent(const QString &event)
{
	kdebugf();

	if (currentNotifyEvent != "")
		hintProperties[currentNotifyEvent] = currentProperties;

	if (hintProperties.contains(event))
	{
		currentProperties = hintProperties[event];
		currentNotifyEvent = event;
	}
	else
	{
		currentNotifyEvent = event;
		currentProperties.eventName = event;

		config_file.addVariable("osdhints", event + "_font",
										defaultFont);
		config_file.addVariable("osdhints", event + "_fgcolor",
										paletteForegroundColor());
		config_file.addVariable("osdhints", event + "_bgcolor",
										paletteBackgroundColor());
		config_file.addVariable("osdhints", event + "_bordercolor",
										paletteForegroundColor());

		currentProperties.syntax =
				config_file.readEntry("osdhints", event + "_syntax");
		currentProperties.font =
				config_file.readFontEntry("osdhints", event + "_font", defaultFont);
		currentProperties.foregroundColor =
				config_file.readColorEntry("osdhints", event + "_fgcolor");
		currentProperties.backgroundColor =
				config_file.readColorEntry("osdhints", event + "_bgcolor");
		currentProperties.borderColor =
				config_file.readColorEntry("osdhints", event + "_bordercolor");
		currentProperties.timeout =
				config_file.readUnsignedNumEntry("osdhints", event + "_timeout", 10);
		currentProperties.maskEffect =
				config_file.readNumEntry("osdhints", event + "_maskeffect");
	}

	font->setFont(currentProperties.font);
	foregroundColor->setColor(currentProperties.foregroundColor);
	backgroundColor->setColor(currentProperties.backgroundColor);
	borderColor->setColor(currentProperties.borderColor);
	timeout->setValue(currentProperties.timeout);
	mask->setCurrentItem(currentProperties.maskEffect > 1 ? 1 : currentProperties.maskEffect);
	syntax->setText(currentProperties.syntax);

	preview->setFont(currentProperties.font);
	preview->setPaletteForegroundColor(currentProperties.foregroundColor);
	preview->setPaletteBackgroundColor(currentProperties.backgroundColor);
}

void OSDConfigurationWidget::fontChanged(QFont font)
{
	currentProperties.font = font;
	preview->setFont(font);
}

void OSDConfigurationWidget::foregroundColorChanged(const QColor &color)
{
	currentProperties.foregroundColor = color;
	preview->setPaletteForegroundColor(color);
}

void OSDConfigurationWidget::backgroundColorChanged(const QColor &color)
{
	currentProperties.backgroundColor = color;
	preview->setPaletteBackgroundColor(color);
}

void OSDConfigurationWidget::borderColorChanged(const QColor &color)
{
	currentProperties.borderColor = color;
}

void OSDConfigurationWidget::timeoutChanged(int timeout)
{
	currentProperties.timeout = timeout;
}

void OSDConfigurationWidget::maskChanged(int mask)
{
	currentProperties.maskEffect = mask;
}

void OSDConfigurationWidget::syntaxChanged(const QString &syntax)
{
	currentProperties.syntax = syntax;
}

void OSDConfigurationWidget::setAllEnabled(bool enabled)
{
	warning->setShown(enabled);
	font->setDisabled(enabled);
	foregroundColor->setDisabled(enabled);
	backgroundColor->setDisabled(enabled);
	borderColor->setDisabled(enabled);
	timeout->setDisabled(enabled);
	mask->setDisabled(enabled);
	preview->setDisabled(enabled);
}
