/******************************************************************************
	styledromdisplay.cpp - A cool, styled HTML-based ROM info display.
	
	Copyright (c) 2006 by Brian S. Stephan
	bssteph@incorporeal.org
	
	*************************************************************************
	*                                                                       *
	* This program is free software; you can redistribute it and/or modify  *
	* it under the terms of the GNU General Public License as published by  *
	* the Free Software Foundation; either version 2 of the License, or     *
	* (at your option) any later version.                                   *
	*                                                                       *
	*************************************************************************
	
 *****************************************************************************/

#include <kurl.h>
#include <klocale.h>
#include <kdebug.h>

#include "rommetainformation.h"
#include "collectionaccess.h"

#include "styledromdisplay.h"

class StyledRomDisplay::Private
{
public:
	Private() {}
	
	QString currentRomUrl;
	KURL currentCss;
	QWidget *parent;
	unsigned int displayOpts;
};

StyledRomDisplay::StyledRomDisplay(unsigned int options, QWidget *parentWidget, const char *widgetname, QObject *parent, const char * name, GUIProfile prof)
	: KHTMLPart(parentWidget, widgetname), d(new Private)
{
	d->parent = parentWidget;
	d->displayOpts = options;
}
	
StyledRomDisplay::~StyledRomDisplay()
{
	kdDebug() << k_funcinfo << endl;
	delete d;
}

void StyledRomDisplay::setRom(const QString &romUrl)
{
	d->currentRomUrl = romUrl;
	begin();
	// TODO loading of stylesheets from files
	if (d->currentCss.isEmpty()) {
		// load our generic hardcoded stylesheet
		generateCss();
	}
	generateHTML();
	end();
}

void StyledRomDisplay::generateHTML()
{
	QString html;
	
	kdDebug() << "read the rom info for " << d->currentRomUrl << endl;
	
	Kamefu::RomMetaInformation metainfo = Kamefu::CollectionAccess::self()->getRomMetadataFromRomUrl(d->currentRomUrl);
	
	if (!metainfo.isValid()) {
		kdWarning() << "MetadataEdit::showMeta(): Could not retrieve ROM!" << endl;
	} else {
		html += "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\"";
		html += "\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">";
		html += "<html xmlns=\"http://www.w3.org/1999/xhtml\" lang=\"en\" xml:lang=\"en\">";
		html += QString("<head><title>%1</title></head>").arg(i18n("ROM Information"));
		html += "<body>";
		
		// main wrapper
		html += "<div class=\"body\">";
		
		// top bar
		if (d->displayOpts) {		// draw the header if we are supposed to display anything
			html += "<div class=\"header mainheader\">";
			
			if (d->displayOpts & DISPLAY_ROMNAME) {
				html += "<div class=\"headertext romname\"><span>";
				html += metainfo.romName();
				html += "</span></div>";
			}
			if (d->displayOpts & DISPLAY_SYSTEM) {
				html += "<div class=\"headertext romsystem\"><span>";
				html += metainfo.system();
				html += "</span></div>";
			}
			
			html += "<div class=\"pad\"></div>";
			html += "</div>";	// end of mainheader
		}
		
		// more information area
		if (d->displayOpts & (DISPLAY_TITLESCREEN | DISPLAY_DEVELOPER | DISPLAY_YEAR | DISPLAY_COUNTRY | DISPLAY_GENRE)) {
			html += "<div class=\"underheader moreinfo\">";
			html += "<div class=\"info\">";
			
			// area for title image
			if (d->displayOpts & DISPLAY_TITLESCREEN) {
				if (metainfo.titlescreenPath() != "") {
					KURL titleImg;
					titleImg.setPath(metainfo.titlescreenPath());
					
					if (titleImg.isLocalFile()) {
						html += "<div class=\"image titlescreen\"><span>";
						html += QString("<img src=\"%1\"").arg(titleImg.url());
						html += QString(" alt=\"%1\" />").arg(i18n("Title Screen"));
						html += "</span></div>";
					}
				}
			}
			
			// extra info
			if (d->displayOpts & (DISPLAY_DEVELOPER | DISPLAY_YEAR | DISPLAY_COUNTRY | DISPLAY_GENRE)) {
				html += "<div class=\"details\">";
				
				if ((metainfo.editor() != "Unknown") && (d->displayOpts & DISPLAY_DEVELOPER))
					html += QString("<div class=\"developer\"><span class=\"l\">%1: </span><span class=\"t\">%2</span></div>")
						.arg(i18n("Developer"))
						.arg(metainfo.editor());
				if (metainfo.year() != "0" && (d->displayOpts & DISPLAY_YEAR))
					html += QString("<div class=\"year\"><span class=\"l\">%1: </span><span class=\"t\">%2</span></div>")
						.arg(i18n("Year"))
						.arg(metainfo.year());
				if (metainfo.country() != "Unknown" && (d->displayOpts & DISPLAY_COUNTRY))
					html += QString("<div class=\"country\"><span class=\"l\">%1: </span><span class=\"t\">%2</span></div>")
						.arg(i18n("Country"))
						.arg(metainfo.country());
				if (metainfo.genre() != "Unknown" && (d->displayOpts & DISPLAY_GENRE))
					html += QString("<div class=\"genre\"><span class=\"l\">%1: </span><span class=\"t\">%2</span></div>")
						.arg(i18n("Genre"))
						.arg(metainfo.genre());
				
				html += "</div>";	// end of details
			}
			
			html += "</div>";	// end of info
			
			html += "<div class=\"pad\"></div>";
			html += "</div>";	// end of moreinfo
		}
		
		if (metainfo.screenshotPath() != "" && (d->displayOpts & DISPLAY_SCREENSHOTS)) {
			KURL screenImg;
			screenImg.setPath(metainfo.screenshotPath());
			
			// TODO: handling of screenshots (more than one?)
			html += "<div class=\"extrainfo\">";
			html += "<div class=\"header extraheader\">";
			html += "<div class=\"headertext label\">";
			html +=	QString("<span>%1</span>").arg(i18n("Additional Screenshots"));
			html += "</div></div>";		// end of extraheader
			
			html += "<div class=\"underheader screenshots\">";
			html += "<div class=\"image screenshot\"><span>";
			html += QString("<img src=\"%1\"").arg(screenImg.url());
			html += QString(" alt=\"%1\" />").arg(i18n("Screenshot"));
			html += "</span></div>";	// end of screenshot
			
			html += "</div>";
			
			html += "<div class=\"pad\"></div>";
						
			html += "</div>";	// end of extrainfo
		}
		
		html += "</div></body></html>";
		
		// TODO: more data, including details (header type, etc?)
	}
		
	write(html);
}

void StyledRomDisplay::generateCss()
{
	QString css;
	
	// this is mostly a cheap knockoff observed from amarok's default context browser style
	
	QString textColor = d->parent->colorGroup().text().name();
	QString highlightTextColor = d->parent->colorGroup().highlightedText().name();
	QString backgroundColor = d->parent->colorGroup().base().name();
	QString highlightBackgroundColor = d->parent->colorGroup().highlight().name();
	
	css += "body { font-size: 10pt; }";
	css += QString("div.body { color: %1; background-color: %2 }").arg(textColor).arg(backgroundColor);
	css += QString("div.header { background-color: %1; padding: 3px; border: 1px solid black; }")
		.arg(highlightBackgroundColor);
	css += QString("div.headertext { color: %1; }")
		.arg(highlightTextColor);
	css += "div.romname { font-size: 110%; font-weight: bold; }";
	css += "div.romsystem { font-size: 100%; }";
	css += "div.underheader { border: 1px solid black; border-top: none; padding: 5px; }";
	css += "div.info { font-size: 75%; text-align: right; }";
	css += "div.image { margin: 5px; }";
	css += "div.titlescreen { float: left; }";
	css += "div.titlescreen img { max-width: 128px; }";
	css += "div.pad { clear: both; }";
	css += "div.header { margin-top: 10px; }";
	
	setUserStyleSheet(css);
}

#include "styledromdisplay.moc"
