/***************************************************************************
 *   Copyright (C) 2004-2009 by Thomas Fischer                             *
 *   fischer@unix-ag.uni-kl.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <stdio.h>

#include <qfile.h>
#include <qdir.h>
#include <qprocess.h>
#include <qstringlist.h>
#include <qbuffer.h>
#include <qwaitcondition.h>
#include <qapplication.h>

#include <file.h>
#include <encoderlatex.h>
#include "fileimporterbibutils.h"

namespace BibTeX
{

    FileImporterBibUtils::FileImporterBibUtils( BibTeX::File::FileFormat inputFormat )
            : FileImporter(), m_workingDir( createTempDir() ), m_inputFormat( inputFormat ), m_bibTeXImporter( new FileImporterBibTeX( false, "utf-8" ) )
    {
        m_processBuffer = new QBuffer();
    }

    FileImporterBibUtils::~FileImporterBibUtils()
    {
        delete m_processBuffer;
        deleteTempDir( m_workingDir );
        delete m_bibTeXImporter;
    }

    File* FileImporterBibUtils::load( QIODevice *iodevice )
    {
        m_cancelFlag = false;
        if ( !iodevice->isReadable() )
        {
            qDebug( "iodevice is not readable" );
            return NULL;
        }
        if ( !iodevice->isOpen() )
        {
            qDebug( "iodevice is not open" );
            return NULL;
        }

        if ( !iodeviceToXMLbuffer( iodevice ) || m_cancelFlag )
            return NULL;

        return xmlBufferToBibTeXFile();
    }

    bool FileImporterBibUtils::guessCanDecode( const QString & text )
    {
        return guessInputFormat( text ) != BibTeX::File::formatUndefined;
    }

    File::FileFormat FileImporterBibUtils::guessInputFormat( const QString &text )
    {
        if ( text.find( "TY  - " ) >= 0 )
            return BibTeX::File::formatRIS;
        else if ( text.find( "%A " ) >= 0 )
            return BibTeX::File::formatEndNote;
        else if ( text.find( "FN ISI Export Format" ) >= 0 )
            return BibTeX::File::formatISI;
        else
            return BibTeX::File::formatUndefined;
    }

    void FileImporterBibUtils::cancel()
    {
        m_bibTeXImporter->cancel();
        m_cancelFlag = true;
    }

    bool FileImporterBibUtils::iodeviceToXMLbuffer( QIODevice *iodevice )
    {
        QWaitCondition wc;

        m_processBuffer->open( IO_WriteOnly );
        m_process = NULL;
        switch ( m_inputFormat )
        {
        case BibTeX::File::formatISI:
            m_process = new QProcess( QStringList::split( ' ', "isi2xml -i utf8 -u" ) );
            break;
        case BibTeX::File::formatWordBib:
            m_process = new QProcess( QStringList::split( ' ', "wordbib2xml -i utf8 -u" ) );
            break;
        case BibTeX::File::formatAds:
            m_process = new QProcess( QStringList::split( ' ', "ads2xml -i utf8 -u" ) );
            break;
        case BibTeX::File::formatEndNote:
            m_process = new QProcess( QStringList::split( ' ', "end2xml -i utf8 -u" ) );
            break;
        case BibTeX::File::formatEndNoteXML:
            m_process = new QProcess( QStringList::split( ' ', "endx2xml -i utf8 -u" ) );
            break;
        case BibTeX::File::formatRIS:
            m_process = new QProcess( QStringList::split( ' ', "ris2xml -i utf8 -u" ) );
            break;
        case BibTeX::File::formatMODS:
            /* m_process = NULL; */
            break;
        default:
            qDebug( "Cannot handle input format %i", m_inputFormat );
            return false;
        }

        if ( m_process != NULL )
        {
            m_waiting = true;
            connect( m_process, SIGNAL( processExited() ), this, SLOT( wakeUp() ) );
            connect( m_process, SIGNAL( readyReadStdout() ), this, SLOT( slotReadyStdout() ) );
            connect( m_process, SIGNAL( readyReadStderr() ), this, SLOT( slotReadyStderr() ) );

            m_process->start();
            if ( m_process->isRunning() )
            {
                QByteArray inData = iodevice->readAll();
                m_process->writeToStdin( inData );
                qApp->processEvents();
                m_process->closeStdin();

                int nothingHappens = 20;
                while ( m_waiting )
                {
                    wc.wait( 250 );
                    qApp->processEvents();
                    --nothingHappens;
                }

                if ( nothingHappens <= 0 )
                    m_process->kill();

                if ( !m_process->normalExit() )
                {
                    qDebug( "%s did not exit in a clean fashion", m_process->arguments()[0].latin1() );
                    delete m_process;
                    return false;
                }
            }
            else
            {
                qDebug( "%s did not start", m_process->arguments()[0].latin1() );
                delete m_process;
                return false;
            }
        }
        else
        {
            m_processBuffer->writeBlock( iodevice->readAll() );
        }

        m_processBuffer->close();

        delete m_process;
        return true;
    }

    BibTeX::File* FileImporterBibUtils::xmlBufferToBibTeXFile()
    {
        QWaitCondition wc;

        m_waiting = true;
        m_process = new QProcess( QStringList::split( ' ', "xml2bib -i utf8 -o utf8 -sk" ) );
        connect( m_process, SIGNAL( processExited() ), this, SLOT( wakeUp() ) );
        connect( m_process, SIGNAL( readyReadStdout() ), this, SLOT( slotReadyStdout() ) );
        connect( m_process, SIGNAL( readyReadStderr() ), this, SLOT( slotReadyStderr() ) );

        if ( m_process->start() )
        {
            QBuffer *tempBuffer = m_processBuffer;
            m_processBuffer = new QBuffer();

            tempBuffer->open( IO_ReadOnly );
            m_process->writeToStdin( tempBuffer->readAll() );
            qApp->processEvents();
            m_process->closeStdin();
            tempBuffer->close();

            m_processBuffer->open( IO_WriteOnly );
            int nothingHappens = 20;
            while ( m_waiting )
            {
                wc.wait( 250 );
                qApp->processEvents();
                --nothingHappens;
            }
            m_processBuffer->close();

            if ( nothingHappens <= 0 )
                m_process->kill();

            delete tempBuffer;

            if ( ! m_process->normalExit() )
            {
                delete m_process;
                return NULL;
            }

            m_processBuffer->open( IO_ReadOnly );
            File *bibTeXFile = m_bibTeXImporter->load( m_processBuffer );
            m_processBuffer->close();

            delete m_process;
            return bibTeXFile;
        }
        else
        {
            delete m_process;
            return NULL;
        }

        delete m_process;
        return NULL;
    }

    QString FileImporterBibUtils::createTempDir()
    {
        QString result = QString::null;
        QFile *devrandom = new QFile( "/dev/random" );

        if ( devrandom->open( IO_ReadOnly ) )
        {
            Q_UINT32 randomNumber;
            if ( devrandom->readBlock(( char* ) & randomNumber, sizeof( randomNumber ) ) > 0 )
            {
                randomNumber |= 0x10000000;
                result = QString( "/tmp/bibtex-%1" ).arg( randomNumber, sizeof( randomNumber ) * 2, 16 );
                if ( !QDir().mkdir( result ) )
                    result = QString::null;
            }
            devrandom->close();
        }

        delete devrandom;

        return result;
    }

    void FileImporterBibUtils::deleteTempDir( const QString& directory )
    {
        QDir dir = QDir( directory );
        QStringList subDirs = dir.entryList( QDir::Dirs );
        for ( QStringList::Iterator it = subDirs.begin(); it != subDirs.end(); it++ )
        {
            if (( QString::compare( *it, "." ) != 0 ) && ( QString::compare( *it, ".." ) != 0 ) )
                deleteTempDir( *it );
        }
        QStringList allEntries = dir.entryList( QDir::All );
        for ( QStringList::Iterator it = allEntries.begin(); it != allEntries.end(); it++ )
            dir.remove( *it );

        QDir().rmdir( directory );
    }

    void FileImporterBibUtils::wakeUp()
    {
        m_waiting = false;
    }

    void FileImporterBibUtils::slotReadyStdout()
    {
        m_processBuffer->writeBlock( m_process->readStdout() );
    }

    void FileImporterBibUtils::slotReadyStderr()
    {
        QByteArray ba = m_process->readStderr();
        QTextStream bats( ba, IO_ReadOnly );
        bats.setEncoding( QTextStream::UnicodeUTF8 );
        qDebug( "%s", bats.read().latin1() );
    }

}
#include "fileimporterbibutils.moc"

