/*
 * Copyright (C) 2002 Richard Moore <rich@kde.org>
 *
 *   This program is free software; you can redistribute it and/or modify 
 *   it under the terms of the GNU General Public License as published by 
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 */

#include <qtimer.h>

#include <kaboutapplication.h>
#include <kaboutdata.h>
#include <kaction.h>
#include <kapplication.h>
#include <kcmenumngr.h>
#include <kdebug.h>
#include <kglobal.h>
#include <klocale.h>
#include <kpopupmenu.h>
#include <kstandarddirs.h>
#include <kxmlguifactory.h>

#include <kparts/genericfactory.h>

#include "actions.h"
#include "kdetvwidget.h"
#include "libkdetvactions.h"
#include "kdetv.h"
#include "viewmanager.h"

#include "kdetvpart.h"

typedef KParts::GenericFactory<KdetvPart> KdetvPartFactory;
K_EXPORT_COMPONENT_FACTORY( libkdetvpart, KdetvPartFactory );

KdetvPart::KdetvPart( QWidget *wparent, const char *wname,
                      QObject *parent, const char *name,
                      const QStringList &/*args*/ )
    : KParts::ReadOnlyPart( parent ? parent : wparent, name ? name : "kdetv_part" ),
      autoStart(true)
{
    kdDebug() << "Creating KdetvPart" << endl;

    setInstance( KdetvPartFactory::instance() );

    // Create TV Widget
    tv = new KdetvWidget( wparent, wname ? wname : 0 );
    setWidget( tv );

    kdDebug() << "KdetvPart: Created TV widget" << endl;

    // Kdetv actions
    actions = new LibKdetvActions( tv->driver() );
    actions->createActions( actionCollection() );

    // Setup Action Collection
    actionCollection()->setHighlightingEnabled( true );

    // Define UI
    setXMLFile( "kdetvpartui.rc" );

    QTimer::singleShot( 0, this, SLOT(maybeStart()) );
    kdDebug() << "KdetvPart: Created" << endl;
}

KdetvPart::~KdetvPart()
{
}

void KdetvPart::start()
{
    tv->driver()->start();
    addContextMenu( tv->view() );
}

void KdetvPart::maybeStart()
{
    if ( autoStart )
	start();
}

KAboutData *KdetvPart::createAboutData()
{
    // the non-i18n name here must be the same as the directory in
    // which the part's rc file is installed ('partrcdir' in the
    // Makefile)
    KAboutData *about = new KAboutData( "kdetv", I18N_NOOP("kdetv"),
                                        "Development post 0.8.0",
                                        I18N_NOOP("TV for KDE"),
                                        KAboutData::License_LGPL,
                                        I18N_NOOP("(c) 2002-2004 The kdetv Developers"),
                                        0, "http://www.kdetv.org/");
    about->addAuthor("George Staikos", I18N_NOOP("Primary author and maintainer"), "staikos@kde.org");
    about->addAuthor("Richard Moore", I18N_NOOP("GUI design, channel plugins"), "rich@kde.org");
    about->addAuthor("Kevin Hessels", I18N_NOOP("Configuration dialog"), "khessels@shaw.ca");
    about->addAuthor("Stefan Hellwig", I18N_NOOP("Webmaster, developer"), "stefan@stefanhellwig.de");
    about->addAuthor("Rizsanyi Zsolt", I18N_NOOP("Infrared remote support, developer"), "rizsanyi@myrealbox.com");
    about->addAuthor("Dawit Alemayehu", I18N_NOOP("Developer"), "adawit@kde.org");

    return about;
}

void KdetvPart::addContextMenu( bool xmlgui )
{
   tv->addContextMenu( createContextMenu( xmlgui ) );
}

KPopupMenu *KdetvPart::createContextMenu( bool xmlgui ) 
{
    KPopupMenu *pop = 0;
    if ( xmlgui ) {
	pop = createContextMenu( createContextMenu(QString("screen_context_popup")) );
	if ( pop )
	    return pop;
    }

    // Fall back to manual menu creation
    pop = new KPopupMenu;

    actions->channel()->plug( pop );
    actions->channelUp()->plug( pop );
    actions->channelDown()->plug( pop );
    actions->volumeSlider()->plug( pop );
    actions->volumeMute()->plug( pop );
    pop->insertSeparator();

    return pop;
}

KPopupMenu *KdetvPart::createContextMenu( const QString &name ) 
{
    kdDebug() << "KdetvPart: createContextMenu(...)" << endl;

    KXMLGUIFactory *fact = factory();
    if ( !fact )
        return 0;

    QWidget *w = fact->container( name.isNull() ? QString("screen_context_popup") : name, this );
    if ( !w )
        return 0;

    KPopupMenu *pop = static_cast<KPopupMenu *>( w );
    pop->insertTitle( i18n("kdetv"), -1, 0 );
    return pop;
}

bool KdetvPart::openFile()
{
    kdDebug() << "KdetvPart::openFile() '" << filename() << "'" << endl;

    kdDebug() << "KdetvPart::openFile(): Extend Kdetv interface to cover loading" << endl;

    return false;
}

void KdetvPart::about()
{
    KAboutApplication dlg( createAboutData() );
    dlg.exec();
}

void KdetvPart::help()
{
    KApplication::kApplication()->invokeHelp( QString::null, QString("kdetv") );
}


void KdetvPart::preferences()
{
    Kdetv *ktv = tv->driver();
    if ( ktv )
        ktv->views()->launchSettings(0L);
}

void KdetvPart::reportBug()
{
    KApplication::kApplication()->invokeMailer( QString("kwintv@mail.kde.org"), QString::null );
}
