/***************************************************************************
 *   Copyright (C) 2004, 2005, 2006 Thomas Nagy                            *
 *   Thomas Nagy <tnagyemail-com@yahoo.fr>                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation (see COPYING)            *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 ***************************************************************************/

#include <qdir.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qmap.h>
#include <qpixmap.h>

#include <kurl.h>
#include <kdebug.h>
#include <ktempdir.h>
#include <kgenericfactory.h>

#include "settings.h"
#include "datastruct/DItem.h"
#include "datastruct/DDataControl.h"
#include "datastruct/DDataItem.h"

#include "kdissprosperslides.h"

K_EXPORT_COMPONENT_FACTORY(libkdissprosperslides, kdissprosperslidesFactory) 
	QObject *kdissprosperslidesFactory::createObject (QObject *, const char *, const char *, const QStringList &) { return new kdissprosperslides; }


kdissprosperslides::kdissprosperslides() : DGenerator()
{
	m_data = NULL;
}

const QString kdissprosperslides::identifikation()
{
	return "kdissprosperslides";
}

const QString kdissprosperslides::fullName()
{
	return i18n("Prosper presentation");
}

const QString kdissprosperslides::group()
{
	return i18n("LaTeX");
}

const QString kdissprosperslides::description()
{
	return i18n("This template generates a LaTeX presentation using Prosper. You may want to check the prosper website for finding new slide styles. Most of the output is commented and does not appear in the slides. You will have to work a bit on the tex file generated, but the results are probably worth the effort :)");
}

const QString kdissprosperslides::quickstart()
{
	return i18n( "Go into the directory created and run 'make' to create a postscript file\n"
			"View the document created by running 'make view'\n"
			"A pdf document can be generated by 'make pdf' (requires ps2pdf)\n"
			"Using a LaTeX tool such as Kile is recommended for editing the generated file main.tex");
}

void kdissprosperslides::outputPixParagraph(DDataItem* item, QTextStream & s)
{
	if (item->m_pix.isNull())
		return;

	// the output is commented
	s<<"%XXX: to include the pictures, you will have to convert them to eps\n";
	s<<"%\\begin{figure}[htbp]\n";
	s<<"%\\begin{center}\n";
	s<<"%\\includegraphics[width=3cm]{pics/pic-"<<QString::number(item->Id())<<"}\n";
	s<<"%\\caption{\\footnotesize{"<<protectTex(item->m_piccaption)<<"}}\n";
	s<<"%%"<<item->m_picurl<<"\n";
	s<<"%\\end{center}\n";
	s<<"%\\end{figure}\n\n";
}

void kdissprosperslides::outputURL(DDataItem* item, QTextStream & s)
{
	for (unsigned int i=0; i<item->m_urllist.count(); i++)
	{
		QString url = item->m_urllist[i].m_url;
		QString caption = item->m_urllist[i].m_caption;

		if (!caption.length()) caption = url;

		s<<commentTex(url)<<"\n";
		s<<commentTex(caption)<<"\n";
	}
}

void kdissprosperslides::outputComment(DDataItem* item, QTextStream & s)
{
	if (item->m_comment.length()>0)
	{
		if (Settings::commentTex()) s<<item->m_comment<<"\n";
		else s<<commentTex(item->m_comment)<<"\n";
	}
}

void kdissprosperslides::outputData(DDataItem* item, QTextStream & s)
{
	//kdWarning()<<"output data for "<<item<<endl;
	s<<commentTex(htmlToTex(item->m_text))<<"\n\n";
	outputPixParagraph(item, s);
	outputURL(item, s);
	outputComment(item, s);
}

QString randomTrans()
{
	int val = random()%7;
	QString ret;
	switch (val)
	{
		case 0:
			ret = "[Split]";
			break;

		case 1:
			ret = "[Blinds]";
			break;

		case 2:
			ret = "[Box]";
			break;

		case 3:
			ret = "[Wipe]";
			break;

		case 4:	
			ret = "[Dissolve]";
			break;

		case 5:	    
			ret = "[Glitter]";
			break;

		case 6:
			ret = "[Replace]";
			break;

		default:
			break;
	}
	return ret;
}

void kdissprosperslides::writeCompleteSlide(DDataItem* item, QTextStream & s, int level)
{
	//kdWarning()<<"calling kdissprosperslides::writeCompleteSlide "<<level<<" "<<item<<endl;
	if (level == 0)
	{
		s<<"%------------------------------------------------------------------- \n";
		s<<"\\begin{slide}"<<randomTrans()<<"{"<<protectTex(item->m_summary)<<"}\n\n";
		s<<"\\begin{itemize}\n";
		outputData(item, s);

		for (unsigned int i = 0; i<item->countChildren(); i++)
		{
			writeCompleteSlide((DDataItem*) m_data->Item(item->childNum(i)), s, level+1);
		}

		s<<"\\end{itemize}\n";
		s<<"\\end{slide}\n";
		s<<"%------------------------------------------------------------------- \n";
	}
	else if (level == 1)
	{
		s<<"\\item "<<protectTex(item->m_summary)<<"\n%\n";
		outputData(item, s);

		if (item->countChildren() > 0)
		{
			s<<"\\begin{itemize}\n";
			for (unsigned int i = 0; i<item->countChildren(); i++)
			{
				writeCompleteSlide((DDataItem*) m_data->Item(item->childNum(i)), s, level+1);
			}
			s<<"\\end{itemize}\n";
		}
	}
	else if (level == 2)
	{
		s<<"\\item "<<protectTex(item->m_summary)<<"\n";
		outputData(item, s);

		if (item->countChildren() > 0)
		{
			s<<"\\begin{itemize}\n";
			for (unsigned int i = 0; i<item->countChildren(); i++)
			{
				writeCompleteSlide((DDataItem*) m_data->Item(item->childNum(i)), s, level+1);
			}
			s<<"\\end{itemize}\n";
		}
	}
	else if (level == 3)
	{
		s<<"\\item "<<protectTex(item->m_summary)<<"\n";
		outputData(item, s);

		for (unsigned int i = 0; i<item->countChildren(); i++)
		{
			writeCompleteSlide((DDataItem*) m_data->Item(item->childNum(i)), s, level+1);
		}
	}
	else if (level > 3)
	{
		// too much nesting, output in comments
		s<<commentTex(item->m_summary)<<"\n";
		outputData(item, s);

		for (unsigned int i = 0; i<item->countChildren(); i++)
		{
			writeCompleteSlide((DDataItem*) m_data->Item(item->childNum(i)), s, level+1);
		}
	}
}

void kdissprosperslides::writeItem(DDataItem* item, QTextStream & s, int level)
{
	DGenerator::writeItem(item, s, level);

	if (level == 0)
	{
		if (! item->m_pix.isNull())
		{
			s<<"% you will have to convert the picture to .eps to make this work:\n";
			s<<"%\\Logo(2,5){\\includegraphics[width=1cm]{"<<QString::number(item->Id())<<".eps}}\n";
		}

		// comment this text since it is out of the body text
		s<<"\n"<<commentTex( htmlToTex(item->m_text) )<<"\n";
		outputURL(item, s);
		outputComment(item, s);
	}

	for (unsigned int i = 0; i<item->countChildren(); i++)
	{
		DDataItem* subtree = (DDataItem*) m_data->Item(item->childNum(i));

		// 15 is the maximum maximum size of a subtree 
		// that can fit on a single slide
		if (subtree->countFamily() < 15)
		{
			writeCompleteSlide(subtree, s, 0);
		}
		else
		{
			if (level > 3)
			{
				writeCompleteSlide(subtree, s, 0);
				kdWarning()<<"Your mindmap is insane !!"<<endl;
			}

			// there are too many leaves - we then make a transition slide introducing the sub-slides
			s<<"%------------------------------------------------------------------- \n";
			s<<"%------------------------------------------------------------------- \n";

			s<<"\\begin{slide}"<<randomTrans()<<"{"<<protectTex(subtree->m_summary)<<"}\n\n";
			outputData(subtree, s);

			s<<"\\end{slide}\n";
			s<<"%------------------------------------------------------------------- \n";

			writeItem(subtree, s, level+1);

			s<<"%------------------------------------------------------------------- \n";
			s<<"%------------------------------------------------------------------- \n";
		}
	}
}

bool kdissprosperslides::writeMaindoc(DDataItem *root, const QString & path)
{
	QFile textfile(path+"/main.tex");
	if (!textfile.open(IO_WriteOnly | IO_Append) )
	{
		kdWarning()<<"could not open LaTeX file for append :-/"<<endl;
		return false;
	}

	QTextStream u(&textfile);
	if (Settings::templatesUTF8())
	{
		u.setEncoding(QTextStream::UnicodeUTF8);
		u<<"\\usepackage{ucs}\n";
		u<<"\\usepackage[utf8]{inputenc}\n\n\n";
	}
	else
	{
		u<<"\\usepackage[latin1]{inputenc}\n\n\n";
	}

	// make a nice title for the first page
	u<<"\\title{"<<protectTex(root->m_summary)<<"}\n";
	u<<"\\date{\\today}\n";
	u<<"\\author{"<<protectTex(m_data->m_fname)<<" "<<protectTex(m_data->m_sname)<<"}\n";
        if (!m_data->m_company.isEmpty())
        {
          u<<"\\institution{"<<protectTex(m_data->m_company)<<"}\n";
        }
        if (!m_data->m_email.isEmpty())
        {
          u<<"\\email{"<<protectTex(m_data->m_email)<<"}\n";
        }

	// standard headers
	u<<"\n";
	u<<"\\begin{document}\n";
	u<<"\\maketitle\n";

	// now the central part, with text, pictures, etc ..

	writeItem(root, u, 0);

	// central part is finished
	u<<"\\end{document}\n\n";
	textfile.close();

	notifyDocGenerated( path+"/main.tex" );

	return true;
}

#include "kdissprosperslides.moc"
