/*****************************************************************

   Copyright (c) 2006 Stephan Binner <binner@kde.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.

******************************************************************/

#include "search.h"

#include <kconfig.h>

KCMKerrySearch::KCMKerrySearch(QWidget *parent, const char * )
    : KCModule(parent, "kcmkerrysearch")
{
    QVBoxLayout* top_layout = new QVBoxLayout( this, KDialog::spacingHint() );

    // General
    QGroupBox* gb_general = new QGroupBox(0, Qt::Vertical, i18n("General"), this);
    gb_general->setFlat(true);
    top_layout->addWidget(gb_general);
    QVBoxLayout* gb_general_layout = new QVBoxLayout( gb_general->layout(), KDialog::spacingHint() );

    QHBoxLayout* qh_layout = new QHBoxLayout(gb_general_layout);
    QLabel* label_order = new QLabel(i18n("Default result sort order:"), gb_general);
    qh_layout->addWidget(label_order);
    combo_order = new QComboBox(gb_general);
    combo_order->insertItem(i18n("Type"));
    combo_order->insertItem(i18n("Date Modified"));
    combo_order->insertItem(i18n("Name"));
    combo_order->insertItem(i18n("Relevance"));
    QWhatsThis::add(combo_order,i18n("Define the default sort order at startup."));
    label_order->setBuddy(combo_order);
    qh_layout->addWidget(combo_order);

    QHBoxLayout* qh_number = new QHBoxLayout(gb_general_layout);
    QLabel* label_number = new QLabel(i18n( "Maximum number of results displayed:" ), gb_general);
    qh_number->addWidget(label_number);
    maxResultsDisplayed = new QSpinBox( 1, 100, 1, gb_general );
    maxResultsDisplayed->setSpecialValueText(i18n("No Limit"));
    QWhatsThis::add(maxResultsDisplayed,i18n("Define how many results shall be displayed on one result page."));
    label_number->setBuddy(maxResultsDisplayed);
    qh_number->addWidget(maxResultsDisplayed);

    showBigTiles = new QCheckBox(i18n("Show search results with details by default"), gb_general);
    gb_general_layout->addWidget(showBigTiles);

    // Global Shortcuts
    QGroupBox* gb_keys = new QGroupBox(0, Qt::Vertical, i18n("Global Shortcuts"), this);
    QVBoxLayout* gb_keys_layout = new QVBoxLayout( gb_keys->layout(), KDialog::spacingHint() );
    top_layout->addWidget(gb_keys);
    gb_keys->setFlat(true);

    globalKeys = new KGlobalAccel(this);
    globalKeys->insert( "Program:kerry", i18n("Kerry Beagle Search") );

    KShortcut showDialogShortcut = KShortcut(ALT+Key_Space);
    showDialogShortcut.append( KKey( Key_F12 ) );
    globalKeys->insert( "Show Kerry Dialog", i18n("Show Search Dialog"), QString::null, showDialogShortcut, showDialogShortcut, 0, 0);
    globalKeys->insert( "Search Primary Selection with Kerry", i18n("Search Primary Selection"), QString::null, CTRL+ALT+Key_Space, CTRL+ALT+Key_Space, 0, 0);

    KConfig *config = new KConfig("kerryrc");
    globalKeys->readSettings(config);
    delete config;

    keysWidget = new KKeyChooser( globalKeys, gb_keys );
    gb_keys_layout->addWidget(keysWidget);

    // Add some spacing at the end
    QWidget *dummy = new QWidget( this );
    top_layout->setStretchFactor( dummy, 1 );
    top_layout->addWidget(dummy);

    connect(maxResultsDisplayed, SIGNAL(valueChanged(int)), SLOT(changedValue()));
    connect(combo_order, SIGNAL(activated(int)), SLOT(changedValue()));
    connect(keysWidget, SIGNAL(keyChange()), SLOT(changedValue()));
    connect(showBigTiles, SIGNAL(clicked()), SLOT(changedValue()));

    load();
}

KCMKerrySearch::~KCMKerrySearch()
{
}

void KCMKerrySearch::changedValue()
{
    emit changed( true );
}

void KCMKerrySearch::load()
{
    load( false );
}

void KCMKerrySearch::load( bool useDefaults )
{
    if (useDefaults) {
       maxResultsDisplayed->setValue(20);
       combo_order->setCurrentItem(0);
       keysWidget->allDefault();
       showBigTiles->setChecked(false);
    }
    else {
       KConfig *config = new KConfig("kerryrc");
       config->setGroup("General");
       maxResultsDisplayed->setValue(config->readNumEntry("DisplayAmount", 20));
       combo_order->setCurrentItem(config->readNumEntry("DefaultSortOrder",0));
       showBigTiles->setChecked(config->readBoolEntry("ShowBigTiles",false));
       delete config;
    }

    emit changed( useDefaults );
}

void KCMKerrySearch::defaults()
{
     load( true );
}

void KCMKerrySearch::save()
{
    KConfig *config = new KConfig("kerryrc");
    config->setGroup("General");
    config->writeEntry("DisplayAmount", maxResultsDisplayed->value());
    config->writeEntry("DefaultSortOrder", combo_order->currentItem());
    config->writeEntry("ShowBigTiles", showBigTiles->isChecked());

    keysWidget->commitChanges();
    globalKeys->writeSettings(config);

    config->sync();
    delete config;

}

#include "search.moc"
