/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2006 by the KFTPGrabber developers
 * Copyright (C) 2003-2006 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */
#ifndef KFTPENGINESPEEDLIMITER_H
#define KFTPENGINESPEEDLIMITER_H

#include <qdatetime.h>

#include "socket.h"

namespace KFTPEngine {

/**
 * This class is used by Socket implementations to enforce speed limits for
 * uploads or downloads.
 *
 * @author Jernej Kos <kostko@jweb-network.net>
 */
class SpeedLimiter {
public:
    SpeedLimiter(Socket *socket);
    
    /**
     * Set speed limit.
     *
     * @param limit The speed limit (in bytes/sec)
     */
    void setLimit(filesize_t limit);
    
    /**
     * Check if there is a sped limit set.
     *
     * @return True if the speed limit is set, false otherwise
     */
    bool hasLimit() { return m_limit > 0; }
    
    /**
     * Reset the speed limiter.
     */
    void reset();
    
    /**
     * Update the number of bytes transfered.
     *
     * @param bytes The number of new bytes transfered
     */
    void update(filesize_t bytes) { m_transferedBytes += bytes; }
    
    /**
     * Get the number of bytes the socket is allowed to transfer.
     */
    int allowBytes();
private:
    Socket *m_socket;
    filesize_t m_limit;
    filesize_t m_transferedBytes;
    QTime m_time;
};

}

#endif
