/* $Id: we_general.c,v 1.9 2006/06/22 22:03:39 ekalin Exp $ */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <kcconfig.h>
#endif

#include <string.h>
#include <libintl.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <glade/glade.h>

#include "kildclient.h"
#include "perlscript.h"


/***********************
 * Function prototypes *
 ***********************/
static void     we_general_add_edit_char_cb(GtkWidget *widget, gpointer data);
static void     view_row_dblclick_cb(GtkTreeView        *view,
                                     GtkTreePath        *path,
                                     GtkTreeViewColumn  *col,
                                     gpointer            data);
static gboolean edit_char(World        *world,
                          GtkTreeModel *model,
                          GtkTreeIter  *iter);
static void     we_general_delete_char_cb(GtkWidget *widget, gpointer data);

static gboolean we_general_view_keypress_cb(GtkWidget   *widget,
                                            GdkEventKey *evt,
                                            gpointer     data);
static void     we_general_move_char_cb(GtkWidget *widget, gpointer data);



GtkWidget *
get_we_general_tab(GladeXML *gladexml)
{
  GtkWidget *panel;

  panel = glade_xml_get_widget(gladexml, "we_panel_general");

  return panel;
}


void
fill_we_general_tab(World *world, GladeXML *gladexml)
{
  gchar             *tmp;
  GtkWidget         *txtName;
  GtkWidget         *txtHost;
  GtkWidget         *txtPort;
  GtkWidget         *chkTLS;
  GtkWidget         *cmbConnStyle;
  GtkTreeView       *viewCharacters;
  GtkCellRenderer   *renderer;
  GtkTreeViewColumn *column;
  GtkWidget         *btnAdd;
  GtkWidget         *btnEdit;
  GtkWidget         *btnDelete;
  GtkWidget         *btnUp;
  GtkWidget         *btnDown;

  if (!world) {
    return;
  }

  txtName        = glade_xml_get_widget(gladexml, "txtName");
  txtHost        = glade_xml_get_widget(gladexml, "txtHost");
  txtPort        = glade_xml_get_widget(gladexml, "txtPort");
  chkTLS         = glade_xml_get_widget(gladexml, "chkTLS");
  cmbConnStyle   = glade_xml_get_widget(gladexml, "cmbConnStyle");
  btnAdd         = glade_xml_get_widget(gladexml, "btnCharAdd");
  btnEdit        = glade_xml_get_widget(gladexml, "btnCharEdit");
  btnDelete      = glade_xml_get_widget(gladexml, "btnCharDelete");
  btnUp          = glade_xml_get_widget(gladexml, "btnCharUp");
  btnDown        = glade_xml_get_widget(gladexml, "btnCharDown");
  viewCharacters = GTK_TREE_VIEW(glade_xml_get_widget(gladexml, "viewCharacters"));

  if (world->name) {
    gtk_entry_set_text(GTK_ENTRY(txtName), world->name);
  }
  if (world->host) {
    gtk_entry_set_text(GTK_ENTRY(txtHost), world->host);
  }
  tmp = g_strdup_printf("%d", world->port);
  gtk_entry_set_text(GTK_ENTRY(txtPort), tmp);
  g_free(tmp);
#ifdef HAVE_LIBGNUTLS
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chkTLS), world->use_tls);
#else
  gtk_widget_destroy(chkTLS);
#endif

  gtk_combo_box_set_active(GTK_COMBO_BOX(cmbConnStyle),
                           world->connection_style);

  /* Configure view */
  gtk_tree_view_set_model(viewCharacters, world->logon_characters);
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(viewCharacters,
                                              0,
                                              _("Character"),
                                              renderer,
                                              "text", LOGON_CHAR,
                                              NULL);

  renderer = gtk_cell_renderer_text_new();
  g_object_set(G_OBJECT(renderer), "text", "******", NULL);
  column = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(column, _("Password"));
  gtk_tree_view_column_pack_start(column, renderer, TRUE);
  gtk_tree_view_append_column(viewCharacters, column);
  g_signal_connect(G_OBJECT(viewCharacters), "row-activated",
                   G_CALLBACK(view_row_dblclick_cb), world);
  g_signal_connect(G_OBJECT(viewCharacters), "key-press-event",
                   G_CALLBACK(we_general_view_keypress_cb), world);

  /* Button signals */
  g_object_set_data(G_OBJECT(btnAdd), "isnew", GINT_TO_POINTER(TRUE));
  g_signal_connect(G_OBJECT(btnAdd), "clicked",
                   G_CALLBACK(we_general_add_edit_char_cb), world);
  g_object_set_data(G_OBJECT(btnEdit), "isnew", GINT_TO_POINTER(FALSE));
  g_signal_connect(G_OBJECT(btnEdit), "clicked",
                   G_CALLBACK(we_general_add_edit_char_cb), world);
  g_signal_connect(G_OBJECT(btnDelete), "clicked",
                   G_CALLBACK(we_general_delete_char_cb), world);
  g_object_set_data(G_OBJECT(btnUp), "isup", GINT_TO_POINTER(TRUE));
  g_signal_connect(G_OBJECT(btnUp), "clicked",
                   G_CALLBACK(we_general_move_char_cb), viewCharacters);
  g_object_set_data(G_OBJECT(btnDown), "isup", GINT_TO_POINTER(FALSE));
  g_signal_connect(G_OBJECT(btnDown), "clicked",
                   G_CALLBACK(we_general_move_char_cb), viewCharacters);
}


void
update_we_general_parameters(World *world)
{
  const gchar *newval;
  int          newport;
  GladeXML    *gladexml;
  GtkWidget   *txtName;
  GtkWidget   *txtHost;
  GtkWidget   *txtPort;
  GtkWidget   *cmbConnStyle;
#ifdef HAVE_LIBGNUTLS
  GtkWidget   *chkTLS;
#endif

  gladexml     = glade_get_widget_tree(world->dlgEditWorld);
  txtName      = glade_xml_get_widget(gladexml, "txtName");
  txtHost      = glade_xml_get_widget(gladexml, "txtHost");
  txtPort      = glade_xml_get_widget(gladexml, "txtPort");
  cmbConnStyle = glade_xml_get_widget(gladexml, "cmbConnStyle");

  newval = gtk_entry_get_text(GTK_ENTRY(txtName));
  if (!world->name || strcmp(world->name, newval) != 0) {
    g_free(world->name);
    world->name = g_strdup(newval);
    world->new_text_str = g_strdup_printf("<span foreground=\"red\">%s</span>",
                                          newval);
  }

  newval = gtk_entry_get_text(GTK_ENTRY(txtHost));
  if (!world->host || strcmp(world->host, newval) != 0) {
    g_free(world->host);
    world->host = g_strdup(newval);
  }

  newval = gtk_entry_get_text(GTK_ENTRY(txtPort));
  newport = atoi(newval);
  if (world->port != newport) {
    world->port = newport;
  }

  world->connection_style = gtk_combo_box_get_active(GTK_COMBO_BOX(cmbConnStyle));

#ifdef HAVE_LIBGNUTLS
  chkTLS = glade_xml_get_widget(gladexml, "chkTLS");
  world->use_tls = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chkTLS));
#endif
}


static
void
we_general_add_edit_char_cb(GtkWidget *widget, gpointer data)
{
  World            *world = (World *) data;
  GladeXML         *gladexml;
  gboolean          is_new;
  GtkTreeView      *view;
  GtkTreeModel     *model;
  GtkTreeSelection *selection;
  GtkTreeIter       iter;

  gladexml = glade_get_widget_tree(widget);
  view = GTK_TREE_VIEW(glade_xml_get_widget(gladexml, "viewCharacters"));
  model = gtk_tree_view_get_model(view);

  is_new = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "isnew"));
  if (is_new) {
    gtk_list_store_append(GTK_LIST_STORE(model), &iter);
  } else {
    selection = gtk_tree_view_get_selection(view);
    if (!gtk_tree_selection_get_selected(selection, NULL, &iter)) {
      return;
    }
  }

  if (!edit_char(world, model, &iter) && is_new) {
    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
  }
}


static
void
view_row_dblclick_cb(GtkTreeView        *view,
                     GtkTreePath        *path,
                     GtkTreeViewColumn  *col,
                     gpointer            data)
{
  World        *world;
  GtkTreeModel *model;
  GtkTreeIter   iter;

  world = (World *) data;
  model = gtk_tree_view_get_model(view);

  if (gtk_tree_model_get_iter(model, &iter, path)) {
    edit_char(world, model, &iter);
  }
}


static
gboolean
edit_char(World *world, GtkTreeModel *model, GtkTreeIter *iter)
{
  GladeXML    *gladexml;
  GtkWidget   *dlgEditChar;
  GtkEntry    *txtName;
  GtkEntry    *txtPassword;
  gchar       *name = NULL;
  gchar       *password = NULL;
  const gchar *newname;
  const gchar *newpassword;

  gladexml = glade_xml_new(get_kildclient_installed_file("kildclient.glade"),
                           "dlgEditChar", NULL);
  dlgEditChar  = glade_xml_get_widget(gladexml, "dlgEditChar");
  gtk_dialog_set_default_response(GTK_DIALOG(dlgEditChar), GTK_RESPONSE_OK);
  txtName     = (GtkEntry *) glade_xml_get_widget(gladexml,
                                                  "txtCharName");
  txtPassword = (GtkEntry *) glade_xml_get_widget(gladexml,
                                                  "txtCharPassword");

  /* We don't need it anymore */
  g_object_unref(gladexml);

  /* Fill-in values */
  gtk_tree_model_get(model, iter,
                     LOGON_CHAR, &name,
                     LOGON_PASS, &password,
                     -1);
  if (name) {
    gtk_entry_set_text(txtName, name);
    g_free(name);
  }
  if (password) {
    gtk_entry_set_text(txtPassword, password);
    g_free(password);
  }

  /* Run the dialog until the input is valid or cancelled */
  gtk_widget_show_all(dlgEditChar);
  while (1) {
    if (gtk_dialog_run(GTK_DIALOG(dlgEditChar)) == GTK_RESPONSE_OK) {
      /* Validate */
      newname = gtk_entry_get_text(txtName);
      if (strcmp(newname, "") == 0) {
        GtkWidget *dlg = gtk_message_dialog_new(GTK_WINDOW(dlgEditChar),
                                                GTK_DIALOG_MODAL,
                                                GTK_MESSAGE_ERROR,
                                                GTK_BUTTONS_OK,
                                                _("You must specify the name."));
        gtk_dialog_run(GTK_DIALOG(dlg));
        gtk_widget_destroy(dlg);
        continue;
      }

      newpassword = gtk_entry_get_text(txtPassword);
      if (strcmp(newpassword, "") == 0) {
        GtkWidget *dlg = gtk_message_dialog_new(GTK_WINDOW(dlgEditChar),
                                                GTK_DIALOG_MODAL,
                                                GTK_MESSAGE_ERROR,
                                                GTK_BUTTONS_OK,
                                                _("You must specify the password."));
        gtk_dialog_run(GTK_DIALOG(dlg));
        gtk_widget_destroy(dlg);
        continue;
      }

      /* Update values */
      gtk_list_store_set(GTK_LIST_STORE(model), iter,
                         LOGON_CHAR, newname,
                         LOGON_PASS, newpassword,
                         -1);

      /* We've finished successfully */
      gtk_widget_destroy(dlgEditChar);
      return TRUE;
    } else {
      /* Cancel pressed */
      gtk_widget_destroy(dlgEditChar);
      return FALSE;
    }
  }
}


static
void
we_general_delete_char_cb(GtkWidget *widget, gpointer data)
{
  World            *world = (World *) data;
  GladeXML         *gladexml;
  GtkTreeView      *view;
  GtkTreeModel     *model;
  GtkTreeSelection *selection;
  GtkTreeIter       iter;

  gladexml = glade_get_widget_tree(widget);
  view = GTK_TREE_VIEW(glade_xml_get_widget(gladexml, "viewCharacters"));
  selection = gtk_tree_view_get_selection(view);
  if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
    GtkWidget *msgdlg;
    GtkWidget *dlgbutton;
    GtkWidget *dlgbuttonimage;

    msgdlg = gtk_message_dialog_new(GTK_WINDOW(world->dlgEditWorld),
                                    GTK_DIALOG_MODAL,
                                    GTK_MESSAGE_QUESTION,
                                    GTK_BUTTONS_NONE,
                                    _("Do you really want to delete this character information?"));
    dlgbutton = gtk_button_new_with_label(_("Do not delete"));
    dlgbuttonimage = gtk_image_new_from_stock(GTK_STOCK_NO,
                                              GTK_ICON_SIZE_MENU);
    gtk_button_set_image(GTK_BUTTON(dlgbutton), dlgbuttonimage);
    gtk_widget_show_all(dlgbutton);
    gtk_dialog_add_action_widget(GTK_DIALOG(msgdlg),
                                 dlgbutton, GTK_RESPONSE_NO);

    dlgbutton = gtk_button_new_with_label(_("Delete"));
    dlgbuttonimage = gtk_image_new_from_stock(GTK_STOCK_DELETE,
                                              GTK_ICON_SIZE_MENU);
    gtk_button_set_image(GTK_BUTTON(dlgbutton), dlgbuttonimage);
    gtk_widget_show_all(dlgbutton);
    gtk_dialog_add_action_widget(GTK_DIALOG(msgdlg),
                                 dlgbutton, GTK_RESPONSE_YES);


    if (!world->confirm_delete
        || gtk_dialog_run(GTK_DIALOG(msgdlg)) == GTK_RESPONSE_YES) {
      gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
    }
    gtk_widget_destroy(msgdlg);
  }
}


static
gboolean
we_general_view_keypress_cb(GtkWidget   *widget,
                            GdkEventKey *evt,
                            gpointer     data)
{
  if (evt->keyval == GDK_Delete) {
    we_general_delete_char_cb(widget, data);
    return TRUE;
  }

  return FALSE;
}


static
void
we_general_move_char_cb(GtkWidget *widget, gpointer data)
{
  GtkTreeView      *view;
  GtkTreeModel     *model;
  GtkTreeSelection *selection;
  GtkTreeIter       iter;
  GtkTreeIter       other_iter;
  gboolean          is_up;

  view = (GtkTreeView *) data;
  selection = gtk_tree_view_get_selection(view);
  is_up = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "isup"));
  if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
    if (is_up) {
      GtkTreePath *path;

      path = gtk_tree_model_get_path(model, &iter);
      if (gtk_tree_path_prev(path)) {
        gtk_tree_model_get_iter(model, &other_iter, path);
        gtk_list_store_swap(GTK_LIST_STORE(model), &iter, &other_iter);
      }
      gtk_tree_path_free(path);
    } else {
      other_iter = iter;
      if (gtk_tree_model_iter_next(model, &other_iter)) {
        gtk_list_store_swap(GTK_LIST_STORE(model), &iter, &other_iter);
      }
    }
  }
}
