/* $Id: multilinesend.c 1027 2009-02-12 18:27:11Z ekalin $ */

/*
 * Copyright (C) 2004-2009 Eduardo M Kalinowski <eduardo@kalinowski.com.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <kcconfig.h>
#endif

#include <string.h>
#include <libintl.h>
#include <locale.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <gmodule.h>

#include "kildclient.h"
#include "perlscript.h"



/***********************
 * Function prototypes *
 ***********************/
static GtkWidget *create_multi_line_send_dialog(GtkWindow *parent,
                                                World *world);
static gboolean   multi_line_send_timer_cb(gpointer data);
static gchar     *get_text_buffer_line(GtkTextBuffer *buffer,
                                       int line,
                                       int nlines,
                                       gboolean *last);
static gchar     *get_file_line(FILE *fp, gboolean *last);
static void       send_line(gchar *line, MLSendData  *context);
static void       string_free(gpointer string, gpointer user_data);
static void       start_multi_line_send_cb(GtkButton *btn, gpointer data);
/* Glade signals */
G_MODULE_EXPORT void multi_line_send_cb(GtkWidget *widget, gpointer data);
G_MODULE_EXPORT void clear_multi_line_send_dialog_cb(GtkWidget *widget,
                                                     gpointer   data);



void
multi_line_send_cb(GtkWidget *widget, gpointer data)
{
  if (!currentWorld->dlgMLSend) {
    currentWorld->dlgMLSend
      = create_multi_line_send_dialog(GTK_WINDOW(wndMain), currentWorld);
  }

  gtk_widget_show_all(currentWorld->dlgMLSend);
  gtk_window_present(GTK_WINDOW(currentWorld->dlgMLSend));
}


static
GtkWidget *
create_multi_line_send_dialog(GtkWindow *parent, World *world)
{
  GladeXML  *gladexml;
  GtkWidget *dlg;
  GtkWidget *text;
  GtkWidget *txtInitial;
  GtkWidget *txtFile;
  GtkWidget *btnClearFile;
  GtkWidget *btnSend;
  GtkWidget *spnDelay;
  GtkWidget *spnLinesATime;

  PangoFontDescription *fontDesc;
  PangoLayout          *layout;
  gint                  char_width;
  PangoRectangle        logical_rect;
  GtkRequisition        requisition;

  /* Create the dialog */
  gladexml = glade_xml_new(get_kildclient_installed_file("kildclient.glade"),
                           "dlgMLSend", NULL);
  dlg      = glade_xml_get_widget(gladexml, "dlgMLSend");

  /* Set the fonts */
  fontDesc = pango_font_description_from_string(world->entryfont);

  txtInitial = glade_xml_get_widget(gladexml, "txtInitial");
  gtk_widget_modify_font(txtInitial, fontDesc);

  text = glade_xml_get_widget(gladexml, "txtFinal");
  gtk_widget_modify_font(text, fontDesc);

  text  = glade_xml_get_widget(gladexml, "txtStart");
  gtk_widget_modify_font(text, fontDesc);

  text    = glade_xml_get_widget(gladexml, "txtEnd");
  gtk_widget_modify_font(text, fontDesc);

  pango_font_description_free(fontDesc);

  /* Connect signals */
  glade_xml_signal_autoconnect(gladexml);

  btnSend = glade_xml_get_widget(gladexml, "btnMLSend");
  g_signal_connect(G_OBJECT(btnSend), "clicked",
                   G_CALLBACK(start_multi_line_send_cb), world);

  txtFile       = glade_xml_get_widget(gladexml, "txtFile");
  btnClearFile = glade_xml_get_widget(gladexml, "btnClearFile");
  g_signal_connect_swapped(G_OBJECT(btnClearFile), "clicked",
                           G_CALLBACK(gtk_file_chooser_unselect_all), txtFile);

  /* Default values */
  spnDelay = glade_xml_get_widget(gladexml, "spnMLSDelay");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spnDelay),
                            globalPrefs.multi_cmd_delay);
  spnLinesATime = glade_xml_get_widget(gladexml, "spnLinesATime");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spnLinesATime),
                            globalPrefs.multi_cmd_group_size);

  /* Set width of entries to be slightly larger than 80 columns. */
  layout = gtk_widget_create_pango_layout(txtInitial, "W");
  pango_layout_get_pixel_extents(layout, NULL, &logical_rect);
  char_width = logical_rect.width;
  /* This hack allows the initial size to be set, but without preventing
     the window from getting smaller. */
  gtk_widget_set_size_request(txtInitial, 81 * char_width, -1);
  gtk_widget_size_request(dlg, &requisition);
  gtk_widget_set_size_request(txtInitial, -1, -1);
  gtk_window_resize(GTK_WINDOW(dlg), requisition.width, requisition.height);

  g_object_unref(layout);

  return dlg;
}


void
start_multi_line_send_cb(GtkButton *btn, gpointer data)
{
  World      *world = (World *) data;
  GladeXML   *gladexml;
  GtkWidget  *txtInitial;
  GtkWidget  *txtFinal;
  GtkWidget  *txtFile;
  GtkWidget  *txtStart;
  GtkWidget  *txtEnd;
  GtkWidget  *spnDelay;
  GtkWidget  *spnLinesATime;
  gdouble     delay;
  GtkWidget  *chkDontClose;
  MLSendData *mlcontext;

  gladexml = glade_get_widget_tree(GTK_WIDGET(btn));

  txtInitial    = glade_xml_get_widget(gladexml, "txtInitial");
  txtFinal      = glade_xml_get_widget(gladexml, "txtFinal");
  txtFile       = glade_xml_get_widget(gladexml, "txtFile");
  txtStart      = glade_xml_get_widget(gladexml, "txtStart");
  txtEnd        = glade_xml_get_widget(gladexml, "txtEnd");
  spnDelay      = glade_xml_get_widget(gladexml, "spnMLSDelay");
  spnLinesATime = glade_xml_get_widget(gladexml, "spnLinesATime");
  chkDontClose  = glade_xml_get_widget(gladexml, "chkDontClose");

  delay = gtk_spin_button_get_value(GTK_SPIN_BUTTON(spnDelay));

  mlcontext = g_new0(MLSendData, 1);
  mlcontext->state       = ML_TRY_INITIALBUFFER;
  mlcontext->world       = world;
  mlcontext->textInitial = gtk_text_view_get_buffer(GTK_TEXT_VIEW(txtInitial));
  mlcontext->textFinal   = gtk_text_view_get_buffer(GTK_TEXT_VIEW(txtFinal));
  mlcontext->file        = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(txtFile));
  mlcontext->textStart   = g_strdup(gtk_entry_get_text(GTK_ENTRY(txtStart)));
  mlcontext->textEnd     = g_strdup(gtk_entry_get_text(GTK_ENTRY(txtEnd)));
  mlcontext->linesatime  =
    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spnLinesATime));
  mlcontext->dont_close
    = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chkDontClose));

  if (!mlcontext->dont_close) {
    gtk_widget_hide(world->dlgMLSend);
  }

  do_multi_line_send(mlcontext, delay);
}


void
do_multi_line_send(MLSendData *mlcontext, gdouble delay)
{
  /* Send the first line immediately */
  if (multi_line_send_timer_cb(mlcontext)) {
    if (delay) {
      g_timeout_add_full(G_PRIORITY_HIGH,
                         delay * 1000,
                         multi_line_send_timer_cb,
                         mlcontext,
                         NULL);
    } else {    /* If delay is zero, send as fast as possible */
      while (multi_line_send_timer_cb(mlcontext))
        ;
    }
  }
}


static
gboolean
multi_line_send_timer_cb(gpointer data)
{
  MLSendData  *context = (MLSendData *) data;
  int          last = FALSE;
  FILE        *fp;
  gchar       *line;
  int          sent_lines;

  sent_lines = 0;
  while (sent_lines < context->linesatime) {
    switch (context->state) {
    case ML_TRY_INITIALBUFFER:
      if (context->textInitial
          && gtk_text_buffer_get_char_count(context->textInitial)) {
        context->state = ML_INITIALBUFFER;
        context->nlines = gtk_text_buffer_get_line_count(context->textInitial);
        context->currline = 0;
      } else {
        context->state = ML_TRY_INITIALLINES;
      }
      break;

    case ML_TRY_INITIALLINES:
      if (context->linesInitial) {
        context->state  = ML_INITIALLINES;
        context->list_iterator = context->linesInitial;
      } else {
        context->state = ML_TRY_FILE;
      }
      break;

    case ML_TRY_FILE:
      if (context->file) {
        fp = fopen(context->file, "r");
        if (fp) {
          context->state = ML_FILECONTENTS;
          context->fp = fp;
          break;
        }
      }
      context->state = ML_TRY_FINALBUFFER;
      break;

    case ML_TRY_FINALBUFFER:
      if (context->textFinal
          && gtk_text_buffer_get_char_count(context->textFinal)) {
        context->state = ML_FINALBUFFER;
        context->nlines = gtk_text_buffer_get_line_count(context->textFinal);
        context->currline = 0;
      } else {
        context->state = ML_TRY_FINALLINES;
      }
      break;

    case ML_TRY_FINALLINES:
      if (context->linesFinal) {
        context->state = ML_FINALLINES;
        context->list_iterator = context->linesFinal;
      } else {
        context->state = ML_FINISH;
      }
      break;

    case ML_INITIALBUFFER:
      line = get_text_buffer_line(context->textInitial,
                                  context->currline,
                                  context->nlines,
                                  &last);
      send_line(line, context);
      g_free(line);
      ++sent_lines;
      if (last) {
        context->state = ML_TRY_FILE;
      } else {
        ++context->currline;
      }
      break;

    case ML_FINALBUFFER:
      line = get_text_buffer_line(context->textFinal,
                                  context->currline,
                                  context->nlines,
                                  &last);
      send_line(line, context);
      g_free(line);
      ++sent_lines;
      if (last) {
        context->state = ML_FINISH;
      } else {
        ++context->currline;
      }
      break;

    case ML_INITIALLINES:
      line = (gchar *) context->list_iterator->data;
      send_line(line, context);
      ++sent_lines;
      context->list_iterator = context->list_iterator->next;
      if (!context->list_iterator) {
        context->state = ML_TRY_FILE;
      }
      break;

    case ML_FINALLINES:
      line = (gchar *) context->list_iterator->data;
      send_line(line, context);
      ++sent_lines;
      context->list_iterator = context->list_iterator->next;
      if (!context->list_iterator) {
        context->state = ML_FINISH;
      }
      break;

    case ML_FILECONTENTS:
      line = get_file_line(context->fp, &last);
      send_line(line, context);
      ++sent_lines;
      if (last) {
        context->state = ML_TRY_FINALBUFFER;
        fclose(context->fp);
      }
      break;

    case ML_FINISH:
      if (context->textInitial && !context->dont_close) {
        clear_multi_line_send_dialog_cb(GTK_WIDGET(context->world->dlgMLSend),
                                        NULL);
      }
      g_free(context->textStart);
      g_free(context->textEnd);
      g_free(context->file);
      if (context->linesInitial) {
        g_slist_foreach(context->linesInitial, string_free, NULL);
        g_slist_free(context->linesInitial);
      }
      if (context->linesFinal) {
        g_slist_foreach(context->linesFinal, string_free, NULL);
        g_slist_free(context->linesFinal);
      }
      g_free(context);
      return FALSE;
    }
  }

  return TRUE;
}


void
clear_multi_line_send_dialog_cb(GtkWidget *widget, gpointer data)
{
  GladeXML      *gladexml;
  GtkWidget     *dlgMLSend;
  GtkWidget     *txtInitial;
  GtkWidget     *txtFile;
  GtkWidget     *txtFinal;
  GtkWidget     *txtStart;
  GtkWidget     *txtEnd;
  GtkTextBuffer *buffer;

  gladexml   = glade_get_widget_tree(widget);
  dlgMLSend  = glade_xml_get_widget(gladexml, "dlgMLSend");
  txtInitial = glade_xml_get_widget(gladexml, "txtInitial");
  txtFile    = glade_xml_get_widget(gladexml, "txtFile");
  txtFinal   = glade_xml_get_widget(gladexml, "txtFinal");
  txtStart   = glade_xml_get_widget(gladexml, "txtStart");
  txtEnd     = glade_xml_get_widget(gladexml, "txtEnd");

  buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(txtInitial));
  gtk_text_buffer_set_text(buffer, "", 0);
  gtk_file_chooser_unselect_all(GTK_FILE_CHOOSER(txtFile));
  buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(txtFinal));
  gtk_text_buffer_set_text(buffer, "", 0);
  gtk_entry_set_text(GTK_ENTRY(txtStart), "");
  gtk_entry_set_text(GTK_ENTRY(txtEnd), "");

  gtk_widget_hide(dlgMLSend);
}


static
gchar *
get_text_buffer_line(GtkTextBuffer *buffer,
                     int line,
                     int nlines,
                     gboolean *last)
{
  GtkTextIter  start;
  GtkTextIter  end;
  gchar       *to_send;

  gtk_text_buffer_get_iter_at_line(buffer, &start, line);
  if (line == nlines - 1) {
    gtk_text_buffer_get_end_iter(buffer, &end);
    *last = TRUE;
  } else {
    gtk_text_buffer_get_iter_at_line(buffer, &end, line+1);
    *last = FALSE;
  }
  to_send = gtk_text_buffer_get_text(buffer, &start, &end, FALSE);

  return to_send;
}


static
gchar *
get_file_line(FILE *fp, gboolean *last)
{
  static char line[MAX_BUFFER];
  int         c;

  fgets(line, MAX_BUFFER, fp);
  c = fgetc(fp);
  if (c != -1) {
    ungetc(c, fp);
    *last = FALSE;
  } else {
    *last = TRUE;
  }

  return line;
}


static
void
send_line(gchar *line, MLSendData *context)
{
  int len;

  /* Remove final newline */
  len = strlen(line);
  if (line[len-1] == '\n') {
    line[len - 1] = '\0';
    --len;
    if (line[len-1] == '\r') {
      line[len - 1] = '\0';
      --len;
    }
  }

  send_to_world_no_check(context->world,
                         context->textStart, strlen(context->textStart),
                         NULL, 0,
                         FALSE,
                         FALSE);
  send_to_world_no_check(context->world,
                         line, len,
                         NULL, 0,
                         FALSE,
                         FALSE);
  send_to_world_no_check(context->world,
                         context->textEnd, strlen(context->textEnd),
                         NULL, 0,
                         TRUE,
                         FALSE);
}


static
void
string_free(gpointer string, gpointer user_data)
{
  g_free(string);
}
