/******************************************************************************
                                   chat.cpp
                              -------------------
  begin                : Mon May 8 2000
  copyright            : (C) 2000 by Benjamin Meyer
  email                : ben@meyerhome.net
 ******************************************************************************/

#include "chat.h"
#include "kinkattalinkdialog.h"
#include "smiliespopup.h"
#include "kinkattatextbrowser.h"
#include "chatplugin.h"

#include <klocale.h>
#include <kstdaction.h>
#include <kaction.h>
#include <kwin.h>
#include <kconfig.h>

#include <qstylesheet.h>
#include <qtimer.h>
#include <kfiledialog.h>
#include <kprinter.h>
#include <kpopupmenu.h>
#include <qmsgbox.h>
#include <qdatetime.h>
#include <qclipboard.h>
#include <kcolordialog.h>
#include <kapp.h>
#include <qpixmap.h>
#include <qpalette.h>
#include <qpainter.h>
#include <qregexp.h>
#include <qaccel.h>
#include <qpaintdevicemetrics.h>
#include <qfont.h>
#include <qsimplerichtext.h>
#include <kiconloader.h>
#include <qsplitter.h>
#include <qmultilineedit.h>
#include <qcolor.h>
#include <qfile.h>
#include <qtextstream.h>

#define APP_NAME                    "Kinkatta"
#define APP_LOWERCASE_NAME          "kinkatta"

#define DEFAULT_POINT_SIZE 12

/**
 * Initialize settings, set caption
 * @param userName the name of the person controlling this app.
 * @param contactsName the person userName is talking too.
 * @param parent the parent widget for this window.
 * @param alias contactsName's alias.
 */
Chat::Chat(QString userName, QString contactsName, QWidget *parent,
           QString alias) : KMainWindow( parent,contactsName.latin1() ) {
  resize(330,330);
  myName = userName;
  contactName = contactsName;
  contactAlias = alias;

  chatWindow = NULL;
  shiftIsPressed = false;
  fontSize = 3;
  flashChoice = false;
  cleared = false;
  temp_ContactBgColor.setRgb(255,255,255);
  temp_ContactTextColor.setRgb(0,0,0);
  lastMessage = "";
  currentMessageOut = "";
  chat_log = false;
  chat_sound = false;
 	
  if (alias.length() > 0)
    this->setCaption(contactAlias + " (" + contactName + ")");
  else
    this->setCaption(contactName);

  // File
  log = new KToggleAction(i18n("&Log"), CTRL+Key_L, this, SLOT(slotChatLog()), actionCollection(), "file_log");
  sound = new KToggleAction(i18n("Sound"), 0, this, SLOT(slotChatSound()), actionCollection(), "file_sound");
  (void)new KAction(i18n("&Clear"), CTRL+Key_D, this, SLOT(slotChatClear()), actionCollection(), "file_clear");
  (void)new KAction(i18n("&SaveAs..."), 0, this, SLOT(slotChatSaveAs()), actionCollection(), "chatFile_saveAs" );

  KStdAction::print(this, SLOT(slotChatPrint()), actionCollection(), "chatFile_print" );
  KAction *action = KStdAction::close(this, SLOT(slotChatClose()), actionCollection(), "chatFile_close" );
  action->setAccel(Key_Escape); 

  // Edit
  undo = KStdAction::undo(this, SLOT(slotEditUndo()), actionCollection(), "chatEdit_undo" );
  undo->setEnabled(false);
  redo = KStdAction::redo(this, SLOT(slotEditRedo()), actionCollection(), "chatEdit_redo" );
  redo->setEnabled(false);
  KStdAction::cut(this, SLOT(slotEditCut()), actionCollection(), "chatEdit_cut" );
  KStdAction::copy(this, SLOT(slotEditCopy()), actionCollection(), "chatEdit_copy" );
  KStdAction::paste(this, SLOT(slotEditPaste()), actionCollection(), "chatEdit_paste" );
  KStdAction::selectAll(this, SLOT(slotEditSelectAll()), actionCollection(), "chatEdit_selectAll" );
  KStdAction::configureToolbars(this, SLOT(options_configuretoolbars() ), actionCollection(), "configuretoolbars");

  // View
  time = new KToggleAction(i18n("T&ime Stamp"),  Key_F2, this, SLOT(slotViewTimeStamp()), actionCollection(), "view_time");
  viewTextToolBar = new KToggleAction( i18n( "Show &Text Toolbar" ), 0, this, SLOT( slotViewTextToolBar() ), actionCollection(), "view_text" );
  viewChatToolBar = new KToggleAction( i18n( "Show &Chat Toolbar" ), 0, this, SLOT( slotViewChatToolBar() ), actionCollection(), "view_chat" );
  viewPluginToolBar = new KToggleAction( i18n( "Show &Plugin Toolbar" ), 0, this, SLOT( slotViewPluginToolBar() ), actionCollection(), "view_plugin" );
  
  // Insert
  KActionMenu *smilies = new KActionMenu( i18n( "&Smilies" ), actionCollection(), "smilies" );
  Smilies *smileyPopup = new Smilies(smilies->popupMenu(), "smileyPopup");
  connect(smileyPopup, SIGNAL(smileyButtonClicked(int)), this, SLOT(slotInsertFace(int)));
  smilies->popupMenu()->insertItem(smileyPopup, 1, 1);

  bold = new KAction(i18n("&Bold"), CTRL+Key_B, this, SLOT(slotInsertBold()), actionCollection(), "insert_bold");
  italic = new KAction(i18n("&Italic"), CTRL+Key_I, this, SLOT(slotInsertItalic()), actionCollection(), "insert_italic");
  (void)new KAction(i18n("&Underline"), CTRL+Key_U, this, SLOT(slotInsertUnderline()), actionCollection(), "insert_underline");
  (void)new KAction(i18n("S&hrink Font"), CTRL+Key_Minus, this, SLOT(slotInsertShrinkFont()), actionCollection(), "insert_shrink");
  (void)new KAction(i18n("&Normal Font"), 0, this, SLOT(slotInsertNormalFont()), actionCollection(), "insert_normal");
  (void)new KAction(i18n("In&crease Font"), CTRL+Key_Plus, this, SLOT(slotInsertIncreaseFont()), actionCollection(), "insert_increase");
  (void)new KAction(i18n("B&ackground Color..."), 0, this, SLOT(slotInsertBackGroundColor()), actionCollection(), "insert_background");
  (void)new KAction(i18n("&Pen Color..."), 0, this, SLOT(slotInsertPenColor()), actionCollection(), "insert_pen");
  (void)new KAction(i18n("&Link..."), CTRL+Key_K, this, SLOT(slotInsertLink()), actionCollection(), "insert_link");
  (void)new KAction(i18n("L&ast Message"), Key_Insert, this, SLOT(slotInsertLastMessage()), actionCollection(), "insert_last");
  
  (void)new KAction(i18n("&Send Message"), CTRL+Key_S, this, SLOT(slotContactSendMessage()), actionCollection(), "contact_sendMessage");
 
  default_yourFont = true;
  default_contactFont = true;
}

/**
 * Close the log if it is open.
 */
Chat::~Chat(){
  if ( chat_log == true ){
    // End log file and close it.
    if (chat_log_inHtml == true)
      logStream << "</BODY>\n</HTML>";
    closeLogFile(false);
  }

  // Save settings.
  KConfig &config = *KGlobal::config();
  toolBar("Chat ToolBar")->saveSettings(&config, "Chat Toolbar");
  toolBar("Text ToolBar")->saveSettings(&config, "Text Toolbar");
  toolBar("Plugin ToolBar")->saveSettings(&config, "Plugin Toolbar");
}

/**
 * Load the new chat settings
 */
void Chat::updateChatSettings(){
  // Get settings.
  KConfig &config = *KGlobal::config();
  config.setGroup("chatwindow");
  chat_raise_window = config.readBoolEntry("chat_raise_window", true);
  chat_flash_on_new_message = config.readBoolEntry("chat_flash_on_new_message", true);
  bool t_chat_log = config.readBoolEntry("chat_log", true);
  chat_log_inHtml = config.readBoolEntry("chat_log_inHtml", true);
  chat_show_time = config.readBoolEntry("chat_show_time", true);
  chat_log_time = config.readBoolEntry("chat_log_time", true);
  chat_return_send = config.readBoolEntry("chat_return_send", true);
  chat_sreturn_send = config.readBoolEntry("chat_sreturn_send", false);
  chat_send_blank = config.readBoolEntry("chat_send_blank", false);
  chat_ignore_contacthtml = config.readBoolEntry("chat_ignore_contacthtml", false);
  chat_line_limit = config.readBoolEntry("chat_line_limit", false);
  chat_show_smilies = config.readBoolEntry("chat_show_smilies", true);
  chat_line_limit_number = config.readBoolEntry("chat_line_limit_number", 500);
  chat_log_singleFile = config.readBoolEntry("chat_log_singleFile", false);

  if( chat_log != t_chat_log ){
    slotChatLog();
  }
  if (chat_show_time != chat_show_time)
    slotViewTimeStamp();
  
  toolBar("Chat ToolBar")->applySettings(&config, "Chat Toolbar");
  toolBar("Text ToolBar")->applySettings(&config, "Text Toolbar");
  toolBar("Plugin ToolBar")->applySettings(&config, "Plugin Toolbar");

  // Get font settings.
  QFont defaultFont = qApp->font();
  defaultFont.setPointSize(DEFAULT_POINT_SIZE);
  chat_yourFont = config.readFontEntry( "chat_yourFont", &defaultFont );
  chat_contactFont = config.readFontEntry( "chat_contactFont", &defaultFont );

  // Must show before you can see if something is visible...
  this->show();
  // The toolbar checkbox's
  KToolBar *toolbar = toolBar("Text ToolBar");
  if(toolbar->isVisible())
    viewTextToolBar->setChecked(true);
  else
    viewTextToolBar->setChecked(false);

  toolbar = toolBar("Chat ToolBar");
  if(toolbar->isVisible())
    viewChatToolBar->setChecked(true);
  else
    viewChatToolBar->setChecked(false);

  toolbar = toolBar("Plugin ToolBar");
  if(toolbar->isVisible())
    viewPluginToolBar->setChecked(true);
  else
    viewPluginToolBar->setChecked(false);

  // Color's
  updateChatColorSettings();

  // Sound's
  updateChatSoundSettings();

}

/**
 * Get the toolbar where plugins toolbars stuff should go.
 * @return the plugin toolbar.
 */
KToolBar *Chat::getPluginToolbar()  {
  return toolBar("Plugin ToolBar");
}

/**
 * Find out who is the person that this window is chattting with.
 * @return the person whith which we are chatting with.
 */
QString Chat::getContactName() const {
  return contactName;
}

/**
 * The contact for this window signed off and so somthing should be
 * written in the log.
 */
void Chat::contactSignedOff(){
  QString CurrentTime = QTime::currentTime().toString();
  QString message = QString("<B>") + contactName + " Signed off at: " + CurrentTime + "</B>\n";
  postMessage(message);
}

/**
 * The contact for this window signed on and so somthing is written in the log.
 */
void Chat::contactSignedOn(){
  QString CurrentTime = QTime::currentTime().toString();
  QString message = QString("<B>") + contactName + " Signed on at: " + CurrentTime + "</B>\n";
  postMessage(message);
}

/**
 * From another part of the system a message is being sent from you.
 * This is usefull for such things as pounces and automated systems that you
 * want the message to look like it came from this window.
 * @param message the message that you wish to send out.
 */
void Chat::sendOutAMessage( QString message ){
  inputWindow->setText( message );
  slotContactSendMessage();
}

/**
 * Posts a message directly to the main window without any pre-processing.
 * Logging options still apply.
 * @param message the message to post.
 */
void Chat::postMessage( QString message ){
  if (postNumber == 0)
    chatWindow->setText(message);
  else
    chatWindow->append(message);
  postNumber++;

  conversationBuffer += message;
  conversationTimeBuffer += message;

  chatWindow->ensureVisible(0, chatWindow->contentsHeight());
  // This should not be a hard coded value like this!!!
  chatWindow->repaintContents(0,0, 10000, 20000, true);
  // Add to the logs.
  if ( chat_log == true ){
    if (chat_log_inHtml == true){
      logStream << message + "<br/>\n";
    }
    else{
      removeAllTags(message);
      logStream << message + "\n";
    }
  }
}

/**
 * Occurs when a message comes in.  If set in the settings the chat window will
 * flash and make noise. Posts to the log (both on screen and hard log if
 * choosen). 
 * @param message the incoming message.    
 */
void Chat::messageIn( QString message ){
  if(message.length() == 0 || message == "" )
    message = "  ";
  QString contactRGB;
  contactRGB = chat_contactName_color.name();

  if (chat_raise_window == true){
    this->show();
    this->raise();
  }

  // render the incoming message in the chatwindow
  if (chat_ignore_contacthtml == true){
    removeAllTags(message);
    updateHTML(message, contactName, contactRGB );
  }
  else
    updateHTML(message, contactName, contactRGB );

  // flash icon on panel bar
  if (chat_flash_on_new_message == true && flashWindow == false){
    flashWindow = true;
    flashIcon();
  }

  // make a sound if set to
  if (( chat_sound == true ) &&
      ( chat_sound_beepReceive == true) &&
      ( chat_sound_playFile == false)){
    emit( soundBeep() );
  }
  if (( chat_sound == true ) &&
      ( chat_sound_beepReceive == true) &&
      ( chat_sound_playFile == true)){
    emit( soundPlayFile(chat_sound_fileReceive) );
  }
}

/**
 * Inserts a plugin that is of type ChatPlugin into the list of plugins that
 * should be allowed to enhance the text before rendering/sending out.
 * @param plugin the plugin to add to our list.
 */
void Chat::insertChatPlugin( const ChatPlugin *plugin ){
  if( plugin == NULL ){
    // Don't want to add nothing!
    return;
  }

  if( plugins.find( plugin ) != -1 ){
    // Don't want to add something we already have.
    return;
  }

  plugins.append( plugin );
}

/**
 * Removes a plugin that is of type ChatPlugin from the list of plugins that
 * should be allowed to enhance the text before rendering/sending out.
 * @param plugin the plugin to remove from our list.
 */
void Chat::removeChatPlugin( const ChatPlugin *plugin ){
  if( plugin == NULL ){
    // Don't want to remove nothing!
    return;
  }
  if( plugins.remove( plugin ) == false ){
    qDebug("Could not remove plugin from Chat class");
  }

}

/**
 * Removes all of the plugins from the list of plugins.
 */
void Chat::removeAllPlugins(){
  while( !plugins.isEmpty() )
    plugins.remove( plugins.first() );
} 

/**
 * Load the new color settings
 */
void Chat::updateChatColorSettings(){
  // Get settings.
  KConfig &config = *KGlobal::config();
  config.setGroup("chatwindow");

  QColor c;
  c.setRgb(0,0,0);
  chat_yourText_color = config.readColorEntry( "chat_yourText_color", &c );
  c.setRgb(0,60,128);
  chat_yourName_color = config.readColorEntry( "chat_yourName_color", &c );
  c.setRgb(0,0,0);
  chat_contactText_color = config.readColorEntry( "chat_contactText_color", &c );
  c.setRgb(128,0,0);
  chat_contactName_color = config.readColorEntry( "chat_contactName_color", &c );
  c.setRgb(255,255,255);
  chat_yourbg_color = config.readColorEntry( "chat_yourbg_color", &c );
  chat_contactbg_color = config.readColorEntry( "chat_contactbg_color", &c );

  applytoInputWindow();
  applytoChatWindow();
}

/**
 * Load the new Font settings
 */
void Chat::updateChatFontSettings(){
  // Get font settings.
  KConfig &config = *KGlobal::config();
  config.setGroup("chatwindow");

  QFont defaultFont = qApp->font();
  defaultFont.setPointSize(DEFAULT_POINT_SIZE);
  chat_yourFont = config.readFontEntry( "chat_yourFont", &defaultFont );
  chat_contactFont = config.readFontEntry( "chat_contactFont", &defaultFont );


  /* set to boolean default flags to true if user is using default font */
  QFont tempfont = qApp->font();
  default_yourFont = (chat_yourFont.family() == tempfont.family())
    && (chat_yourFont.pointSize() == 12);
  default_contactFont = (chat_contactFont.family() == tempfont.family())
    && (chat_contactFont.pointSize() == 12);
	
  fontSize = normalFontSize = int(chat_yourFont.pointSize()/2) - 3;

  // toggle menu items and toolbars depending on font...

  if ( chat_yourFont.weight() == QFont::Bold )
    bold->setEnabled ( false );
  else 
    bold->setEnabled ( true );
 
  if ( chat_yourFont.italic() ) 
    italic->setEnabled( false );
  else 
    italic->setEnabled( true );

  applytoInputWindow();
}

/**
 * Load the new Sound settings
 */
void Chat::updateChatSoundSettings(){
  KConfig &config = *KGlobal::config();
  config.setGroup("chatwindow");

  bool t_chat_sound = config.readBoolEntry( "chat_sound", true );
  chat_sound_beepInitial = config.readBoolEntry( "chat_sound_beepInitial", true );
  chat_sound_beepReceive = config.readBoolEntry( "chat_sound_beepReceive", true );
  chat_sound_beepSend = config.readBoolEntry( "chat_sound_beepSend", true );
  chat_sound_playFile = config.readBoolEntry( "chat_sound_playFile", false );
  chat_sound_fileSend = config.readEntry( "chat_sound_fileSend", DATADIR "/sounds/Send.au" );
  chat_sound_fileReceive = config.readEntry( "chat_sound_fileReceive", DATADIR "/sounds/Receive.au" );
  chat_sound_fileInitial = config.readEntry( "chat_sound_fileInitial", DATADIR "/sounds/Receive.au" );

  if ( chat_sound != t_chat_sound){
    slotChatSound();
  }
}

/**
 * Sets up the input window, chat window and splitter size
 */
void Chat::initChatWindow(){
  splitter->setOrientation(QSplitter::Vertical);

  chatWindow = new KinkattaTextBrowser(splitter, "ChatWindow");
  chatWindow->setHScrollBarMode(QScrollView::AlwaysOff);
  QFont tempFont = chatWindow->font();
  tempFont.setPointSize(DEFAULT_POINT_SIZE);
  chatWindow->setFont(tempFont);
  chatWindow->setNotifyClick(true);

  connect(chatWindow, SIGNAL(urlClick( const QString &)), this, SIGNAL(openURL(const QString &)));
  connect(chatWindow, SIGNAL(mailClick( const QString &, const QString &)), this, SLOT(mailToClicked(const QString &, const QString &)));

  inputWindow = new QMultiLineEdit( splitter , "InputWindow");
  inputWindow->setWordWrap(QMultiLineEdit::WidgetWidth);
  connect(inputWindow, SIGNAL(undoAvailable(bool)), undo, SLOT(setEnabled(bool)));
  connect(inputWindow, SIGNAL(redoAvailable(bool)), redo, SLOT(setEnabled(bool)));
  connect(inputWindow, SIGNAL(textChanged()), this, SLOT(inputTextChanged()));
  connect(inputWindow, SIGNAL(copyAvailable(bool)), this, SLOT( copyChanged(bool)));
  postNumber = 0;
  flashWindow = false;
  splitter->setResizeMode(inputWindow, QSplitter::KeepSize);
  splitter->setResizeMode(chatWindow, QSplitter::Stretch);

  connect(inputWindow, SIGNAL(returnPressed()), this, SLOT(enterKeyPressed()));
  QValueList<int> list;
  list = splitter->sizes();
  list[1] = 40;
  splitter->setSizes(list);

  this->setCentralWidget( splitter );
  inputWindow->setFocus();
}

/**
 * Makes sure that the chatWindow is scrolled all the way down.
 * Occurs when the window size is changed.
 */
void Chat::resizeEvent( QResizeEvent * ){
  if(chatWindow != NULL)
    chatWindow->ensureVisible(0, chatWindow->contentsHeight()+100);
}


/**
 * Sends the message when shift-enter is hit.
 * The return key itself will add a line so it is removed
 */
void Chat::enterKeyPressed(){
  if((shiftIsPressed == true) && (chat_return_send == true)){
    // By the nature of the the fact that they enter key was pressed..
    // we comment out this line for it is already done.
    //inputWindow->append("\n");
  }
  else
  if((shiftIsPressed == true && (chat_sreturn_send == true))
       ||(chat_return_send == true)){
    if (currentMessageOut.length() > 2000){
      QMessageBox::information(0, APP_NAME, i18n("Your message has more than 2000 letters.  It is too long and can not be sent, make it shorter."), "OK");
      return;
    }
    // Repace the text with what was just there before the return. :)
    inputWindow->setText(currentMessageOut);
    lastMessage = currentMessageOut;
    slotContactSendMessage();
    currentMessageOut = QString("");
  }
}

/**
 * Saves what is in the input window in a QString.
 * Occurs when the text in the input window changes.
 */
void Chat::inputTextChanged(){
  //if ((inputWindow->text()).length() != 0)
    currentMessageOut = inputWindow->text();
}

/**
 * If the shift key is down, sets shiftIsPressed = true
 * Occurs when the shift key is pressed.
 * @param e the event
 */
void Chat::keyPressEvent( QKeyEvent *e ){
  if(e->key() == Key_Shift){
    shiftIsPressed = true;
  }
}

/**
 * If the shift key is down, sets shiftIsPressed = false
 * Occurs when the shift key is released .
 * @param e the event
 */
void Chat::keyReleaseEvent( QKeyEvent *e ){
  if(e->key() == Key_Shift){
    shiftIsPressed = false;
  }
}

/**
 * Swaps between two icons (light and dark most likely) until the screen has
 * focus.
 */
void Chat::flashIcon(){
  if( this->isActiveWindow() == true ){
    flashWindow = false;
    flashChoice = true;
    KWin::setIcons(this->winId(), SmallIcon( APP_LOWERCASE_NAME ), SmallIcon( APP_LOWERCASE_NAME ));
  }

  // Swap the icons and check again in half a second.
  if( flashWindow == true ){
    if ( flashChoice == false ){
      KWin::setIcons(this->winId(), SmallIcon( APP_LOWERCASE_NAME ), SmallIcon( APP_LOWERCASE_NAME ));
      flashChoice = true;
    }
    else{
      flashChoice = false;
      KWin::setIcons(this->winId(), SmallIcon( APP_LOWERCASE_NAME "_small_blink" ), SmallIcon( APP_LOWERCASE_NAME "_small_blink" ));
      //this->setIcon(SmallIcon(APP_LOWERCASE_NAME "_small_blink"));
    }
    QTimer::singleShot( 500, this, SLOT(flashIcon()));
  }
}

/**
 * The multi-line edit sent out a signal stating whether it is able to copy
 * text, save
 * that data here.
 */
void Chat::copyChanged( bool copy ){
  // We can't enable/disable the copy buttons because the textview doesn't have a signal.
  canCopy = copy;
}

/**
 * Open this mail address in the system mail editor.
 * @param address the address that someone wants to send a e-mail too.
 */
void Chat::mailToClicked( const QString &, const QString &address ){
  kapp->invokeMailer( address, QString::null );
}

/**
 *  Applies the current colors to the chat window.
 */
void Chat::applytoChatWindow(){
  // apply colors
  QPalette tempPalette = inputWindow->palette();
  //tempPalette.setColor(QColorGroup::Text, chat_yourText_color);
  tempPalette.setColor(QColorGroup::Base, chat_yourbg_color);
  chatWindow->setPalette(tempPalette);
}

/** 
 * apply font and color settings to inputWindow widget
 */
void Chat::applytoInputWindow(){
  QFont tempFont = chat_yourFont;

  QStyleSheet* tempStyle;

  // set to adjust font pointSize by equivalent html size from up/down arrows in chat
  if (fontSize != 3) {
    if (this->chatWindow) {
      tempStyle = chatWindow->styleSheet();
      tempFont.setPointSize(DEFAULT_POINT_SIZE);
      tempStyle->scaleFont(tempFont,fontSize);
    }
  } else {
    tempFont.setPointSize(DEFAULT_POINT_SIZE);
  }

  inputWindow->setFont(tempFont);

  // apply colors
  QPalette tempPalette = inputWindow->palette();
  tempPalette.setColor(QColorGroup::Text, chat_yourText_color);
  tempPalette.setColor(QColorGroup::Base, chat_yourbg_color);
  inputWindow->setPalette(tempPalette);
}

/**
 * Opens a log file.  Options include single file, multiple file, html,
 * plain text.
 */
void Chat::openLogFile(){
  if( chat_log == true )
    return;

  chat_log = true;
  QString fileNamePath;

  // Single File or multiple files.
  if ( chat_log_singleFile == false ){
    // Each chat gets it's own file.
    QString endingTag;
    if (chat_log_inHtml == true)
      endingTag = ".html";
    else
      endingTag = ".txt";
    QString date = ((QDate::currentDate ()).toString());
    date.replace( QRegExp(" "), "" );
    date = date.mid(3,date.length()-3);
    QString month_day = date.left(date.length()-4);
    QString year = date.right(4);
    date = month_day + "-" + year;

    QString logFolder = QDir::homeDirPath() + "/." + QString(APP_LOWERCASE_NAME) + "/logs/" + contactName.lower() + "/";
    QString fileNamePath = logFolder + contactName.lower() + "-" + date + endingTag;

    logFile.setName(fileNamePath);
    int logNumber = 2;
    while (logFile.exists() == true){
      fileNamePath = logFolder + contactName.lower() + "-" + date + "-" + QString("%1").arg(logNumber) + endingTag;
      logFile.setName(fileNamePath);
      logNumber++;
    }
  }
  else{
    fileNamePath = QDir::homeDirPath() + "/." + QString(APP_LOWERCASE_NAME) + "/logs/" + contactName.lower() + "/" + contactName.lower() + ".log";
    logFile.setName(fileNamePath);
  }

  QDir newDir("/dev/null");
  //Make Dir
  QString d = QDir::homeDirPath() + "/." + QString(APP_LOWERCASE_NAME);
  if( !QFile::exists(d) ) newDir.mkdir(d, true);
  d = QDir::homeDirPath() + "/." + QString(APP_LOWERCASE_NAME) + "/logs/";
  if( !QFile::exists(d) ) newDir.mkdir(d, true);
  d = QDir::homeDirPath() + "/." + QString(APP_LOWERCASE_NAME) + "/logs/" + contactName.lower();
  if( !QFile::exists(d) ) newDir.mkdir(d, true);
  d = QDir::homeDirPath() + "/." + QString(APP_LOWERCASE_NAME) + "/logs/" + contactName.lower();
  if( !QFile::exists(d) ) newDir.mkdir(d, true);
  
  // If we can not open the file to write to.
  if (logFile.open( IO_WriteOnly | IO_Append ) == false){
    QMessageBox::critical(0, APP_NAME, QString(i18n("Unable to open logfile for appending: ")) + fileNamePath , "OK");
    chat_log = false;
    log->setChecked( false );
    return;
  } 
  else{
    log->setChecked( true );
    logStream.setDevice(&logFile);
    if (chat_log_inHtml == true){
      logStream << "<HTML>\n<HEAD>\n<TITLE>Conversations with ";
      logStream << contactName;
      logStream << "</TITLE>\n</HEAD>\n<BODY BGCOLOR=\"#FFFFFF\">\n";
      logStream << "<HR><BR><H3 Align=Center> ---- New Conversation @ ";
      logStream << QDate::currentDate().toString().latin1();
      logStream << " ----</H3><BR>";
    }
    else{
      logStream << "Conversations with " << contactName << "\n";
      logStream << " at " << QDate::currentDate().toString().latin1() << "\n";
    }
  }

  // Put in any converstaion that we have already had.
  if (chat_log_inHtml == true){
    if (( chat_log_time == true) && ( chat_log == true))
      logStream << conversationTimeBuffer;
    else
      logStream << conversationBuffer;
  }
  else {
    if (( chat_log_time == true) && ( chat_log == true)){
      removeAllTags(conversationTimeBuffer);
      logStream << conversationTimeBuffer;
    }
    else{
      removeAllTags(conversationTimeBuffer);
      logStream << conversationBuffer;
    }
  }

  // Flush everything we have so far.
  logFile.flush();
}

/**
 * If we are logging, stops logging.
 * @param remove whether the log file be deleted after it is closed.
 */
void Chat::closeLogFile( const bool remove ){
  if( chat_log == false )
    return;
  
  // Stop Logging.
  chat_log = false;
  log->setChecked( false );
  
  logFile.close();

  if( remove ){
    if ( logFile.remove() == false)
      QMessageBox::critical(0, APP_NAME, i18n("Can not remove logfile."), "OK");
  }
}

/**
 * Starts or stops the logging.  Deletes the log file if there is one.
 */
void Chat::slotChatLog(){
  if (chat_log == false){
    // Start Logging.
    openLogFile();
  }
  else
    closeLogFile( true );
}

/**
 * Clear the chat window.  This also clears the text buffers.
 */
void Chat::slotChatClear(){
  cleared = true;
  chatWindow->setText( "", chatWindow->context());
  postNumber = 0;
  // Clear conversation
  if (chat_log_inHtml == true){
    conversationBuffer = "<HTML>\n<HEAD>\n<TITLE>Conversations with ";
    conversationBuffer += contactName;
    conversationBuffer += "</TITLE>\n</HEAD>\n<BODY BGCOLOR=\"#FFFFFF\">\n";
    // Clear other conversation buffer
    conversationTimeBuffer = "<HTML>\n<HEAD>\n<TITLE>Conversations with ";
    conversationTimeBuffer += contactName;
    conversationTimeBuffer += "</TITLE>\n</HEAD>\n<BODY BGCOLOR=\"#FFFFFF\">\n";
  }
  else{
    conversationBuffer = "Conversations with " + contactName + "\n";
    // Clear other conversation buffer
    conversationTimeBuffer = "Conversations with " + contactName + "\n";
  }
}

/**
 * Save the chat to a file.  Prompt the user for a location and try to save.
 * Use same options as normal logging.
 */
void Chat::slotChatSaveAs(){
  QString fileName = KFileDialog::getSaveFileName(0, 0, this);
  if (fileName.isEmpty())
    return;
  
  // open file for appending, to add the new message
  QFile SaveFile(fileName);
  if (SaveFile.open( IO_WriteOnly ) == false){
    QMessageBox::critical(0, APP_NAME, QString(i18n("Unable to open the file for appending: ")) + fileName, "OK");
    SaveFile.close();
    return;
  }
  
  // All text
  QTextStream fileStream(&SaveFile);
  if (cleared == false){
    if (chat_log_inHtml == true){
      fileStream << "<HTML>\n<HEAD>\n<TITLE>Conversations with ";
      fileStream << contactName;
      fileStream << "</TITLE>\n</HEAD>\n<BODY BGCOLOR=\"#FFFFFF\">\n";
    }
    else{
      fileStream << "Conversations with " << contactName << "\n";
    }
  }
  if (( chat_log_time == true) || ( chat_log == true))
    if (chat_log_inHtml == false){
      removeAllTags(conversationTimeBuffer);
      fileStream << conversationTimeBuffer;
    }
    else
      fileStream << conversationTimeBuffer;
  else
    fileStream << conversationBuffer;

  if (chat_log_inHtml == true){
    fileStream << "</BODY>\n</HTML>";
  }
  SaveFile.close();
  QMessageBox::information(0, APP_NAME, QString(i18n("Chat up to this point has been saved in: ")) + fileName , "OK");
}

/**
 * Print out the current chat window.
 */
void Chat::slotChatPrint(){
  KPrinter printer;
  printer.setFullPage(true);
  if ( !printer.setup() )
    return;
  QPainter p( &printer );
  QPaintDeviceMetrics metrics(p.device());
  int dpix = metrics.logicalDpiX();
  int dpiy = metrics.logicalDpiY();
  const int margin = 72; // pt
  QRect body(margin*dpix/72, margin*dpiy/72, metrics.width()-margin*dpix/72*2,
    metrics.height()-margin*dpiy/72*2 );
  QFont font("times", 10);
  QSimpleRichText richText( conversationBuffer, font, chatWindow->context(), chatWindow->styleSheet(),
	chatWindow->mimeSourceFactory(), body.height() );
  richText.setWidth( &p, body.width() );
  QRect view( body );
  int page = 1;
  do {
    // Draw the first page
    richText.draw( &p,body.left(), body.top(), view, colorGroup() );
    view.moveBy( 0, body.height() );
    p.translate( 0 , -body.height() );
    // Draw page number
    p.setFont( font );
    p.drawText( view.right() - p.fontMetrics().width( QString::number(page) ),
      view.bottom() + p.fontMetrics().ascent() + 5, QString::number(page) );
    // Check if it is time to stop printing
    if ( view.top()  >= richText.height() ) break;
    // Guess it isn't so start a new page
    printer.newPage();
    // Set the new page number
    page++;
  } while (true);
}

/**
 * Turn Sound on and off
 */
void Chat::slotChatSound(){
  if (chat_sound == false){
    chat_sound = true;
    sound->setChecked( true );
    sound->setText( "Sound Enabled" );
    if (chat_sound_beepReceive == false &&
	chat_sound_beepSend == false){
      chat_sound_beepReceive = true;
      chat_sound_beepSend = true;
    }
  }
  else{
    sound->setChecked( false );
    sound->setText( "Sound Disabled" );
    chat_sound = false;
  }
}

/**
 * Close the Chat.  Send out a close event.
 */
void Chat::slotChatClose(){
  QApplication::sendEvent(this, new QCloseEvent());
}

/**
 * Close out of the Chat, emit quit for parent to clean up anything.
 */
void Chat::closeEvent( QCloseEvent * ){
  // Close window
  emit quiting( contactName );
}

/**
 * If the input window has focus tell it to cut the highlighted text.
 */
void Chat::slotEditCut(){
  if (inputWindow->hasFocus())
    inputWindow->cut();
}

/**
 * Copy highlighted text to the clipboard.
 */
void Chat::slotEditCopy(){
  if ( inputWindow->hasFocus() ){
    inputWindow->copy();
    return;
  }
  if ( chatWindow->hasFocus() ){
    chatWindow->copy();
    return;
  }
  qDebug("Trying to copy without focus.");
}

/**
 * Paste the clipboard to edit box
 */
void Chat::slotEditPaste(){
  //if ( inputWindow->hasFocus() )
    inputWindow->paste();
}

/**
 * Select All in the window with the focus.
 */
void Chat::slotEditSelectAll(){
  if( inputWindow->hasFocus() ){
    inputWindow->selectAll();
    return;
  }
  if( chatWindow->hasFocus() ){
    chatWindow->selectAll();
    return;
  }
}

/**
 * Edit Undo
 */
void Chat::slotEditUndo(){
  if (inputWindow->hasFocus())
    inputWindow->undo();
}

/**
 * Edit Redo
 */
void Chat::slotEditRedo(){
  //if (inputWindow->hasFocus())
    inputWindow->redo();
}


/**
 * Toggle the Chat Toolbar's visability.
 */
void Chat::slotViewChatToolBar(){
  KToolBar *chatToolBar = toolBar("Chat ToolBar");
  if(chatToolBar->isVisible())
    chatToolBar->hide();
  else
    chatToolBar->show();
}

/**
 * Toggle the Text Toolbar's visability.
 */
void Chat::slotViewTextToolBar(){
  KToolBar *textToolBar = toolBar("Text ToolBar");
  if(textToolBar->isVisible())
    textToolBar->hide();
  else
    textToolBar->show();
}

/**
 * Toggle the Plugin Toolbar's visability.
 */
void Chat::slotViewPluginToolBar(){
  KToolBar *pluginToolBar = toolBar("Plugin ToolBar");
  if(!pluginToolBar)
    return;

  if(pluginToolBar->isVisible())
    pluginToolBar->hide();
  else
    pluginToolBar->show();
}

/**
 * Toggle the TimeStamp Variable.  Reload log.
 */
void Chat::slotViewTimeStamp(){
  if (chat_show_time == false){
    time->setChecked(true);
    chat_show_time = true;
    chat_log_time = true;
    chatWindow->setText(conversationTimeBuffer);
    if (chat_log == true){
      // Turn off
      slotChatLog();
      // And back on, but with time (in case)
      slotChatLog();
    }
  }
  else{
    chat_show_time = false;
    time->setChecked(false);
    chat_log_time = true;
    chatWindow->setText(conversationBuffer);
  }
  
  // scroll to the new text
  chatWindow->ensureVisible(0, chatWindow->contentsHeight());
}

/**
 * Insert one of those "emotions" into the window.
 * When done update the currentMessageOut sense no signal is emmited
 * by changing the text.
 * @param face the number of the face to insert.
 */
void Chat::slotInsertFace( int face ){

  #define doface(x, y) \
  if (face == x){ \
    inputWindow->insert(y); \
    currentMessageOut = inputWindow->text(); \
    return; \
  }

  doface(FACE_SMILE_ID, FACE_SMILE_TEXT);
  doface(FACE_SAD_ID, FACE_SAD_TEXT);
  doface(FACE_WINK_ID, FACE_WINK_TEXT);
  doface(FACE_TONGUE_ID, FACE_TONGUE_TEXT);
  doface(FACE_SCREAM_ID, FACE_SCREAM_TEXT);
  doface(FACE_KISS_ID, FACE_KISS_TEXT);
  doface(FACE_YELL_ID, FACE_YELL_TEXT);
  doface(FACE_GLASSES_ID, FACE_GLASSES_TEXT);
  doface(FACE_ONEEYE_ID, FACE_ONEEYE_TEXT);
  doface(FACE_MONEYMOUTH_ID, FACE_MONEYMOUTH_TEXT);
  doface(FACE_BURP_ID, FACE_BURP_TEXT);
  doface(FACE_EMBARRASSED_ID, FACE_EMBARRASSED_TEXT);
  doface(FACE_ANGEL_ID, FACE_ANGEL_TEXT);
  doface(FACE_THINK_ID, FACE_THINK_TEXT);
  doface(FACE_CRY_ID, FACE_CRY_TEXT);
  doface(FACE_CROSSEDLIPS_ID, FACE_CROSSEDLIPS_TEXT);
  doface(FACE_BIGSMILE_ID, FACE_BIGSMILE_TEXT);
  doface(FACE_LUKE_ID, FACE_LUKE_TEXT);

  #undef doface
}

/**
 * Inserts a set of tags into the input window.
 * @param open the start of the tag.
 * @param close the end of the tag.
 */
void Chat::slotInsertTags( QString open, QString close ){
  QString oldtext = "";
  if (canCopy){
    // Cut any selected text to clipboard
    inputWindow->cut();
    // Copy text from the clipboard (paste)
    oldtext = QApplication::clipboard()->text();
  }
  // Add tags
  oldtext = open  + oldtext + close;
  // paste new text into input window
  inputWindow->insert(oldtext);
  int a,b;
  inputWindow->getCursorPosition(&a,&b);
  inputWindow->setCursorPosition(a,b-close.length(),false);
  currentMessageOut = inputWindow->text();
}

/**
 * Insert bold tags in the input window
 */
void Chat::slotInsertBold(){
  slotInsertTags( "<b>" , "</b>" );
}

/**
 * Insert italic tags in the input window
 */
void Chat::slotInsertItalic(){
  slotInsertTags( "<i>" , "</i>" );
}

/**
 * Insert underline tags in the input window
 */
void Chat::slotInsertUnderline(){
  slotInsertTags( "<u>" , "</u>" );
}

/**
 * Insert small tags in the input window
 */
void Chat::slotInsertShrinkFont(){
  if (fontSize > 0)
    fontSize--;
  applytoInputWindow();
}

/**
 * Set the Font back to normal
 */
void Chat::slotInsertNormalFont(){
  fontSize = normalFontSize;
  applytoInputWindow();
}

/**
 * Increase font tags
 */
void Chat::slotInsertIncreaseFont(){
  if (fontSize < 7)
    fontSize++;
  applytoInputWindow();
}

/**
 *  Get a background color for the input and log window.
 */
void Chat::slotInsertBackGroundColor(){
  KColorDialog::getColor ( chat_yourbg_color, this );
  applytoInputWindow();
  applytoChatWindow();
}

/**
 * Brings up the color dialog and get a pen color.
 */
void Chat::slotInsertPenColor(){
  KColorDialog::getColor ( chat_yourText_color, this );
  applytoInputWindow();
  applytoChatWindow();
}

/**
 * Inserts the link tags into the input window.
 */
void Chat::slotInsertLink(){
  KinkattaLinkDialog linkDlg(this, "link dialog", true);

  linkDlg.setURLText( QApplication::clipboard()->text() );
  linkDlg.setDescText( QApplication::clipboard()->text() );

  QString tag = linkDlg.go();

  if(!tag.isEmpty()){
    inputWindow->cut();
    inputWindow->insert(tag);
    currentMessageOut = inputWindow->text();
  }
}

/**
 * Puts the last message sent into the input box.
 */
void Chat::slotInsertLastMessage(){
  if (inputWindow->text() == QString(""))
    inputWindow->setText(lastMessage);
  else
    inputWindow->append(lastMessage);
}

/**
 * Send the message in the input window out.
 */
void Chat::slotContactSendMessage(){
  //currentMessageOut = "";
  QString fontsizeTag = "";
  QString fontfaceTag = "";
  QString fontcolorTag = "";

  // Example Output HTML
  //<HTML><BODY BGCOLOR="#FFFF00"><FONT COLOR="#FF00FF">Room for one more?</FONT></BODY></HTML>
  bool send = true;
  if ((inputWindow->length() == 0) && (chat_send_blank == false)){
    send = false;
    QString message = inputWindow->text();
    if(message == "\n")
      inputWindow->setText( "" );
  }
  if (send != true)
    return;

  // Create color tags
  QString nameRGB;
  nameRGB = chat_yourName_color.name();
  QString myRGB = "";
  if( chat_yourText_color != Qt::black )
    myRGB= chat_yourText_color.name();
  QString bgRGB = "";
  if(chat_yourbg_color != Qt::white )
    bgRGB = chat_yourbg_color.name();

  // Send the input window data
  QString message = inputWindow->text();
  inputWindow->setText( "" );
  // Save this message in case people want to use it.
  QString lastLetter = message.mid(message.length()-1,1);
  if (lastLetter == "\n"){
    message = message.mid(0, message.length()-1);
    lastMessage = message;
  }
  lastLetter = message.mid(message.length()-1,1);
  if (lastLetter == "\n"){
    message = message.mid(0, message.length()-1);
    lastMessage = message;
  }

  // Ok we have the message now!
  ChatPlugin *plugin = plugins.first();
  while( plugin ){
    plugin->enhanceText( message );
    plugin = plugins.next();
  }

  // Automaticly link anything we can.
  autoLinkUrls( message );
  
  //qDebug("Sending This Message: %s\n", message.latin1());

  // wrap bold around if bold font selected...
  if ( chat_yourFont.weight() == QFont::Bold )
    message = QString("<b>") + message + "</b>";

  // wrap italics around if italic font selected...
  if ( chat_yourFont.italic() )
    message = QString("<i>") + message + "</i>";

  // render message to be sent with HTML in the chatwindow (TextBrowser widget)
  updateHTML(message, myName, nameRGB);

  // get the font tag...
  QString fontTag;
  fontTag = createFontTag(chat_yourFont.family(), myRGB, fontSize);
  if(!fontTag.isEmpty()){
    fontTag = "<FONT" + fontTag + ">";
    message = fontTag + message + "</FONT>";
  }

  // add background color tag
  if(!bgRGB.isEmpty())
    message = QString("<BODY BGCOLOR=\"") + bgRGB + "\">" + message + "</BODY>";
  else{
    // Winaim requires this... grr
    //message = QString("<BODY>") + message + "</BODY>";
  }
 
  // fix for smily rendering in winAim
  fontFixSmilies(message, fontTag);

  //message = QString("<HTML>") + message + "</HTML>";
  emit( messageOut( message, contactName ) );
  

  // Sound.
  if (( chat_sound == true ) &&
    ( chat_sound_beepSend == true) &&
    ( chat_sound_playFile == false)){
    emit(soundBeep());
  }
  else{
    if(( chat_sound == true ) &&
      ( chat_sound_beepSend == true) &&
      ( chat_sound_playFile == true)){
      emit(soundPlayFile(chat_sound_fileSend));
    }
  }
}

/**
 * Any text that has a url inside of it, the tag is added to it so it becomes a link.
 * @param text the text to parse and link.
 */
void Chat::autoLinkUrls( QString &text ){
  QRegExp url;
  QString link;
  int len = 0, start = 0, end = 0, endanchor = 0;
  int starthttp = 0, startwww = 0, startanchor = 0, skipto = 0;
  int lenhttp = 0, lenwww = 0;
  bool http;
  url.setCaseSensitive(FALSE);

  //qDebug("text %s\n", text.latin1());
  while(start != -1){
    //first we search for any already-inserted anchors, that would be here
    //from "Insert->Link" so as not to process them and leave them alone.
    url.setPattern("<a href");
    startanchor = url.match(text, end);
    if(startanchor != -1){
      url.setPattern("</a>");
      endanchor = url.match(text, startanchor, &len);
      skipto = endanchor + len;
      //qDebug("skipto %d end %d\n", skipto, end);
    }

    url.setPattern( "http://[a-zA-Z0-9][^ ^\n^\r^\t]*" );
    starthttp = url.match(text, end, &lenhttp);
    url.setPattern( "www\\.[a-zA-Z0-9][^ ^\n^\r^\t]*" );
    startwww = url.match(text, end, &lenwww);

    //qDebug("startanchor %d startwww %d starthttp %d\n", startanchor, startwww, starthttp);
    //qDebug("lenwww %d lenhttp %d\n", lenwww, lenhttp);

    if( (startanchor != -1)
      && (startanchor < (starthttp > -1 ? starthttp : startanchor+1) )
      && (startanchor < (startwww > -1 ? startwww : startanchor+1) ) ){
      end = skipto;
      //qDebug("SKIP skipto: %d\n", skipto);
      goto SKIP;
    }

    if( (starthttp == -1) && (startwww == -1) ) //neither found
      break;
    else if(starthttp == -1){ //http not found
      start = startwww;
      end = startwww + lenwww;
      http = FALSE;
    }else if(startwww == -1){ //www not found
      start = starthttp;
      end = starthttp + lenhttp;
      http = TRUE;
    } else { //both found. use the first one!
      start = (starthttp < startwww) ? starthttp : startwww;
      end = (starthttp < startwww) ? starthttp + lenhttp : startwww + lenwww;
      http = (starthttp < startwww) ? TRUE : FALSE;
    }

    //url.setPattern( QString("[ \n\r\t]") );  //e.g, any whitespace
    //end = url.match(text, start+len, 0, TRUE);

    //this happens when the end of the url is at the end of message
    if(end == -1)
      end = text.length();

    //insert this first, makes it easier.
    text.insert(end, "</a>");

    if(http)
      link = "<a href=\"" + text.mid(start, end-start) + "\">";
    else //need the leading "http://"
      link = "<a href=\"http://" + text.mid(start, end-start) + "\">";

    text.insert(start, link );
    end += link.length() + 4; // </a> == 4

   SKIP:
     ; //pacifying g++
  }
}

/**
 * Update (render) the HTML in chatwindow widget
 * @param message the message to write out.
 * @param sender the person who sent it.
 * @param HTMLColor the color that should be used??
 */
void Chat::updateHTML(QString message, QString sender, QString HTMLColor){
  checkOverHtml(message);
  formatHtmlForDisplay(message);

  // Add img tags for smilies
  if( chat_show_smilies )
    replaceSmiliesWithImgTags(message);

  // render at least a space
  if (message.length() == 1 || message.length() == 0 || message == "")
    message += " ";

	//printf("message:%s\n", message.latin1());

	// Get Timestamp info
	QString CurrentTime = QTime::currentTime().toString();


	// Get the font tag...
	QString fontTag;
	if (sender == myName)
	  fontTag = createFontTag(chat_yourFont.family(),
	     chat_yourText_color.name(), fontSize);
	else
	  fontTag = createFontTag(chat_contactFont.family(),
	    chat_contactText_color.name(), 3);

	// Wrap it around the message...
	if(!fontTag.isEmpty()){
	   fontTag = "<FONT" + fontTag + ">";
	  message = fontTag + message + "</FONT>";
	}


  int found = -1;
  bool done = false;
  while ( done == false )
  {
    found = message.find("/me", found+1);
    if (found != -1)
    {
      QString before = message.left(found);
      int left = before.contains("<");
      int right = before.contains(">");
      if ( left == right )
      {
        if ( QString(message.at(found-1)) != QString(">") )
          found = -1;
        done = true;
      }
    }
    else
      done = true;
  }

  if ( found != -1 ){
    sender = QString( "***" ) + sender;
    message = message.mid( 0, found) + message.mid( found+ 4, message.length()-4-found);
  }
  else
    sender += ":";

        /***** Get the name and : ****/
	QString TimeMessage = QString("<B><font color=\"") +
	  HTMLColor + QString("\">") + CurrentTime + " " + sender + " </font></B>"  + message;

	QString Message = QString("<B><font color=\"") +
	  HTMLColor + QString("\">") + " " + sender + " </font></B>"  + message;

	QString bgRGB = "#ffffff";
	if (sender == myName)
		bgRGB = chat_yourbg_color.name();
	else
		bgRGB = chat_contactbg_color.name();

	// Add to the logs.
	if (chat_log == true){
		if (chat_log_inHtml == true){
			if (chat_log_time == true)
				logStream << TimeMessage << "<br>\n";
			else
				logStream << Message << "<br>\n";
		}
		else{
			// Write correct buffer
			if ( chat_log_time == true){
				removeAllTags(TimeMessage);
				logStream << TimeMessage << "\n";
			}else{
				removeAllTags(Message);
				logStream << Message << "\n";
			}
		}
		logFile.flush();
	}


	Message.replace( QRegExp("\n"), "<br/>" );
	TimeMessage.replace( QRegExp("\n"), "<br/>" );

	// now add the message to the buffer
	conversationBuffer += Message;
	conversationTimeBuffer += TimeMessage;
	if (chat_log_inHtml == true){
		conversationBuffer += "<br/>\n";
		conversationTimeBuffer += "<br/>\n";
	}
	//printf("bgRGB = %s\n",bgRGB.latin1());

	// Always add a background so there will be a return.
	//if (bgRGB != "#ffffff"){
		Message = QString("<table bgcolor=" + bgRGB + " border=0 cellpadding=0 cellspacing=0><tr><td>") + Message + "</td></tr></table><br/>";
		TimeMessage = QString("<table bgcolor=" + bgRGB + " border=0 cellpadding=0 cellspacing=0><tr><td>") + TimeMessage + "</td></tr></table><br/>";
	//}

	if(chat_show_time == true){
		if (postNumber == 0)
			chatWindow->setText(TimeMessage);
		else
			chatWindow->append(TimeMessage);
	}
	else{
		// NO TIME
		if (postNumber == 0)
			chatWindow->setText(Message);
		else
			chatWindow->append(Message);
	}

	// Incriment the post numbers
	postNumber++;
	if (postNumber == 1){
		if (( chat_sound == true ) &&
			(chat_sound_beepInitial == true) &&
			chat_sound_playFile == false){
			emit(soundBeep());
		}
		else{
			if ((chat_sound == true ) &&
			(chat_sound_beepInitial == true) &&
			(chat_sound_playFile == true)){
				emit(soundPlayFile(chat_sound_fileInitial));
			}
		}
	}

	// scroll to the new text
	chatWindow->ensureVisible(0, chatWindow->contentsHeight());
	// This should not be a hard coded value like this!!!
	chatWindow->repaintContents(0,0, 10000, 29000, true);

	// if conversationLength is getting too long, cut the top out
	if (chat_line_limit == true)
		if(postNumber > chat_line_limit_number){
			int spos = conversationBuffer.find("<B>");
			conversationBuffer.remove(spos, 4);
			spos = conversationBuffer.find("<B>");
			conversationBuffer.remove(0, spos);
			postNumber--;
			chatWindow->setText( conversationBuffer, chatWindow->context());
		}

	// reset contact's colors
	chat_contactText_color = temp_ContactTextColor;
	chat_contactbg_color = temp_ContactBgColor;
}

/**
 * Makes sure that in every open tag there are matching ""
 * Goes through one by and an makes sure that every tag is known.
 * If a greater then or less then is not known then it is displayed.
 * Formats the html for our window, shoudn't be used when sending out text.
 * @param text the text to check over.
 */
void Chat::formatHtmlForDisplay( QString &text ){
  //    Weirdness from winAIM.  Sometimes <B> or <I> is immediately
  //    followed by </FONT> and this breaks Textview...which
  //    is weird itself in some ways :-)
  //    note: winaim always seems to send combos as <B><I>
  text.replace( QRegExp("<B></FONT>"), "</FONT><B>");
  text.replace( QRegExp("<I></FONT>"), "</FONT><I>");
  text.replace( QRegExp("<B><I></FONT>"), "</FONT><B><I>");
  //     Weirdness in TextView. Doesn't like </B></FONT> etc.
  //     apparently </FONT> is enough and it breaks otherwise.
  text.replace( QRegExp("</B></FONT>"), "</FONT>");
  text.replace( QRegExp("</I></FONT>"), "</FONT>");
  text.replace( QRegExp("</B></I></FONT>"), "</FONT>");

#define CHECK_QUOTES { \
  int quotecount = 0; \
  while(text[++i] != '>' && i < text.length()){ \
    if(text[i] == '"') \
      quotecount++; \
  } \
  if(quotecount % 2 == 1){ \
    text.replace(i, 1, "\">"); \
    i++; \
  } \
}

#define SKIP_TO_GT {while(text[++i] != '>' && i < text.length());}

#define KILL_LT { text.replace(i, 1, "&lt;"); i += 3; }

  bool bodyUsed = false;
  bool divUsed = false;
  bool pUsed = false;

  for(unsigned i = 0; i < text.length(); i++) {
    if(text[i] == '>') {
      text.replace(i, 1, "&gt;");
      i += 3;
    }
    else if(text[i] == '<') {
      if(text.mid(i, 3).upper() == "<A ") CHECK_QUOTES
      else if(text.mid(i, 3).upper() == "<B>") i += 2;
      else if(text.mid(i, 5).upper() == "<BIG>") i += 4;
      else if(text.mid(i, 4).upper() == "<BR>") i += 3;
      else if(text.mid(i, 6).upper() == "<CODE>") i += 5;
      else if(text.mid(i, 4).upper() == "<EM>") i += 3;
      else if(text.mid(i, 5).upper() == "<FONT") CHECK_QUOTES
      else if(text.mid(i, 3).upper() == "<I>") i += 2;
      else if(text.mid(i, 7).upper() == "<SMALL>") i += 6;
      else if(text.mid(i, 4).upper() == "<TT>") i += 3;
      else if(text.mid(i, 3).upper() == "<U>") i += 2;
      else if(!divUsed && text.mid(i, 4).upper() == "<DIV")
      {
        divUsed = true;
        CHECK_QUOTES
      }
      else if(!pUsed && text.mid(i, 2).upper() == "<P") {
        pUsed = true;
        CHECK_QUOTES
      }
      else if(!bodyUsed && text.mid(i, 5).upper() == "<BODY") {
        bodyUsed = true;
        CHECK_QUOTES
      }
      else if(text.mid(i, 7).upper() == "</HTML>") KILL_LT
      else if(text.mid(i, 7).upper() == "</DIV>") KILL_LT
      else if(text.mid(i, 7).upper() == "</BODY>") KILL_LT
      else if(text.mid(i, 2).upper() == "</") SKIP_TO_GT
      else KILL_LT;
    }
  }

  #undef CHECK_QUOTES
  #undef KILL_LT
  #undef SKIP_TO_GT
}

/**
 * remove <html>/<pre> tags, replace <p> with <br> Fix common tag errors.
 * @param text the text to clean up.
 */
void Chat::checkOverHtml( QString &text ){
  // 1) cut out <html> and </html> tags.
  int bodyPos = text.find( QRegExp("<html>", false) );
  while( bodyPos != -1 ){
    text.replace(bodyPos, 6, "");
    bodyPos = text.find( QRegExp("<html>", false) );
  }
  bodyPos = text.find( QRegExp("</html>", false) );
  while( bodyPos != -1 ){
    text.replace(bodyPos, 7, "");
    bodyPos = text.find( QRegExp("</html>", false) );
  }

  // 2a) cut <p> tags, replace </p> with <br>
  text.replace( QRegExp("<p>", false), "" );
  text.replace( QRegExp("</p>", false), "<br>" );
  // 2b) cut <pre> and </pre> tags
  text.replace( QRegExp("<pre>", false), "" );
  text.replace( QRegExp("</pre>", false), "" );

  // 3) replace <body> with <div>, inserting a STYLE= value to apply text and
  // background colors
  bodyPos = text.find("<BODY BGCOLOR=");
  if(bodyPos != -1){
    temp_ContactBgColor = chat_contactbg_color;
    QString BGCOLOR = text.mid(bodyPos+16,8);
    if (text.mid(bodyPos+16,6) != "ffffff") {
      chat_contactbg_color.setRgb(
			((BGCOLOR.mid(0,2)).toLong(0,16)),
			((BGCOLOR.mid(2,2)).toLong(0,16)),
			((BGCOLOR.mid(4,2)).toLong(0,16))
			);
    }
    // Get rid of the BODY BGCOLOR TAG :)
    int value = text.find('>', bodyPos+16);
    if( value != -1 )
      text.replace(bodyPos, value+1-bodyPos, "");
  }

  // parse for AOL's weird BACK tag for bgRGB
  bodyPos = text.find(QRegExp("BACK=\"#......\">"), false);
  if (bodyPos != -1) {
    QString BACK = text.mid(bodyPos+7, 6);
    chat_contactbg_color.setRgb(
			((BACK.mid(0,2)).toLong(0,16)),
			((BACK.mid(2,2)).toLong(0,16)),
			((BACK.mid(4,2)).toLong(0,16))
			);
  }
  // BACK tag can be combined with FONT COLOR (next) so not removing :)
  bodyPos = text.find("<FONT COLOR=", 0, false);
  if(bodyPos != -1){
    temp_ContactTextColor = chat_contactText_color;
    QString COLOR = text.mid(bodyPos+14,8);
    chat_contactText_color.setRgb(
			((COLOR.mid(0,2)).toLong(0,16)),
			((COLOR.mid(2,2)).toLong(0,16)),
			((COLOR.mid(4,2)).toLong(0,16))
			);
    // Get rid of the FONT TAG :)
    int value = text.find('>', bodyPos);
    if( value != -1 )
      text.replace(bodyPos, value+1-bodyPos, "");

    // Get rid of the next </FONT> TAG :)
    value = text.find("</font>", bodyPos);
    if( value != -1 )
      text.replace(value, 7, "");
  }
  //printf("string new: %s\n", text.latin1());

  // 3b) remove </body>
  text.replace( QRegExp("</body>", false), "" );

  // 4) make sure the allowed tags are balanced, inserting a </tag>
  //    at the end of the message to balance if needed.
  #define CLOSE_TAG(tag) \
    if( text.contains("<" tag ">", false) > text.contains("</" tag ">", false) ) \
      text += ("</" tag ">");
  CLOSE_TAG("a");
  CLOSE_TAG("b");
  CLOSE_TAG("big");
  CLOSE_TAG("code");
  CLOSE_TAG("em");
  CLOSE_TAG("font");
  CLOSE_TAG("i");
  CLOSE_TAG("small");
  CLOSE_TAG("tt");
  CLOSE_TAG("u");
  
  #undef CLOSE_TAG

  // If diasplaying on our chatBoard call that function
}

/**
 * Creates the inerds of a font tag, but without the <FONT or >
 * Ex: createFontTag( "brizil", "#123456", 5) returns:
 *  SIZE="5" FACE="brizil" COLOR="#123456"
 * Ignores default settings and does not add them.
 * @return HTML fontTag
 * @param face font to use in the created tag.
 * @param color color to use in the created tag.
 * @param size text size to use in the created tag.
 */
QString Chat::createFontTag( const QString face, const QString color, const int size ){
  QString fontSizeTag;
  QString fontFaceTag;
  QString fontColorTag;

  // Add font current size
  if ( fontSize != 3 ) {
    fontSizeTag.setNum(size);
  }

  // Add font face tag
  if ( !default_yourFont ) {
    fontFaceTag = face;
  };

  // Add color
  if ( color != "#000000" ) {
    fontColorTag = color;
  }

  // Build tag
  QString fontTag;
  
  if(!fontSizeTag.isEmpty()){
    fontTag += " SIZE=\"" + fontSizeTag + "\"";
  }
  if(!fontFaceTag.isEmpty()){
    fontTag += " FACE=\"" + fontFaceTag + "\"";
  }
  if(!fontColorTag.isEmpty()){
    fontTag += " COLOR=\"" + fontColorTag + "\"";
  }

  return fontTag;
}

/**
 * Removes all tags (not just html) from the text.
 * @param text the text to remove the tags from.
 */ 
void Chat::removeAllTags( QString &text ){
  int start = 0;
  int end = 0;
  while ( end!=-1 && start!=-1 ){
    start = text.find("<");
    end = text.find(">");
    if( end!=-1 && start!=-1 ){
      text.remove(start, end-start+1);
    }
  }
}

/**
 * Return message with HTML fixed for winAim smilie rendering.  This
 * does not replace the text with the images.  It is used when sending out a
 * message to make sure that aol's im will be able to render the image.
 * @param message the message to fix
 * @param frontTag if we are already inside a font tag it is passed in. 
 */
void Chat::fontFixSmilies( QString &message, QString fontTag ){
  //If we dont have an already open fontTag, then we dont need to send the
  //close font tag before enclosing the smiley in its own font tags.
  QString preTag = fontTag.isEmpty() ? "" : "</FONT>";

  #define FONTTAG(x) \
    QString(preTag + "<FONT SIZE=\"3\">" + QString(x) + "</FONT>" + fontTag)

  // plug in fix so that aol users get smiles in font size 3 (so that it matches what we see)
  // please do not remove the extra fontag--winaim won't render reliably w/o it ;-)
  if( (message.contains('-') == 0) &&     //find a nose
      (message.contains(')') == 0) &&       //or a mouth
      (message.contains(':') == 0) )       //or some eyes.
    return;

  message.replace(QRegExp("=-[Oo]"), FONTTAG("=-O") );
  message.replace(QRegExp("&gt;:[Oo]"), FONTTAG(">:O") );
  message.replace(QRegExp("[Oo]:-)"), FONTTAG("O:-)") );
  message.replace(QRegExp("C:-)"), FONTTAG("C:-)") );

  // the "not" character gets counted in replace function..so it eats the one b4
  // message.replace(QRegExp("[^COo]:-?)"), FONTTAG(":-)") ); // :-) :)

  // wrap FONTTAG around smily w/o messing up the variations...
  int i = 0;
  while((i = message.find( QRegExp("[^COo]:-)"), i )) > 0)  {
    message.replace(i+1, 3, FONTTAG(":-)") );
    i = i + 29; // min length of FONTAG string macro
  }
  i = 0;
  while((i = message.find( QRegExp("[^COo]:)"), i )) > 0)  {
    message.replace(i+1, 2, FONTTAG(":)") );
    i = i + 29; // min length of FONTAG string macro
  }
  message.replace(QRegExp(":-?("), FONTTAG(":-(") ); // :-( :(
  message.replace(QRegExp(";-?)"), FONTTAG(";-)") ); // ;-) ;)
  message.replace(QRegExp(":-?P"), FONTTAG(":-P") ); // :-P :P

  message.replace(QRegExp(":-\\*"), FONTTAG(":-*") );
  message.replace(QRegExp("8-)"), FONTTAG("8-)") );
			
  message.replace(QRegExp("[Oo]-)"), FONTTAG("O-)") );
			
  message.replace(QRegExp(":-\\$"), FONTTAG(":-$") );
  message.replace(QRegExp(":-!"), FONTTAG(":-!") );
  message.replace(QRegExp(":-\\["), FONTTAG(":-[") );
			
  message.replace(QRegExp(":-\\"), FONTTAG(":-/") );
  message.replace(QRegExp(":-/"), FONTTAG(":-/") );
  message.replace(QRegExp(":'("), FONTTAG(":'(") );
  message.replace(QRegExp(":-X"), FONTTAG(":-X") );
  message.replace(QRegExp(":-?D"), FONTTAG(":-D") ); // :-D :D
  #undef FONTTAG
}

/**
 * Convert smilies text into images (via img tags)
 * @param message the message to change. 
 */
void Chat::replaceSmiliesWithImgTags( QString &message ){
  #define FACE(x) \
    QString("<img width=\"19\" height=\"19\" src=\"" + QString(DATADIR) + "/pics/face_" + QString(x) + ".png\">")

  /* This first if is to check if a smiley is contained in the message.  It
   * allows us to skip the big mess of message.replace's if there are no
   * smilies, resulting in (a little) faster performance.  All the smilies
   * contain either a '-' (nose) or a ':' (eyes) or a ')' (mouth)
   */
  if((message.contains('-') == 0) &&    //find a nose
     (message.contains(')') == 0) &&     //or a mouth
     (message.contains(':') == 0))      //or some eyes.
    return;

  // tokenize odd combos that might be mistaken for smilies
  // see untokenize below
  message.replace(QRegExp("&gt;)"), "&gt;%29"); // when user types >)

  message.replace(QRegExp("=-[Oo]"), FACE("scream") );
  message.replace(QRegExp("&gt;:[Oo]"), FACE("yell") );
  message.replace(QRegExp("[Oo]:-)"), FACE("angel") );
  message.replace(QRegExp("C:-)"), FACE("luke") );

  message.replace(QRegExp(":-?)"), FACE("smile") ); // :-) :)

  message.replace(QRegExp(":-?("), FACE("sad") ); // :-( :(
  message.replace(QRegExp(";-?)"), FACE("wink") ); // ;-) ;)
  message.replace(QRegExp(":-?P"), FACE("tongue") ); // :-P :P

  message.replace(QRegExp(":-\\*"), FACE("kiss") );
  message.replace(QRegExp("8-)"), FACE("glasses") );

  message.replace(QRegExp("[Oo]-)"), FACE("oneeye") );

  message.replace(QRegExp(":-\\$"), FACE("moneymouth") );
  message.replace(QRegExp(":-!"), FACE("burp") );
  message.replace(QRegExp(":-\\["), FACE("embarrassed") );

  message.replace(QRegExp(":-\\"), FACE("think") );
  message.replace(QRegExp(":-/"), FACE("think") );
  message.replace(QRegExp(":'("), FACE("cry") );
  message.replace(QRegExp(":-X"), FACE("crossedlips") );
  message.replace(QRegExp(":-?D"), FACE("bigsmile") ); // :-D :D

  // UNtokenize odd combos that might be mistaken for smilies
  message.replace(QRegExp("&gt;%29"), "&gt;)"); // when user types >)

  #undef FACE
}

// chat.cpp



