/**************************************************************************
                          				setup.cpp
                             -------------------
    begin                : Thu Feb 17 2000
    copyright            : (C) 2000 by Benjamin Meyer
    email                : benjamin@csh.rit.edu
 ***************************************************************************/

#include "setup.h"
#include "network/aim.h"
#include "application.h"
#include <qprogressdialog.h>
#include <qfont.h>
#include <kprocess.h>
#include <stdlib.h>
#include "preferences.h"
#include <kconfig.h>
#include <kglobal.h>
#include <qmultilineedit.h>
#include <qdir.h>
#include <qmessagebox.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qstring.h>

//The default profile for new users, or users with no profile set yet.
#define DEFAULT_PROFILE "<P>Visit the Kinkatta website at: <A HREF=\"http://kinkatta.sourceforge.net/\">http://kinkatta.sourceforge.net/</A>.</P>"

#define KINKATTA_DIR (KDEHOMEDIR +  "/share/apps/kinkatta/").latin1()
#define QT_KINKATTA_DIR "/.kaim"

setup *__instance = 0;

/***************************************************************************
 * Instance.  This is a static method that returns the current reference to
 * setup.  This is so we can avoid creating/deleting setup's all over the
 * place, and we dont need static methods to do simple things.
 ***************************************************************************/
setup *setup::instance(void){
  if( !__instance )
    __instance = new setup();
  return __instance;
}

/***************************************************************************
 * This is just a shortcut from having to write
 * setup::instance()->i_currentSettings every time
 ***************************************************************************/
settings_t *setup::settings(void){
  return setup::instance()->i_currentSettings;
}

/**
 * Constructor
 **/
setup::setup(){
  KDEHOMEDIR = getenv("KDEHOME");
  if (KDEHOMEDIR.length() == 0)
    KDEHOMEDIR = QDir::homeDirPath() + "/.kde/";

  i_currentSettings = new settings_t;
  
  // Set up basic defaults
  i_currentSettings->Version = release_number;
  i_currentSettings->ReleaseDate = release_date;

  i_currentSettings->buddylistFontSettings = new buddylist_Font_Settings;
  i_currentSettings->mapAwayMessages = new QMap<QString, QString>;
}

/***************************************************************************
 * Deconstructor
 ***************************************************************************/
setup::~setup(){
  delete i_currentSettings->buddylistFontSettings;
  delete i_currentSettings->mapAwayMessages;

  delete i_currentSettings;
}

/***************************************************************************
 * Returns a list of all the registered users of kinkatta. By this, it means all
 * users that have a settings file under ~/.kde/share/apps/kinkatta/
 ***************************************************************************/
QStringList setup::registeredUsers(){
	QString buddy = "";
	QString extension = "";
	QFileInfo fileInfo;
	QStringList l;

	QString homeDirr = KINKATTA_DIR;
	if(!QFile::exists(homeDirr)) {
		return l; //no registered users, return empty StringList
	}

  QDir home(homeDirr);
  home.setFilter( QDir::Files | QDir::NoSymLinks );

  if ( home.exists() ){
   for (unsigned int i = 0; i < home.count(); i++){
			fileInfo.setFile(home[i]);
      buddy = fileInfo.baseName().lower();
			extension = fileInfo.extension( FALSE );
      if ( (extension == "xml") && buddy != "settings"){ //settings.xml is the global settings file
				if(l.contains(buddy) == 0)  //make them unique
					l.append(buddy);
      }
    }
  }
	return l;
}

/***************************************************************************
 * Returns whether or not the passed user has used kinkatta before (ie, if he is
 * in the list returned by setup::registeredUsers())
 ***************************************************************************/
bool setup::isRegisteredUser(QString &user){
	if( setup::registeredUsers().contains(user) != 0)
		return TRUE; //has used kinkatta, is registered
	else
		return FALSE; //has not used kinkatta, is not registered
}

/***************************************************************************
 * Save the settings to the file
 ***************************************************************************/
void setup::save(){
  // int winX = this->x();
  // int winY = this->y();
	//qDebug("::save() i_currentSettings->login_name %s\n", i_currentSettings->login_name.latin1());
	/*
  if (i_currentSettings->login_name == QString("<new user>"))
    return;
	*/

	QString kinkattaDirStr = KINKATTA_DIR;
  QString settingsFile = kinkattaDirStr + "settings"; //the old settings file

  QString settingsFileXml =  kinkattaDirStr + "settings.xml";

	//first we remove the settings.xml file.  this is so that old/unused settings
	//from things we are not using anymore do not get parsed by
	//Preferences::readData() (which is called from Preferences::Preferences()).
	//Our format does not change too often now, but sometimes it does, so it does
	//not hurt.
	QFile::remove(settingsFileXml);

	Preferences prefs(settingsFileXml, QString("kinkatta prefs"), QString("1.0"));
	prefs.setGroup("sound");
	prefs.setNumber("SoundMode", i_currentSettings->SoundMode);
	prefs.setBool("sound_enable_login", i_currentSettings->sound_enable_login);
	prefs.setBool("sound_enable_logout", i_currentSettings->sound_enable_logout);
	prefs.setString("sound_file_command", i_currentSettings->sound_file_command);
	prefs.setString("sound_file_login", i_currentSettings->sound_file_login);
	prefs.setString("sound_file_logout", i_currentSettings->sound_file_logout);

	prefs.setGroup("network");
	prefs.setString("login_server_toc", i_currentSettings->login_server_toc );
	prefs.setString("login_server_login", i_currentSettings->login_server_login);
	prefs.setString("login_port_toc", i_currentSettings->login_port_toc);
	prefs.setString("login_port_login", i_currentSettings->login_port_login );
	prefs.setBool("login_enable_auto", i_currentSettings->login_enable_auto);
	prefs.setString("login_name", i_currentSettings->login_name );
	prefs.setNumber("proxy_Choice", i_currentSettings->proxy_Choice);
	prefs.setString("proxy_server" , i_currentSettings->proxy_server );
	prefs.setString("proxy_port" , i_currentSettings->proxy_port );

	prefs.setGroup("away");

	prefs.setBool("away_comeBackOnlyOnBackClick", i_currentSettings->away_comeBackOnlyOnBackClick );
	prefs.setBool("away_disable_sound", i_currentSettings->away_disable_sound );
	prefs.setBool("away_enable_auto", i_currentSettings->away_enable_auto );
	prefs.setNumber("away_auto_delay_minutes", i_currentSettings->away_auto_delay_minutes);
	prefs.setString("away_auto_default_message", i_currentSettings->away_auto_default_message);
	prefs.setBool("ignore_automated_get_away", i_currentSettings->ignore_automated_get_away );

	prefs.setGroup("general");
	prefs.setBool("main_resizeHeight", i_currentSettings->main_resizeHeight);
	prefs.setNumber("main_resizeHeightMax", i_currentSettings->main_resizeHeightMax);
	prefs.setNumber("main_resizeHeightMin", i_currentSettings->main_resizeHeightMin);
	prefs.setBool("main_resizeWidth", i_currentSettings->main_resizeWidth);
	prefs.setNumber("main_resizeWidthMax", i_currentSettings->main_resizeWidthMax);
	prefs.setNumber("main_resizeWidthMin", i_currentSettings->main_resizeWidthMin);
	prefs.setBool("main_loginMoveRight", i_currentSettings->main_loginMoveRight);
	prefs.setBool("main_loginMoveLeft", i_currentSettings->main_loginMoveLeft);
	prefs.setBool("main_loginMoveNo", i_currentSettings->main_loginMoveNo);
	prefs.setBool("main_showIdle", i_currentSettings->main_showIdle);
	prefs.setBool("main_showLastOn", i_currentSettings->main_showLastOn);
	prefs.setBool("main_noIcons", i_currentSettings->main_noIcons);
	prefs.setNumber("main_moveDown", i_currentSettings->main_moveDown);
	prefs.setNumber("main_idleLayout", i_currentSettings->main_idleLayout);
	prefs.setBool("main_noHeader", i_currentSettings->main_noHeader);
	prefs.setBool("main_useX11ForIdle", i_currentSettings->main_useX11ForIdle);
	prefs.setBool("main_showTraySignonTooltip", i_currentSettings->main_showTraySignonTooltip);
	prefs.setNumber("main_traySignonTooltipPosX",
		i_currentSettings->main_traySignonTooltipPosX);
	prefs.setNumber("main_traySignonTooltipPosY",
		i_currentSettings->main_traySignonTooltipPosY);
	prefs.setNumber("main_traySignonTooltipAnchorPosition",
		i_currentSettings->main_traySignonTooltipAnchorPosition);
	prefs.setBool("main_mute_on_login", i_currentSettings->main_mute_on_login);

	prefs.setNumber("xSize", i_currentSettings->xSize);
	prefs.setNumber("ySize", i_currentSettings->ySize);

	prefs.setGroup("chat");

  	prefs.setFont("buddylist_OnlineFont", i_currentSettings->buddylistFontSettings->buddylist_OnlineFont);
	prefs.setFont("buddylist_OfflineFont", i_currentSettings->buddylistFontSettings->buddylist_OfflineFont);
	prefs.setFont("buddylist_SignOnFont", i_currentSettings->buddylistFontSettings->buddylist_SignOnFont);

  prefs.setString("IgnoreThisUpdate", i_currentSettings->IgnoreThisUpdate);

	prefs.setGroup("away list");
	QMap <QString,QString>::Iterator it;
	int i = 1;
	for(i = 1; i <= prefs.getNumber("away_message_count"); i++){
		prefs.removeKey(QString("away_message_%1").arg(i));
		//prefs.removeKey(QString("away_message_text%1").arg(i));
	}
	for(i = 1, it = i_currentSettings->mapAwayMessages->begin();
				it != i_currentSettings->mapAwayMessages->end(); ++it, ++i){
		prefs.setString(QString("away_message_%1").arg(i), it.key());
		prefs.setAttr1(QString("away_message_%1").arg(i), it.data());
		//prefs.setString(QString("away_message_text%1").arg(i), it.latin1());
	}
	prefs.setNumber("away_message_count", i-1); //i-1 because of the last ++i
	prefs.flush();

}

/******************************************************************************
 * Returns a string (in "config" format) of the passed user's buddy list
 * (in other words, converts the xml buddy list to config format)
 * this is only a helper function for the below function
 ******************************************************************************/
QString setup::buddyListFileToString(const QString &file){
  if (file.isEmpty())
		return QString("");

	TBuddyList buddyList, permitList, denyList;
	int status;

	parseBuddyListFile(file, &buddyList, &permitList, &denyList, &status);

	return buddyListToString(&buddyList, &permitList, &denyList, status);

}

/******************************************************************************
 * Returns a string (in "config" format) of the passed user's buddy list
 * (in other words, converts the xml buddy list to config format)
 * this is only a helper function for the below function
 ******************************************************************************/
QString setup::buddyListToString(const QString &user){
  if (user.isEmpty() || user == QString("<new user>"))
		return QString("");

	TBuddyList buddyList, permitList, denyList;
	int status;

	parseBuddyList(user, &buddyList, &permitList, &denyList, &status);

	return buddyListToString(&buddyList, &permitList, &denyList, status);

}

/******************************************************************************
 * Returns a string (in "config" format) of the passed user's buddy list
 * (in other words, converts the xml buddy list to config format)
 ******************************************************************************/
QString setup::buddyListToString(const TBuddyList *buddyList, const TBuddyList *permitList, const TBuddyList *denyList, int permitStatus){
  QString data;
  data.sprintf("m %1i\n", permitStatus);

  int i = -1;

  for (int group =0; group < buddyList->getCountGroup(); group++){
    data += "g " + buddyList->getNameGroup(group) + "\n";

   // int currentGroup = buddyList->getGroup(group);
    i = -1;
    while(++i < buddyList->getCount()){
      if(group == buddyList->getGroup(i))
         data += "b " + buddyList->getName(i) + ":" + buddyList->getAlias(i) + "\n";
      }
    }


  i = -1;
  while(++i < permitList->getCount())
  {
    data += "p " + permitList->getName(i) + "\n";
  }
  i = -1;
  while(++i < denyList->getCount())
  {
    data += "d " + denyList->getName(i) + "\n";
  }

	//qDebug("config :%s\n", data.latin1());
	return data;

}

/******************************************************************************
 * Saves the config
 ******************************************************************************/
void setup::saveBuddyList(const QString &user, const TBuddyList *buddyList, const TBuddyList *permitList, const TBuddyList *denyList, int permitStatus){
  if (user.isEmpty() || user == QString("<new user>"))
    return;

  QString settingsFileXml = KINKATTA_DIR;
  settingsFileXml += user.lower();
  settingsFileXml += ".xml";

	TBuddy buddy;
	Preferences prefs(settingsFileXml, QString("kinkatta user prefs"), QString("1.0"));
	prefs.setGroup("buddy list");

	prefs.setNumber( "permit status", permitStatus );
	int buddyCount = 0;
	int group = 0;

	//first reset the list to remove any deleted options
	for(int i =1; i <= prefs.getNumber("group_count"); i++){
		prefs.removeKey( QString("group_%1").arg(i) );
	}
	//first reset the list to remove any deleted options
	for(int i =1; i <= prefs.getNumber("buddy_count"); i++){
		prefs.removeKey( QString("buddy_%1").arg(i) );
	}

	group = buddyCount = 0;

	for (group =0; group < buddyList->getCountGroup(); group++){
		prefs.setString( QString("group_%1").arg(group + 1),
					buddyList->getNameGroup(group));

		for(int i = 0; i < buddyList->getCount(); i++){
			if(group == buddyList->getGroup(i)){
				buddyCount++;
				prefs.setString( QString("buddy_%1").arg(buddyCount), buddyList->getName(i) );
				prefs.setAttr1( QString("buddy_%1").arg(buddyCount), buddyList->getNameGroup(group) );
				prefs.setAttr2( QString("buddy_%1").arg(buddyCount), "buddy" );
				if( !buddyList->getAlias(i).isEmpty() )
					prefs.setAttr3( QString("buddy_%1").arg(buddyCount), buddyList->getAlias(i) );
			}
		}
	}
	prefs.setNumber( "group_count", group );

	buddyCount++; //fixes an offset problem
	for(int i = 0; i < permitList->getCount(); i++, buddyCount++){
		prefs.setString( QString("buddy_%1").arg(buddyCount), permitList->getName(i) );
		//prefs.setAttr1( QString("buddy_%1").arg(buddyCount), permitList->getNameGroup(group) );
		prefs.setAttr2( QString("buddy_%1").arg(buddyCount), "permit" );
	}

	for(int i = 0; i < denyList->getCount(); i++, buddyCount++){
		prefs.setString( QString("buddy_%1").arg(buddyCount), denyList->getName(i) );
		//prefs.setAttr1( QString("buddy_%1").arg(buddyCount), denyList->getNameGroup(group) );
		prefs.setAttr2( QString("buddy_%1").arg(buddyCount), "deny" );
	}

	prefs.setNumber("buddy_count", buddyCount-1);//-1 for the offset problem

	prefs.flush();
}

/******************************************************************************
 * Reads the buddylist (helper function for below)
 ******************************************************************************/
void setup::parseBuddyList(const QString &user, TBuddyList *buddyList, TBuddyList *permitList, TBuddyList *denyList, int *permitStatus){
  if (user.isEmpty() || user == QString("<new user>"))
		return;

  QString settingsFileXml = KINKATTA_DIR;
  settingsFileXml += user.lower();
  settingsFileXml += ".xml";

	return parseBuddyListFile(settingsFileXml, buddyList, permitList, denyList, permitStatus);

}

/******************************************************************************
 * Reads the buddylist
 ******************************************************************************/
void setup::parseBuddyListFile(const QString &settingsFileXml, TBuddyList *buddyList, TBuddyList *permitList, TBuddyList *denyList, int *permitStatus){
	if( !QFile::exists(settingsFileXml) ){
		buddyList->addGroup("Buddies"); //default group, if there are none
		(*permitStatus) = 1;
		return;
	}

	TBuddy buddy;
	Preferences prefs(settingsFileXml, QString("kinkatta user prefs"), QString("1.0"));

	prefs.setGroup("buddy list");
	*permitStatus = prefs.getNumber("permit status", 1); //default to 1!  important!
	//qDebug(" %s defaulting to %d\n", user.latin1(), *permitStatus);

	for(int g = 1; g <= prefs.getNumber("group_count"); g++){
		QString groupStr = prefs.getString( QString("group_%1").arg(g) );
		int groupNumber = buddyList->addGroup( groupStr );
		if (groupNumber == -1)
			groupNumber = buddyList->getNumGroup(groupStr);
		for(int i = 1; i <= prefs.getNumber("buddy_count"); i++){
			buddy.status = TAIM_OFFLINE;
			buddy.lastOn = 0;
			buddy.name = prefs.getString( QString("buddy_%1").arg(i) );
			QString alias = prefs.getAttr3( QString("buddy_%1").arg(i),buddy.name );
			if( !alias.isEmpty() )
				buddy.alias = alias;

			// Easter Egg Developer names get put to their real names.
      if ( buddy.name.lower() == QString("icefox2") ){
				if (buddy.name == buddy.alias){
				  buddy.alias = "Ben Meyer";
				}
			}
      if ( buddy.name.lower() == QString("remo177") ){
				if (buddy.name == buddy.alias){
				  buddy.alias = "Dimi Shahbaz";
				}
			}
      if ( buddy.name.lower() == QString("jimwinme") ){
				if (buddy.name == buddy.alias){
				  buddy.alias = "Jim Wilson";
				}
			}
        if( buddy.name.lower() == QString("Dr HP Baldy") )
          if( buddy.name == buddy.alias)
            buddy.alias = "Adam Keys";

			QString type = prefs.getAttr2( QString("buddy_%1").arg(i));
			if( type == "permit" ){
				permitList->add(&buddy);
			} else if (type == "deny" ){
				denyList->add(&buddy);
			} else { // type = "buddy"
				if(prefs.getAttr1( QString("buddy_%1").arg(i) ) == groupStr){
					buddy.group = groupNumber; //prefs.getAttr1( QString("buddy_%1").arg(i) );
					buddyList->add(&buddy);
				}
			}

		}
	}

	if(buddyList->getCountGroup() == 0){
		buddyList->addGroup("Buddies"); //default group, if there are none
	}
}

/***************************************************************************
 * Remove an entire group of settings (ie, "sound")
 ***************************************************************************/
void setup::removeGroup(QString group){
  QString kinkattaDirStr = KINKATTA_DIR;
  QString settingsFileXml = kinkattaDirStr;
  settingsFileXml +=  "settings.xml";
  Preferences prefs(settingsFileXml, QString("kinkatta prefs"), QString("1.0"));
  prefs.setGroup(group);
  prefs.removeGroup();
}

/***************************************************************************
 * Reset the default settings.
 ***************************************************************************/
void setup::resetDefaults(){
  removeGroup("away");
  //removeGroup("away list");
  removeGroup("chat");
  removeGroup("general");
  removeGroup("network");
  removeGroup("sound");

  read();
}

/***************************************************************************
 * Read the settings from the file
 ***************************************************************************/
void setup::read(){
  KConfig &config = *KGlobal::config();
  config.setGroup("chatwindow");

  int i, r, g, b;
  bool convertoldfiles = false;
  //Get File Name
  //qDebug("setup::read() i_currentSettings->login_name %s\n", i_currentSettings->login_name.latin1());

  QString kinkattaDirStr = KINKATTA_DIR;
  QString qtkinkattaDir;
  QDir kinkattaDir(kinkattaDirStr);
  //Make Dir
  bool madeKinkattaDir = false;
  if( !kinkattaDir.exists() ){
  // if no kdeDir and qt dir (~/.kinkatta) exists then copy settings to kdeDir
	qtkinkattaDir = QT_KINKATTA_DIR;
	if(QFile::exists(qtkinkattaDir)) {
		// copy qt settings to kdedir
		KShellProcess *shell = new KShellProcess();
		*shell << "cp";
		*shell << "-r";
		*shell << qtkinkattaDir.latin1();
		*shell << kinkattaDirStr.latin1();
		shell->start( KProcess::Block );
	} else {
	    madeKinkattaDir = true;
	    kinkattaDir.mkdir(kinkattaDirStr, true);
	}
  }

  QString settingsFileXml = kinkattaDirStr;
  settingsFileXml +=  "settings.xml";

	if(!QFile::exists(settingsFileXml)){ //both xml and old prefs files dont exist
		//There is a .kinkatta folder but no xml file, convert
	  //XXX: temporary
	  if (madeKinkattaDir == false){
                        // set flag so conversion can be done after defaults load
                        convertoldfiles = true;
		}
	}
	Preferences prefs(settingsFileXml, QString("kinkatta prefs"), QString("1.0"));

	r = g = b = 0;

	prefs.setGroup("sound");
	i_currentSettings->SoundMode= prefs.getNumber("SoundMode", 1);
	i_currentSettings->sound_enable_login	= prefs.getBool("sound_enable_login", true);
	i_currentSettings->sound_enable_logout= prefs.getBool("sound_enable_logout", false );
	i_currentSettings->sound_file_command= prefs.getString("sound_file_command", "artsplay" );
	i_currentSettings->sound_file_login= prefs.getString("sound_file_login", DATADIR "/sounds/BuddyArrive.au");
	i_currentSettings->sound_file_logout= prefs.getString("sound_file_logout",  DATADIR "/sounds/BuddyLeave.au");

	prefs.setGroup("network");
	i_currentSettings->login_server_toc = prefs.getString("login_server_toc", "toc.oscar.aol.com");
	i_currentSettings->login_server_login= prefs.getString("login_server_login", "login.oscar.aol.com");
	i_currentSettings->login_port_toc= prefs.getString("login_port_toc", "9898");
	i_currentSettings->login_port_login= prefs.getString("login_port_login", "5190");
	i_currentSettings->login_enable_auto= prefs.getBool("login_enable_auto", false);
	i_currentSettings->login_name = prefs.getString("login_name", "<new user>");
	i_currentSettings->proxy_Choice= prefs.getNumber("proxy_Choice", 0);
	i_currentSettings->proxy_server = prefs.getString("proxy_server", "");
	i_currentSettings->proxy_port = prefs.getString("proxy_port", "");

	prefs.setGroup("away");
	i_currentSettings->away_enable_auto= prefs.getBool("away_enable_auto" , false);
	i_currentSettings->away_comeBackOnlyOnBackClick= prefs.getBool("away_comeBackOnlyOnBackClick" , false);
	i_currentSettings->away_disable_sound= prefs.getBool("away_disable_sound" , false);
	i_currentSettings->away_auto_delay_minutes= prefs.getNumber("away_auto_delay_minutes", 30);
	i_currentSettings->away_auto_default_message= prefs.getString("away_auto_default_message", "Away");
	i_currentSettings->ignore_automated_get_away= prefs.getBool("ignore_automated_get_away", true);

	prefs.setGroup("general");
	i_currentSettings->main_resizeHeight= prefs.getBool("main_resizeHeight", false);
	i_currentSettings->main_resizeHeightMax= prefs.getNumber("main_resizeHeightMax", 600);
	i_currentSettings->main_resizeHeightMin= prefs.getNumber("main_resizeHeightMin", 200);
	i_currentSettings->main_resizeWidth= prefs.getBool("main_resizeWidth", true);
	i_currentSettings->main_resizeWidthMax= prefs.getNumber("main_resizeWidthMax", 300);
	i_currentSettings->main_resizeWidthMin= prefs.getNumber("main_resizeWidthMin", 50);
	i_currentSettings->main_loginMoveRight= prefs.getBool("main_loginMoveRight", false);
	i_currentSettings->main_loginMoveLeft= prefs.getBool("main_loginMoveLeft", false);
	i_currentSettings->main_loginMoveNo	= prefs.getBool("main_loginMoveNo", true);
	i_currentSettings->main_showIdle= prefs.getBool("main_showIdle", true);
	i_currentSettings->main_showLastOn= prefs.getBool("main_showLastOn", false);
	i_currentSettings->main_noIcons= prefs.getBool("main_noIcons", false);
	i_currentSettings->main_moveDown= prefs.getNumber("main_moveDown", 0);
	i_currentSettings->main_idleLayout= prefs.getNumber("main_idleLayout", 0);
	i_currentSettings->main_noHeader= prefs.getBool("main_noHeader", false);
	i_currentSettings->main_useX11ForIdle= prefs.getBool("main_useX11ForIdle", false );
	i_currentSettings->main_showTraySignonTooltip= 
				prefs.getBool("main_showTraySignonTooltip", true );
	i_currentSettings->main_traySignonTooltipPosX= 
				prefs.getNumber("main_traySignonTooltipPosX", 0 );
	i_currentSettings->main_traySignonTooltipPosY= 
				prefs.getNumber("main_traySignonTooltipPosY", 0 );
	i_currentSettings->main_traySignonTooltipAnchorPosition= 
				prefs.getNumber("main_traySignonTooltipAnchorPosition", 0 );
	i_currentSettings->main_mute_on_login= prefs.getBool("main_mute_on_login", true );

	i_currentSettings->xSize= prefs.getNumber("xSize", -1);
	i_currentSettings->ySize= prefs.getNumber("ySize", -1);

	prefs.setGroup("chat");
  bool chat_raise_window;         // Done
  bool chat_flash_on_new_message; // Done
  bool chat_log;                  // Done
  bool chat_log_inHtml;           // Done
  bool chat_show_time;            // Done
  bool chat_log_time;             // Done
  bool chat_return_send;	  			// Done
  bool chat_sreturn_send;         // Done
  bool chat_send_blank;           // Done
  bool chat_ignore_contacthtml;   // Done
  bool chat_line_limit;           // Done
  bool chat_show_smilies;         // Done
  int chat_line_limit_number;     // Done
  bool chat_log_singleFile;

chat_raise_window= config.readBoolEntry("chat_raise_window", true);
chat_flash_on_new_message = config.readBoolEntry("chat_flash_on_new_message", true);
chat_log = config.readBoolEntry("chat_log", true);
chat_log_inHtml= config.readBoolEntry("chat_log_inHtml", true);
chat_show_time = config.readBoolEntry("chat_show_time", true);
chat_log_time= config.readBoolEntry("chat_log_time", true);
chat_return_send= config.readBoolEntry("chat_return_send", true);
chat_sreturn_send= config.readBoolEntry("chat_sreturn_send", false);
chat_send_blank= config.readBoolEntry("chat_send_blank", false);
chat_ignore_contacthtml= config.readBoolEntry("chat_ignore_buddyhtml", false);
chat_show_smilies= config.readBoolEntry("chat_show_smilies", true);
chat_line_limit= config.readBoolEntry("chat_line_limit", false);
chat_line_limit_number= config.readNumEntry("chat_line_limit_number",200 );
chat_log_singleFile= config.readBoolEntry("chat_log_singleFile", false);

chat_raise_window= prefs.getBool("chat_raise_window", chat_raise_window);
chat_flash_on_new_message = prefs.getBool("chat_flash_on_new_message", chat_flash_on_new_message);
chat_log = prefs.getBool("chat_log", chat_log);
chat_log_inHtml= prefs.getBool("chat_log_inHtml", chat_log_inHtml);
chat_show_time = prefs.getBool("chat_show_time", chat_show_time);
chat_log_time= prefs.getBool("chat_log_time", chat_log_time);
chat_return_send= prefs.getBool("chat_return_send", chat_return_send);
chat_sreturn_send= prefs.getBool("chat_sreturn_send", chat_sreturn_send);
chat_send_blank= prefs.getBool("chat_send_blank", chat_send_blank);
chat_ignore_contacthtml= prefs.getBool("chat_ignore_buddyhtml", chat_ignore_contacthtml);
chat_show_smilies= prefs.getBool("chat_show_smilies", chat_show_smilies);
chat_line_limit= prefs.getBool("chat_line_limit", chat_line_limit);
chat_line_limit_number= prefs.getNumber("chat_line_limit_number",chat_line_limit_number );
chat_log_singleFile= prefs.getBool("chat_log_singleFile", chat_log_singleFile);

  config.writeEntry("chat_raise_window", chat_raise_window);
  config.writeEntry("chat_flash_on_new_message", chat_flash_on_new_message);
  config.writeEntry("chat_log", chat_log);
  config.writeEntry("chat_log_inHtml",chat_log_inHtml );
  config.writeEntry("chat_log_singleFile", chat_log_singleFile);
  config.writeEntry("chat_show_time", chat_show_time);
  config.writeEntry("chat_log_time", chat_log_time);
  config.writeEntry("chat_return_send", chat_return_send);
  config.writeEntry("chat_sreturn_send", chat_sreturn_send);
  config.writeEntry("chat_send_blank", chat_send_blank);
  config.writeEntry("chat_ignore_contacthtml", chat_ignore_contacthtml);
  config.writeEntry("chat_show_smilies", chat_show_smilies);
  config.writeEntry("chat_line_limit", chat_line_limit);
  config.writeEntry("chat_line_limit_number", chat_line_limit_number);
  config.writeEntry("chat_log", chat_log);

  QColor chat_yourText_color;     // Done
  QColor chat_yourName_color;     // Done
  QColor chat_contactText_color;  // Done
  QColor chat_contactName_color;  // Done
  QColor chat_yourbg_color;       // Done
  QColor chat_contactbg_color;    // Done

  r = config.readNumEntry("chat_yourText_color.R", 0);
  g = config.readNumEntry("chat_yourText_color.G", 0);
  b = config.readNumEntry("chat_yourText_color.B", 0);
  r = prefs.getNumber("chat_yourText_color.R", r);
  g = prefs.getNumber("chat_yourText_color.G", g);
  b = prefs.getNumber("chat_yourText_color.B", b);
  chat_yourText_color.setRgb(r, g, b);

  r = config.readNumEntry("chat_yourName_color.R", 0);
  g = config.readNumEntry("chat_yourName_color.G", 60);
  b = config.readNumEntry("chat_yourName_color.B", 128);
  r = prefs.getNumber("chat_yourName_color.R", r);
  g = prefs.getNumber("chat_yourName_color.G", g);
  b = prefs.getNumber("chat_yourName_color.B", b);
  chat_yourName_color.setRgb(r, g, b);

  r = config.readNumEntry("chat_buddyText_color.R", 0);
  g = config.readNumEntry("chat_buddyText_color.G", 0);
  b = config.readNumEntry("chat_buddyText_color.B", 0);
  r = prefs.getNumber("chat_buddyText_color.R", r);
  g = prefs.getNumber("chat_buddyText_color.G", g);
  b = prefs.getNumber("chat_buddyText_color.B", b);
  chat_contactText_color.setRgb(r, g, b);

  r = config.readNumEntry("chat_buddyName_color.R", 128);
  g = config.readNumEntry("chat_buddyName_color.G", 0);
  b = config.readNumEntry("chat_buddyName_color.B", 0);
  r = prefs.getNumber("chat_buddyName_color.R", r);
  g = prefs.getNumber("chat_buddyName_color.G", g);
  b = prefs.getNumber("chat_buddyName_color.B", b);
  chat_contactName_color.setRgb(r, g, b);

  r = config.readNumEntry("chat_buddyBg_color.R", 255);
  g = config.readNumEntry("chat_buddyBg_color.G", 255);
  b = config.readNumEntry("chat_buddyBg_color.B", 255);
  r = prefs.getNumber("chat_buddyBg_color.R", r);
  g = prefs.getNumber("chat_buddyBg_color.G", g);
  b = prefs.getNumber("chat_buddyBg_color.B", b);
  chat_contactbg_color.setRgb(r, g, b);

  r = config.readNumEntry("chat_yourBg_color.R", 255);
  g = config.readNumEntry("chat_yourBg_color.G", 255);
  b = config.readNumEntry("chat_yourBg_color.B", 255);
  r = prefs.getNumber("chat_yourBg_color.R", r);
  g = prefs.getNumber("chat_yourBg_color.G", g);
  b = prefs.getNumber("chat_yourBg_color.B", b);
  chat_yourbg_color.setRgb(r, g, b);

  config.writeEntry( "chat_yourText_color", chat_yourText_color );
  config.writeEntry( "chat_yourName_color", chat_yourName_color );
  config.writeEntry( "chat_contactText_color", chat_contactText_color );
  config.writeEntry( "chat_contactName_color", chat_contactName_color );
  config.writeEntry( "chat_yourbg_color", chat_yourbg_color );
  config.writeEntry( "chat_contactbg_color", chat_contactbg_color );

  QFont defaultFont = qApp->font();
  i_currentSettings->buddylistFontSettings->buddylist_OnlineFont=prefs.getFont("buddylist_OnlineFont", defaultFont);
  i_currentSettings->buddylistFontSettings->buddylist_OfflineFont=prefs.getFont("buddylist_OfflineFont", defaultFont);
  i_currentSettings->buddylistFontSettings->buddylist_SignOnFont=prefs.getFont("buddylist_SignOnFont", defaultFont);
  
  //for chat_*Font, we need the default to be 12 points.
  defaultFont.setPointSize(12);
	
  // chatwindow updates!
  QFont chat_yourFont = config.readFontEntry( "chat_yourFont", &defaultFont );
  QFont chat_contactFont = config.readFontEntry( "chat_contactFont", &defaultFont );

  chat_yourFont=prefs.getFont("chat_yourFont", chat_yourFont);
  chat_contactFont=prefs.getFont("chat_buddyFont", chat_contactFont);
  
  config.writeEntry( "chat_yourFont", chat_yourFont );
  config.writeEntry( "chat_contactFont", chat_contactFont );

  bool chat_sound;                // Done
  bool chat_sound_beepInitial;    // Done
  bool chat_sound_beepReceive;    // Done
  bool chat_sound_beepSend;       // Done
  bool chat_sound_playFile;       // Done
  QString chat_sound_fileSend;    // Done
  QString chat_sound_fileReceive; // Done
  QString chat_sound_fileInitial; // Done

  chat_sound= config.readBoolEntry("chat_sound_enable", true);
  chat_sound_beepInitial= config.readBoolEntry("chat_sound_beepInitial", true);
  chat_sound_beepReceive= config.readBoolEntry("chat_sound_beepReceive", true);
  chat_sound_beepSend= config.readBoolEntry("chat_sound_beepSend", true);
  chat_sound_playFile = config.readBoolEntry("chat_sound_playFile", false);
  chat_sound_fileSend= config.readEntry("chat_sound_fileSend", DATADIR "/sounds/Send.au");
  chat_sound_fileReceive= config.readEntry("chat_sound_fileReceive", DATADIR "/sounds/Receive.au");
  chat_sound_fileInitial= config.readEntry("chat_sound_fileInitial", DATADIR "/sounds/Receive.au");

  chat_sound= prefs.getBool("chat_sound_enable", chat_sound);
  chat_sound_beepInitial= prefs.getBool("chat_sound_beepInitial", chat_sound_beepInitial);
  chat_sound_beepReceive= prefs.getBool("chat_sound_beepReceive",chat_sound_beepReceive );
  chat_sound_beepSend= prefs.getBool("chat_sound_beepSend", chat_sound_beepSend);
  chat_sound_playFile = prefs.getBool("chat_sound_playFile", chat_sound_playFile);
  chat_sound_fileSend= prefs.getString("chat_sound_fileSend", chat_sound_fileSend);
  chat_sound_fileReceive= prefs.getString("chat_sound_fileReceive", chat_sound_fileReceive);
  chat_sound_fileInitial= prefs.getString("chat_sound_fileInitial", chat_sound_fileInitial);

  config.writeEntry("chat_sound", chat_sound);
  config.writeEntry("chat_sound_beepInitial", chat_sound_beepInitial);
  config.writeEntry("chat_sound_beepReceive", chat_sound_beepReceive);
  config.writeEntry("chat_sound_beepSend", chat_sound_beepSend);
  config.writeEntry("chat_sound_playFile", chat_sound_playFile);
  config.writeEntry("chat_sound_fileSend", chat_sound_fileSend);
  config.writeEntry("chat_sound_fileReceive", chat_sound_fileReceive);
  config.writeEntry("chat_sound_fileInitial", chat_sound_fileInitial);

	i_currentSettings->IgnoreThisUpdate= prefs.getString("IgnoreThisUpdate", i_currentSettings->ReleaseDate);

	prefs.setGroup("away list");
	i_currentSettings->mapAwayMessages->clear();
	for(i = 1; i <= prefs.getNumber("away_message_count"); i++){
	  i_currentSettings->mapAwayMessages->insert(
	    prefs.getString(QString("away_message_%1").arg(i)),
	    prefs.getAttr1(QString("away_message_%1").arg(i)) );
	}
  if (i_currentSettings->mapAwayMessages->count() == 0){
    i_currentSettings->mapAwayMessages->insert("Default Away",
      "I am sorry, but I am away from the computer right now.");

  }
}

void setup::saveProfile(const QString &user, const QString &profile){
	if(user.isEmpty() || user == QString("<new user>"))
		return;

  QString settingsFileXml = KINKATTA_DIR;
  settingsFileXml += user.lower();
  settingsFileXml += ".xml";

	Preferences prefs(settingsFileXml, QString("kinkatta user prefs"), QString("1.0"));

	prefs.setGroup("profile");
	prefs.setCDATA("profile", profile);

	prefs.flush();

}

/*
 * Decrypt a password.  Since the algorithm is circular, this is just a helper
 * function for the below.  It does the same thing.
 */
QString setup::decryptPassword(const QString &cryptedPass){
	return cryptPassword(cryptedPass);
}

QString setup::cryptPassword(const QString &plainPass){
  // The point of this "encryption" is not to accually encrypt the data, but
	// simply to not have the password in plain text
	QString crypted;
	int len = plainPass.length();
  for (int i = 0; i < len; i ++){
    int temp = plainPass[i].latin1();
		//qDebug("'%c' becomes ", temp);
		if( (temp >= 'a') && (temp <= 'm') )
			temp += 13;
		else if ( (temp >= 'A') && (temp <= 'M') )
			temp += 13;
		else if ( (temp >= 'n') && (temp <= 'z') )
			temp -= 13;
		else if ( (temp >= 'N') && (temp <= 'Z') )
			temp -= 13;
		else if ( (temp >= '0') && (temp <= '4') )//temp is a number
			temp += 5;
		else if ( (temp >= '5') && (temp <= '9') )
			temp -= 5;

		//qDebug(" '%c'\n", temp);
		crypted += (char)temp;
 	}

	return crypted;
}


void setup::savePassword(const QString &user, const QString &password){
	if(user.isEmpty() || (user == QString("<new user>")) || password.isEmpty())
		return;

	QString crypted = cryptPassword(password);

  QString settingsFileXml = KINKATTA_DIR;
  settingsFileXml += user.lower();
  settingsFileXml += ".xml";

	Preferences prefs(settingsFileXml, QString("kinkatta user prefs"), QString("1.0"));

	prefs.setGroup("password");
	prefs.setCDATA("password", crypted);

	prefs.flush();
}

/***************************************************************************
 * Save a user's list of pounces.
 ***************************************************************************/
void setup::savePounces(const QString &user, const QDict<pounce> &list){
	if(user.isEmpty() || (user == QString("<new user>")) )
		return;

//	if(list.count() == 0)
//		return;

  QString settingsFileXml = KINKATTA_DIR;
  settingsFileXml += user.lower();
  settingsFileXml += ".xml";

	Preferences prefs(settingsFileXml, QString("kinkatta user prefs"), QString("1.0"));

	prefs.setGroup("pounces");

	//first clear the list so outlying pounces do not get saved if they dont
	//exist.
	int i;
	int pcount = prefs.getNumber("pounce_count", 0);
	if (pcount > 0)
	for(i = 0; i <= pcount; i++){
		prefs.removeKey(QString("pounce_%1").arg(i) + "_buddyName");
		prefs.removeKey(QString("pounce_%1").arg(i) + "_signOn");
		prefs.removeKey(QString("pounce_%1").arg(i) + "_returnAway");
		prefs.removeKey(QString("pounce_%1").arg(i) + "_returnIdle");
		prefs.removeKey(QString("pounce_%1").arg(i) + "_openChat");
		prefs.removeKey(QString("pounce_%1").arg(i) + "_sendMessage");
		prefs.removeKey(QString("pounce_%1").arg(i) + "_message");
		prefs.removeKey(QString("pounce_%1").arg(i) + "_execCommand");
		prefs.removeKey(QString("pounce_%1").arg(i) + "_command");
		prefs.removeKey(QString("pounce_%1").arg(i) + "_doNotRemove");

		prefs.removeKey(QString("pounce_%1").arg(i) + "_hidden");
		prefs.removeKey(QString("pounce_%1").arg(i) + "_signOnSound");
		prefs.removeKey(QString("pounce_%1").arg(i) + "_signOffSound");
	  prefs.removeKey(QString("pounce_%1").arg(i) + "_ignoreSettings");
	}

	QDictIterator<pounce> it(list);
	pounce *p;
	i = 0;
	while( it.current() ){
		p = it.current();
		if (p->doNotRemove == true){
			prefs.setString(QString("pounce_%1").arg(i) + "_buddyName", p->buddyName);
			if (p->signOn == true)
			  prefs.setBool(QString("pounce_%1").arg(i) + "_signOn", p->signOn);
			if (p->returnAway == true)
			  prefs.setBool(QString("pounce_%1").arg(i) + "_returnAway", p->returnAway);
			if (p->returnIdle == true)
			  prefs.setBool(QString("pounce_%1").arg(i) + "_returnIdle", p->returnIdle);
			if (p->openChat == true)
			  prefs.setBool(QString("pounce_%1").arg(i) + "_openChat", p->openChat);
			if (p->sendMessage == true)
			  prefs.setBool(QString("pounce_%1").arg(i) + "_sendMessage", p->sendMessage);
			if ((p->message).length() > 0)
				prefs.setString(QString("pounce_%1").arg(i) + "_message", p->message);
			if (p->execCommand == true)
			  prefs.setBool(QString("pounce_%1").arg(i) + "_execCommand", p->execCommand);
			if ((p->command).length() > 0)
			  prefs.setString(QString("pounce_%1").arg(i) + "_command", p->command);

			if(p->hidden == true){
				prefs.setBool(QString("pounce_%1").arg(i) + "_hidden", p->hidden);
			  prefs.setString(QString("pounce_%1").arg(i) + "_signOnSound", p->signOnSound);
			  prefs.setString(QString("pounce_%1").arg(i) + "_signOffSound", p->signOffSound);
			  if(p->ignoreSettings == true)
				  prefs.setBool(QString("pounce_%1").arg(i) + "_ignoreSettings", p->ignoreSettings);
      }
      i++;
		}
    it += 1;
	}

	prefs.setNumber(QString("pounce_count"), i);

	prefs.flush();

}

/***************************************************************************
 * Returns a list of all the pounces of a user.
 ***************************************************************************/
QDict<pounce> setup::readPounces(const QString &user){
	QDict<pounce> list;
	if(user.isEmpty() || (user == QString("<new user>")) )
		return list;

  QString settingsFileXml = KINKATTA_DIR;
  settingsFileXml += user.lower();
  settingsFileXml += ".xml";

	Preferences prefs(settingsFileXml, QString("kinkatta user prefs"), QString("1.0"));

	prefs.setGroup("pounces");

	pounce *p;
  int count = prefs.getNumber("pounce_count", 0);
	for(int i = 0; i < count; i++){
		p = new pounce;
		p->buddyName = prefs.getString(QString("pounce_%1").arg(i) + "_buddyName", QString(""));
		p->signOn = prefs.getBool(QString("pounce_%1").arg(i) + "_signOn",false);
		p->returnAway = prefs.getBool(QString("pounce_%1").arg(i) + "_returnAway", false);
		p->returnIdle = prefs.getBool(QString("pounce_%1").arg(i) + "_returnIdle", false);
		p->openChat = prefs.getBool(QString("pounce_%1").arg(i) + "_openChat", false);
		p->sendMessage = prefs.getBool(QString("pounce_%1").arg(i) + "_sendMessage", false);
		p->message = prefs.getString(QString("pounce_%1").arg(i) + "_message", QString(""));
		p->execCommand = prefs.getBool(QString("pounce_%1").arg(i) + "_execCommand", false);
		p->command = prefs.getString(QString("pounce_%1").arg(i) + "_command", QString(""));
		p->doNotRemove = true;

		p->hidden = prefs.getBool(QString("pounce_%1").arg(i) + "_hidden", false);
		p->ignoreSettings = prefs.getBool(QString("pounce_%1").arg(i) + "_ignoreSettings", false);
		p->signOnSound = prefs.getString(QString("pounce_%1").arg(i) + "_signOnSound", QString(""));
		p->signOffSound = prefs.getString(QString("pounce_%1").arg(i) + "_signOffSound", QString(""));

		list.insert(p->buddyName, p);
	}
	return list;

}


QString setup::readPassword(const QString &user){
	//qDebug("::readPassword %s\n", user.latin1());

	QString pw;
  QString settingsFileXml = KINKATTA_DIR;
  settingsFileXml += user.lower();
  settingsFileXml += ".xml";

	if(user.isEmpty() || (user == QString("<new user>")) || !QFile::exists(settingsFileXml)){
		//must be a new user
		return QString("");
	}
	Preferences prefs(settingsFileXml, QString("kinkatta user prefs"), QString("1.0"));

	//qDebug("::readPassword settingsFileXml %s\n", settingsFileXml.latin1());

	prefs.setGroup("password");
	pw = prefs.getCDATA("password", QString(""));
	return decryptPassword(pw);
}

void setup::removePassword(const QString &user){
 	//qDebug("::removePassword %s\n", user.latin1());

  QString settingsFileXml = KINKATTA_DIR;
  settingsFileXml += user.lower();
  settingsFileXml += ".xml";

	if(user.isEmpty() || user == QString("<new user>") || !QFile::exists(settingsFileXml)){
		//must be a new or unknown user
		return;
	}
	Preferences prefs(settingsFileXml, QString("kinkatta user prefs"), QString("1.0"));

	//qDebug("::removePassword settingsFileXml %s\n", settingsFileXml.latin1());

	prefs.setGroup("password");
	prefs.setString("password", ""); // this is to make it "dirty"
	prefs.removeKey("password");

	prefs.flush();
}

QString setup::readProfile(const QString &user){
	//qDebug("::readProfile %s\n", user.latin1());
  QString settingsFileXml = KINKATTA_DIR;
  settingsFileXml += user.lower();
  settingsFileXml += ".xml";

	if(user.isEmpty() || (user == QString("<new user>")) || !QFile::exists(settingsFileXml))
		return QString("");

	Preferences prefs(settingsFileXml, QString("kinkatta user prefs"), QString("1.0"));

	//qDebug("setup::readProfile settingsFileXml %s\n", settingsFileXml.latin1());

	prefs.setGroup("profile");
	QString profile = prefs.getCDATA("profile", "default");
	if(profile == "default"){ //must be a new user
		profile = DEFAULT_PROFILE;
	}
	//qDebug("getCDATA %s\n", i_currentSettings->profile.latin1());
	return profile;
}


// setup.cpp















