/*
 * page_capture.cc Notebook Firewire Capture Page Object
 * Copyright (C) 2001 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <iostream>
using std::cout;
using std::endl;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include <math.h>

#include "page_capture.h"

#include "riff.h"
#include "avi.h"
#include "playlist.h"
#include "filehandler.h"
#include "ieee1394io.h"
#include "framedisplayer.h"
#include "message.h"
#include "error.h"
#include "frame.h"


static FrameDisplayer *displayer = NULL;
static Preferences *prefs = NULL;

extern "C" {
	#include "support.h"
	#include "callbacks.h"
	#include "interface.h"
	#include "commands.h"

	#include <pthread.h>

	extern struct navigate_control g_nav_ctl;

    static guint gIdleID;
    // reader changed to pointer to delay creation of Frame
    // objects so prefs dv decoder quality is properly read
    static IEEE1394Reader *reader = NULL;
    static AVIHandler *writer = NULL;
	extern char cmd[16];
	char lastcmd_capture[ 256 ];
	static gchar filename[512];
	static gchar lastPreferenceFilename[ 512 ];
	extern KinoCommon *common;

    gint avcStatusIdler(gpointer p);

	static pthread_t avcThread;
    void *avcStatusThread(gpointer p);
	static gboolean avcActive = FALSE;

	quadlet_t avcStatus;

	//
	// Preview rendering implementations
	//
	
	// Idler implementation (activated via preferences.previewRendering=0 (default)
    gint idler(gpointer p);

	// Threaded implementation (activated via preferences.previewRendering=1
	void *captureThread( void *p );
    static gint videoIdler(void *p);
	gint idleCommand;
	static pthread_t thread;
	gboolean audioOn = FALSE;
	static pthread_mutex_t threadlock = PTHREAD_MUTEX_INITIALIZER;
	static pthread_mutex_t writerlock = PTHREAD_MUTEX_INITIALIZER;
	static pthread_mutex_t avclock = PTHREAD_MUTEX_INITIALIZER;
}

/** Constructor for page.

  	\param common	common object to which this page belongs
*/

PageCapture::PageCapture( KinoCommon *common ) : isCapturing(false), captureMutex(false)
{
	cout << "> Creating Capture Page" << endl;
	this->common = common;
	this->frameArea = GTK_DRAWING_AREA( lookup_widget( common->getWidget(), "capture_drawingarea" ) );
	this->recordButton = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "capture_page_record_button" ) );
	this->stopButton = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "capture_page_stop_button" ) );
	this->muteButton = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "capture_page_mute_button" ) );
	this->fileEntry = GTK_ENTRY( lookup_widget(common->getWidget(), "capture_page_file_entry") );
	this->timecodeLabel = GTK_LABEL( lookup_widget( common->getWidget(), "position_label" ) );
	strcpy( filename, common->getConfig().file );
	lastPreferenceFilename[0] = 0;
	gtk_entry_set_text( fileEntry, filename );
}

/** Destructor for page.
*/

PageCapture::~PageCapture() {
	cout << "> Destroying Capture Page" << endl;
}

/** Start of page.
*/

void PageCapture::start() {
	cout << ">> Starting Capture" << endl;

	// If the preferences have changed, then update the filename
	if ( strcmp( lastPreferenceFilename, common->getConfig().file ) ) {
		strcpy( filename, common->getConfig().file );
		strcpy( lastPreferenceFilename, common->getConfig().file );
		gtk_entry_set_text( fileEntry, filename );
	}

	// Have the GUI reflect the mute status
	gtk_toggle_button_set_active( this->muteButton, !audioOn );

	// If we're inactive, then start eveything up (this should always be the case)
	if ( g_nav_ctl.capture_active == FALSE ) {

        displayer = new FrameDisplayer;
        prefs = &Preferences::getInstance();
        reader = new IEEE1394Reader;

        /* start the reader thread */
        if (reader->StartThread() == true) {

            // disable avc buttons if there is no legitimate phyID
            common->getConfig().phyID = reader->isPhyIDValid( common->getConfig().phyID );

            if ( common->getConfig().phyID < 0 ) {
				this->validComponents = 0;
            } else {
				this->validComponents = VIDEO_REWIND |
					   VIDEO_PLAY |
					   VIDEO_PAUSE |
					   VIDEO_STOP |
					   VIDEO_FAST_FORWARD |
					   VIDEO_FORWARD |
					   VIDEO_BACK |
					   VIDEO_NEXT_SCENE |
					   VIDEO_START_OF_SCENE |
					   VIDEO_START_OF_MOVIE |
					   VIDEO_SHUTTLE |
					   VIDEO_END_OF_MOVIE | 
					   INFO_FRAME;

				// Set up the avc monitoring thread
				monitorAVCStatus();

            }

			// Set up the preview rendering
			if ( prefs->previewRendering == 0 ) {
				g_nav_ctl.capture_active = TRUE;
            	gIdleID = gtk_idle_add(idler, lookup_widget(common->getWidget(), "capture_drawingarea"));
			}
			else if ( prefs->previewRendering == 1 ) {
				g_nav_ctl.capture_active = TRUE;
				pthread_create( &thread, NULL, captureThread, NULL );
				idleCommand = gtk_idle_add_priority( GTK_PRIORITY_DEFAULT, (GtkFunction)videoIdler, lookup_widget( common->getWidget(), "capture_drawingarea") );
			}

        } else {
            modal_message("Could not open the capture window because the ieee1394 subsystem is not available (driver not loaded?)");
        }
	}
	else {
		cout << "Capture is still active from before - serious error" << endl;
	}

	// Make sure we get a splash screen at the start
	this->windowChanged();
}

/** Define active widgets.
*/

gulong PageCapture::activate() {
	return this->validComponents;
}

/** Leaving the page
*/

void PageCapture::clean() {
	cout << ">> Leaving Capture" << endl;

	if ( g_nav_ctl.capture_active == TRUE ) {

		// Stop the preview rendering mechanism

		if ( prefs->previewRendering == 0 ) {
			videoStop();
    		gtk_idle_remove(gIdleID);
    		reader->StopThread();
    		gtk_idle_remove(this->avcStatusIdleId);
			g_nav_ctl.capture_active = FALSE;
		}
		else if ( prefs->previewRendering == 1 ) {
			// Kill the AVC mointor first
			avcActive = FALSE;
			pthread_join( avcThread, NULL );

			// Start the deactivation process for the backround processes
			g_nav_ctl.capture_active = FALSE;
			gtk_idle_remove( idleCommand );

			// Stop the capture
			stopCapture();

			// Stop the video
			videoStop();

			// Join with the capture thread to make sure we're definitely finished
			pthread_join( thread, NULL );

			// Stop the reading thread
    		reader->StopThread();
		}

		// Clean up objects created in start
    	delete displayer;
		displayer = NULL;
    	delete reader;
		reader = NULL;
	}
}


/** Start of movie.
*/

void PageCapture::videoStartOfMovie() {
	audioOn = FALSE;
	gtk_toggle_button_set_active( this->muteButton, !audioOn );
	common->toggleComponents( VIDEO_START_OF_MOVIE, true);
	pthread_mutex_lock( &avclock );
    if (reader->AVCStop( common->getConfig().phyID) < 0)
        on_preferences_activate(NULL, NULL);

	pthread_mutex_unlock( &avclock );
	// DRD> I need this for my camera to respond to the above AVCStop
	struct timespec t;
	t.tv_sec = 0;
	t.tv_nsec = 250000000;
	nanosleep( &t, NULL );
	pthread_mutex_lock( &avclock );

    reader->AVCRewind(common->getConfig().phyID);
	pthread_mutex_unlock( &avclock );
}

/** Start of previous scene.
*/

void PageCapture::videoPreviousScene() {
	audioOn = FALSE;
	gtk_toggle_button_set_active( this->muteButton, !audioOn );
	common->toggleComponents( VIDEO_START_OF_SCENE, true);
	pthread_mutex_lock( &avclock );
    if (reader->AVCPause( common->getConfig().phyID ) < 0)
        on_preferences_activate(NULL, NULL);
    reader->AVCPreviousScene(common->getConfig().phyID);
	pthread_mutex_unlock( &avclock );
}

/** Start of current scene.
*/

void PageCapture::videoStartOfScene() {
	audioOn = FALSE;
	gtk_toggle_button_set_active( this->muteButton, !audioOn );
	common->toggleComponents( VIDEO_START_OF_SCENE, true);
	pthread_mutex_lock( &avclock );
    if (reader->AVCPause( common->getConfig().phyID ) < 0)
        on_preferences_activate(NULL, NULL);
    reader->AVCPreviousScene(common->getConfig().phyID);
	pthread_mutex_unlock( &avclock );
}

/** Rewind.
*/

void PageCapture::videoRewind() {
	audioOn = FALSE;
	gtk_toggle_button_set_active( this->muteButton, !audioOn );
	common->toggleComponents( VIDEO_REWIND, true);
	pthread_mutex_lock( &avclock );
    if (reader->AVCPause( common->getConfig().phyID ) < 0)
        on_preferences_activate(NULL, NULL);
    reader->AVCRewind(common->getConfig().phyID);
	pthread_mutex_unlock( &avclock );
}

/** Frame back.
*/

void PageCapture::videoBack() {
	audioOn = FALSE;
	gtk_toggle_button_set_active( this->muteButton, !audioOn );
	common->toggleComponents( VIDEO_BACK, true);
	pthread_mutex_lock( &avclock );
    if (reader->AVCPause( common->getConfig().phyID ) < 0)
        on_preferences_activate(NULL, NULL);
    reader->AVCBack(common->getConfig().phyID);
	pthread_mutex_unlock( &avclock );
}

/** Play.
*/

void PageCapture::videoPlay() {
	common->toggleComponents( VIDEO_PLAY, true);
	pthread_mutex_lock( &avclock );
    if (reader->AVCPlay(common->getConfig().phyID) < 0)
        on_preferences_activate(NULL, NULL);
	pthread_mutex_unlock( &avclock );
	audioOn = TRUE;
	gtk_toggle_button_set_active( this->muteButton, !audioOn );
}

/** Pause.
*/

void PageCapture::videoPause() {
	audioOn = FALSE;
	gtk_toggle_button_set_active( this->muteButton, !audioOn );
	common->toggleComponents( VIDEO_PAUSE, true);
	pthread_mutex_lock( &avclock );
    if (reader->AVCPause(common->getConfig().phyID) < 0)
        on_preferences_activate(NULL, NULL);
	pthread_mutex_unlock( &avclock );
}

/** Stop.
*/

void PageCapture::videoStop() {
	audioOn = FALSE;
	gtk_toggle_button_set_active( this->muteButton, !audioOn );
	common->toggleComponents( VIDEO_STOP, true);
	pthread_mutex_lock( &avclock );
    if (reader->AVCStop(common->getConfig().phyID) < 0)
        on_preferences_activate(NULL, NULL);
	pthread_mutex_unlock( &avclock );
}

/** Forward.
*/

void PageCapture::videoForward() {
	audioOn = FALSE;
	gtk_toggle_button_set_active( this->muteButton, !audioOn );
	common->toggleComponents( VIDEO_FORWARD, true);
	pthread_mutex_lock( &avclock );
    if (reader->AVCPause( common->getConfig().phyID ) < 0)
        on_preferences_activate(NULL, NULL);
    reader->AVCForward(common->getConfig().phyID);
	pthread_mutex_unlock( &avclock );
}

/** Fast Forward.
*/

void PageCapture::videoFastForward() {
	audioOn = FALSE;
	gtk_toggle_button_set_active( this->muteButton, !audioOn );
	common->toggleComponents( VIDEO_FAST_FORWARD, true);
	pthread_mutex_lock( &avclock );
    if (reader->AVCPause( common->getConfig().phyID ) < 0)
        on_preferences_activate(NULL, NULL);
    reader->AVCFastForward(common->getConfig().phyID);
	pthread_mutex_unlock( &avclock );
}

/** Start of next scene.
*/

void PageCapture::videoNextScene() {
	audioOn = FALSE;
	gtk_toggle_button_set_active( this->muteButton, !audioOn );
	common->toggleComponents( VIDEO_NEXT_SCENE, true);
	pthread_mutex_lock( &avclock );
    if (reader->AVCPause( common->getConfig().phyID ) < 0)
        on_preferences_activate(NULL, NULL);
    reader->AVCNextScene(common->getConfig().phyID);
	pthread_mutex_unlock( &avclock );
}

/** End of current scene.
*/

void PageCapture::videoEndOfScene() {
	audioOn = FALSE;
	gtk_toggle_button_set_active( this->muteButton, !audioOn );
	common->toggleComponents( VIDEO_NEXT_SCENE, true);
	pthread_mutex_lock( &avclock );
    if (reader->AVCPause( common->getConfig().phyID ) < 0)
        on_preferences_activate(NULL, NULL);
    reader->AVCNextScene(common->getConfig().phyID);
	pthread_mutex_unlock( &avclock );
}

/** End of movie.
*/

void PageCapture::videoEndOfMovie() {
	audioOn = FALSE;
	gtk_toggle_button_set_active( this->muteButton, !audioOn );
	common->toggleComponents( VIDEO_END_OF_MOVIE, true);
	pthread_mutex_lock( &avclock );
    if (reader->AVCStop(common->getConfig().phyID) < 0)
        on_preferences_activate(NULL, NULL);

	// DRD> I need this for my camera to respond to the above AVCStop
	pthread_mutex_unlock( &avclock );
	struct timespec t;
	t.tv_sec = 0;
	t.tv_nsec = 250000000;
	nanosleep( &t, NULL );
	pthread_mutex_lock( &avclock );

    reader->AVCFastForward(common->getConfig().phyID);
	pthread_mutex_unlock( &avclock );
}

/** Shuttle.

	Bi-directional variable-speed playback.
	
	\param angle The relative speed.
*/

void PageCapture::videoShuttle( int angle ) {
	audioOn = FALSE;
	pthread_mutex_lock( &avclock );
    if (angle != 0) {
	    if (reader->AVCShuttle( common->getConfig().phyID, angle*2 ) < 0)
	        on_preferences_activate(NULL, NULL);
    } else {
	    if (reader->AVCPause( common->getConfig().phyID ) < 0)
 	       on_preferences_activate(NULL, NULL);
    }
	pthread_mutex_unlock( &avclock );
}


/** Process a keyboard event. 

  	\param event	keyboard event
*/

gboolean PageCapture::processKeyboard( GdkEventKey *event ) {
	gboolean ret = FALSE;

	// Only process while not escape mode
	if ( g_nav_ctl.escaped == FALSE ) {
		// Translate special keys to equivalent command
		switch (event->keyval) {
			case GDK_BackSpace:
			case GDK_Left:
				strcat(cmd, "h");
				break;
			case GDK_Up:
				strcat(cmd, "k");
				break;
			case GDK_Right:
				strcat(cmd, "l");
				break;
			case GDK_Return:
				startCapture();
				cmd[0] = 0;
				break;
			case GDK_Down:
				strcat(cmd, "j");
				break;
			case GDK_Escape:
				if (isCapturing) {
					stopCapture();
					cmd[0] = 0;
				} else if (common->getComponentState() & VIDEO_STOP) {
					common->changePageRequest( PAGE_EDITOR );
					return FALSE;
				} else {
					common->keyboardFeedback(cmd, "Stop");
					common->videoStop( );
					cmd[0] = 0;
				}
				break;
			case GDK_Alt_L:
			case GDK_Alt_R:
				strcpy( lastcmd_capture, "alt");
				return FALSE;
			default:
				strcat(cmd, event->string);
				break;
		}
	
		if ( !strcmp( cmd, "." ) )
			strcpy( cmd, lastcmd_capture );

		ret = processKeyboardCommand( cmd );

	} else if (event->keyval == GDK_Return || event->keyval == GDK_Escape) {
		g_nav_ctl.escaped = FALSE;
		gtk_entry_set_editable( fileEntry, FALSE );
		gtk_widget_grab_focus( common->getWidget() );
		strcpy( filename, gtk_entry_get_text( fileEntry ) );
	}
	return ret;
}

/** Process a menu command.

  	\param command	command to be processed
*/

gboolean PageCapture::processMenuCommand( char *command ) {
	strcpy( cmd, command );
	gboolean ret = processKeyboardCommand( cmd );
	return ret;
}

/** Internal method for handling a complete keyboard scene.

  	\param cmd		command to be processed;
*/

gboolean PageCapture::processKeyboardCommand( char *cmd ) {
	int count = 1;
	char real[ 256 ] = "";

	if (validComponents == 0) return FALSE;

	switch( sscanf( cmd, "%d%s", &count, real ) )
	{
		case 1:
			// Numeric value only - return immediately if the cmd is not "0"
			if ( strcmp( cmd, "0" ) ) {
				common->keyboardFeedback(cmd, "");
				return FALSE;
			}
			break;
		case 0:
			sscanf( cmd, "%s", real );
			count = 1;
			break;
	}

	if ( strcmp( lastcmd_capture, "alt") == 0 && strcmp( cmd, "f" ) == 0 ) 
	{
		gtk_entry_set_editable( fileEntry, TRUE );
		gtk_widget_grab_focus( GTK_WIDGET(fileEntry) );
		g_nav_ctl.escaped = TRUE;
		cmd[0] = 0;
		return FALSE;
	}

	strcpy( lastcmd_capture, cmd );

	/* Navigation */

	/* play, pause */
	
	if (strcmp(cmd, " ") == 0) 
	{
		if ( g_nav_ctl.active == FALSE ) {
			common->keyboardFeedback(cmd, "Play");
			common->videoPlay( );
		} else {
			common->keyboardFeedback(cmd, "Pause");
			common->videoPause( );
		}
		cmd[0] = 0;
	}

	/* advance one frame */

	else if ( strcmp(real, "l") == 0 ) 
	{
		common->keyboardFeedback(cmd, "Move forward" );
		common->videoForward();
		cmd[0] = 0;
	}

	/* backspace one frame */

	else if ( strcmp(real, "h") == 0 ) 
	{
   		common->keyboardFeedback(cmd, "Move backward");
		common->videoBack();
		cmd[0] = 0;
	}

	/* advance one second */

	else if ( strcmp(real, "w") == 0 || strcmp(real, "W") == 0 ||
		  	strcmp(real, "e") == 0 || strcmp(real, "E") == 0 ) 
	{
		common->keyboardFeedback(cmd, "Fast Forward");
		common->videoFastForward();
		cmd[0] = 0;
	}

	/* backspace one second */

	else if ((strcmp(real, "b") == 0) || (strcmp(real, "B") == 0)) 
	{
		common->keyboardFeedback(cmd, "Rewind");
		common->videoRewind();
		cmd[0] = 0;
	}

	/* start of scene */

	else if ((strcmp(cmd, "0") == 0) || (strcmp(real, "^") == 0)) 
	{
		common->videoStartOfScene( );
		for ( ; count > 1; count -- ) 
		{
			common->g_currentFrame --;
			videoStartOfScene( );
		}
		common->keyboardFeedback(cmd, "Move to start of scene");
		cmd[0] = 0;
	}

	/* end of scene */

	else if (strcmp(real, "$") == 0) 
	{
		common->videoEndOfScene( );
		for ( ; count > 1; count -- ) 
		{
			common->g_currentFrame ++;
			videoEndOfScene( );
		}
		common->keyboardFeedback(cmd, "Move to end of scene");
		cmd[0] = 0;
	}

	/* start of next scene */

	else if ((strcmp(real, "j") == 0) || strcmp(real, "+") == 0 ) 
	{
		for ( ; count >= 1; count -- )
			common->videoNextScene( );
		common->keyboardFeedback(cmd, "Move to start of next scene");
		cmd[0] = 0;
	}

	/* start of previous scene */

	else if ((strcmp(real, "k") == 0) || (strcmp(real, "-") == 0)) 
	{
		for ( ; count >= 1; count -- )
			common->videoPreviousScene( );
		common->keyboardFeedback(cmd, "Move to start of previous scene");
		cmd[0] = 0;
	}

	/* first frame */

	else if (strcmp(cmd, "gg") == 0) 
	{
		common->videoStartOfMovie( );
		common->keyboardFeedback(cmd, "Move to first frame");
		cmd[0] = 0;
	}

	/* last frame */

	else if (strcmp(cmd, "G") == 0) 
	{
		common->videoEndOfMovie( );
		common->keyboardFeedback(cmd, "Move to last frame");
		cmd[0] = 0;
	}

	/* copy current frame */

	else if ((strcmp(cmd, "y ") == 0) || (strcmp(real, "yl") == 0)) 
	{
		common->keyboardFeedback(cmd, "Start capture");
		cmd[0] = 0;
	}

	/* paste after current frame */

	else if (strcmp(real, "p") == 0) 
	{
		common->keyboardFeedback(cmd, "Stop capture");
		cmd[0] = 0;
	}

	else 
	{
		// Check for invalid commands
		if ( strlen( real ) > 3 )
			cmd[ 0 ] = 0;
		else if ( strchr( "dgy ", real[ strlen( real ) - 1 ] ) == NULL ) 
			cmd[ 0 ] = 0;

		common->keyboardFeedback(cmd, "");

#if 0
		printf("send_event: %2.2x\n", event->send_event);
		printf("time  : %8.8x\n", event->time);
		printf("state : %8.8x\n", event->state);
		printf("keyval: %8.8x\n", event->keyval);
		printf("length: %8.8x\n", event->length);
		printf("string: %s\n", event->string);
		printf("(hex) : %2.2x\n", event->string[0]);
		printf("cmd   : %s\n", cmd);
#endif
	}

	return FALSE;
}

void PageCapture::monitorAVCStatus() {
	this->avcState = avcStatus = 0;

	if ( prefs->previewRendering == 0 ) {
		if (avcStatusIdleId) gtk_idle_remove( avcStatusIdleId );
	}
	else {
		if ( avcActive )  {
			avcActive = FALSE;
			pthread_join( avcThread, NULL );
		}
	}

	if ( prefs->previewRendering == 0 ) 
		this->avcStatusIdleId = gtk_idle_add_priority( GTK_PRIORITY_DEFAULT, (GtkFunction)avcStatusIdler, this );
	else 
		pthread_create( &avcThread, NULL, avcStatusThread, this );
}

void PageCapture::applyAVCState( quadlet_t newAVCState ) {
    quadlet_t resp2 = AVC1394_MASK_RESPONSE_OPERAND(newAVCState, 2);
    quadlet_t resp3 = AVC1394_MASK_RESPONSE_OPERAND(newAVCState, 3);
    this->avcState = newAVCState;
    
    common->toggleComponents( common->getComponentState(), false );
	g_nav_ctl.active = FALSE;

    if (resp2 == AVC1394_VCR_RESPONSE_TRANSPORT_STATE_PLAY) {
        if (resp3 >= AVC1394_VCR_OPERAND_PLAY_FAST_FORWARD_1
                && resp3 <= AVC1394_VCR_OPERAND_PLAY_FASTEST_FORWARD) {
            common->toggleComponents( VIDEO_FAST_FORWARD, true );
			g_nav_ctl.active = TRUE;
        } else if (resp3 >= AVC1394_VCR_OPERAND_PLAY_FAST_REVERSE_1
                   && resp3 <= AVC1394_VCR_OPERAND_PLAY_FASTEST_REVERSE) {
            common->toggleComponents( VIDEO_REWIND, true );
			g_nav_ctl.active = TRUE;
        } else if (resp3 == AVC1394_VCR_OPERAND_PLAY_FORWARD_PAUSE) {
            common->toggleComponents( VIDEO_PAUSE, true );
        } else {
            common->toggleComponents( VIDEO_PLAY, true );
			g_nav_ctl.active = TRUE;
        }
    } else if (resp2 == AVC1394_VCR_RESPONSE_TRANSPORT_STATE_WIND) {
        if (resp3 == AVC1394_VCR_OPERAND_WIND_HIGH_SPEED_REWIND) {
            common->toggleComponents( VIDEO_START_OF_MOVIE, true );
        } else if (resp3 == AVC1394_VCR_OPERAND_WIND_STOP) {
            common->toggleComponents( VIDEO_STOP, true );
            windowChanged();
        } else if (resp3 == AVC1394_VCR_OPERAND_WIND_REWIND) {
            common->toggleComponents( VIDEO_START_OF_MOVIE, true );
        } else if (resp3 == AVC1394_VCR_OPERAND_WIND_FAST_FORWARD) {
            common->toggleComponents( VIDEO_END_OF_MOVIE, true );
        } else {
            cout << "AVC Status: Unkown winding" << endl;
        }
    } else {
        cout << "AVC Status: Unknown state" << endl;
    }
    common->commitComponentState();
}


void PageCapture::startCapture() {

	if ( !captureMutex && !isCapturing ) {
		captureMutex = true;
		isCapturing = true;
		gtk_toggle_button_set_active( recordButton, true );
		gtk_toggle_button_set_active( stopButton, false );
		gtk_widget_set_sensitive( lookup_widget( common->getWidget(), "capture_page_snapshot_button" ), false );

        if (filename == "") {
            GtkWidget *statusbar = lookup_widget ( common->getWidget(), "statusbar");
            gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 1);
            gtk_statusbar_push (GTK_STATUSBAR (statusbar), 1, "You must enter a filename to capture.");

			gtk_toggle_button_set_active( recordButton, false );
			gtk_toggle_button_set_active( stopButton, true );
			gtk_widget_set_sensitive( lookup_widget( common->getWidget(), "capture_page_snapshot_button" ), true );
			captureMutex = false;
			isCapturing = false;

        } else {
            Frame *frame = NULL;

			pthread_mutex_lock( &avclock );
        	reader->AVCPlay( common->getConfig().phyID );
			pthread_mutex_unlock( &avclock );
 
			pthread_mutex_lock( &writerlock );

            writer = new AVIHandler(common->getConfig().fileFormat);

            /* set command line parameters to the writer object */

            writer->SetMaxFrameCount(common->getConfig().frames);
            writer->SetAutoSplit(common->getConfig().autoSplit);
            writer->SetTimeStamp(common->getConfig().timeStamp);
            writer->SetEveryNthFrame(common->getConfig().every);
            writer->SetBaseName(filename);

            while (frame == NULL)
                frame = reader->GetFrame();
            writer->SetSampleFrame(*frame);
            writer->WriteFrame(*frame);
            reader->DoneWithFrame(frame);

            GtkWidget *statusbar = lookup_widget ( common->getWidget(), "statusbar");
            gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 1);
            gtk_statusbar_push (GTK_STATUSBAR (statusbar), 1, "Now capturing...");
            if (!common->getConfig().preview_capture) windowChanged();

			pthread_mutex_unlock( &writerlock );
        }
    
		captureMutex = false;
    }
}

void PageCapture::stopCapture() {

	if ( !captureMutex  && isCapturing ) {
		captureMutex = true;

		pthread_mutex_lock( &writerlock );
		writer->Close();
        delete writer;
        writer = NULL;
		pthread_mutex_unlock( &writerlock );
        
		gtk_toggle_button_set_active( recordButton, false );
		gtk_toggle_button_set_active( stopButton, true );
		gtk_widget_set_sensitive( lookup_widget( common->getWidget(), "capture_page_snapshot_button" ), true );

		pthread_mutex_lock( &avclock );
		audioOn = FALSE;
		gtk_toggle_button_set_active( this->muteButton, !audioOn );
        reader->AVCPause( common->getConfig().phyID );
		pthread_mutex_unlock( &avclock );

        GtkWidget *statusbar = lookup_widget ( common->getWidget(), "statusbar");
        gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 1);
        
        isCapturing = false;
        captureMutex = false;
    }
}


void PageCapture::saveFrame() {
	Frame         *frame = NULL;
	frame = reader->GetFrame();

	if ( frame != NULL ) {
   		char *filename = common->getFileToSave( "Save Still Frame" );
		if ( strcmp( filename, "" ) ) 
		{
	    	GdkImlibImage *im;
	    	unsigned char   pixels[720*576*4];
		
	    	if (frame != NULL) {
		    	frame->ExtractRGB( pixels );
		    	im = gdk_imlib_create_image_from_data
					(pixels, NULL, 720, frame->IsPALfromHeader() ? 576 : 480);
		    	gdk_imlib_save_image(im, filename, NULL);
	    	}
		}
        reader->DoneWithFrame( frame );
	}
}


/** set the preview drawable size to 50 percent frame size
*/
void PageCapture::view50percent() {
	assert( reader );
	Frame *frame = reader->GetFrame();
	if (frame != NULL) {
		GtkWidget *container = lookup_widget( common->getWidget(), "drawingarea1_frame" );
		AspectRatioCalculator calc( frame->decoder->width/2 +4, frame->decoder->height/2 +4, 
			frame->decoder->width/2 +4, frame->decoder->height/2 +4, frame->IsPALfromHeader(),
			frame->IsWide() );
		gtk_widget_set_usize( container, calc.width, calc.height );
        reader->DoneWithFrame( frame );
	} else {
		modal_message( "Video must be playing in the preview to set the size." );
	}
}

/** set the preview drawable size to 100 percent frame size
*/
void PageCapture::view100percent() {
	assert( reader );
	Frame *frame = reader->GetFrame();
	if (frame != NULL) {
		GtkWidget *container = lookup_widget( common->getWidget(), "drawingarea1_frame" );
		AspectRatioCalculator calc( frame->decoder->width +4, frame->decoder->height +4, 
			frame->decoder->width +4, frame->decoder->height +4, frame->IsPALfromHeader(),
			frame->IsWide() );
		gtk_widget_set_usize( container, calc.width, calc.height );
        reader->DoneWithFrame( frame );
	} else {
		modal_message( "Video must be playing in the preview to set the size." );
	}
}

void PageCapture::windowChanged() {
	common->loadSplash( frameArea );
}

void PageCapture::windowMoved() {
	if ( g_nav_ctl.capture_active == FALSE )
		common->loadSplash( frameArea );
}

void PageCapture::showFrameInfo( int i ) {
	if (!isCapturing || common->getConfig().preview_capture) {
		char msg[256];
		char *tc = NULL;
		fail_null( reader );
		pthread_mutex_lock( &avclock );
		tc = reader->AVCTimecode( common->getConfig().phyID );
		pthread_mutex_unlock( &avclock );
		if (tc != NULL ) {
			sprintf(msg, "-\n-\n%s", tc );
			gtk_label_set_text( timecodeLabel, msg);
			free(tc);
		}
	}
}


/*
 * Callbacks
 */

extern "C"
{

    gint idler(gpointer p) {

        Frame *frame1, *frame2;
        GtkWidget *drawingarea = (GtkWidget*)p;
        frame1 = frame2 = NULL;
        do {
            frame2 = frame1;
            frame1 = reader->GetFrame();
            if (frame1 != NULL) {
				pthread_mutex_lock( &writerlock );
                if (writer != NULL)
                    writer->WriteFrame(*frame1);
				pthread_mutex_unlock( &writerlock );

                if (frame2 != NULL)
                    reader->DoneWithFrame(frame2);
            }
        } while (frame1 != NULL);

        if (frame2 != NULL) {
        	if (writer == NULL || (writer && common->getConfig().preview_capture))
            	if (displayer != NULL)
            	    displayer->Put(*frame2, drawingarea, !audioOn);
            reader->DoneWithFrame(frame2);
        }
        return 1;
    }

	//
	// Threaded implementation
	//
	
	// Define the size of the buffer
	#define BUFFERED_FRAMES		20

	// Buffer and positional information
	static int framePosition = -1;
	static Frame *frameBuffer[ BUFFERED_FRAMES ];
	static int frameCount = 0;

	void *captureThread( void *p ) {
		Frame *frame = NULL;

		g_nav_ctl.capture_active  = TRUE;

		pthread_mutex_lock( &threadlock );

		gdk_threads_enter();
		common->getPageCapture()->windowChanged();
		gdk_threads_leave();

		// Make sure we can tell the difference between a used and unused frame
		for (int index = 0; index < BUFFERED_FRAMES; index ++ ) {
			frameBuffer[ index ] = NULL;
		}

		// Loop until we're informed otherwise
		while ( g_nav_ctl.capture_active  ) {

			// Put the last collected frame into the buffer so it can be picked up
			// by the video thread.
			if ( frame != NULL ) {

				// Make sure we return the oldest frame first
				if ( frameCount == BUFFERED_FRAMES ) {
					int oldest = ( framePosition + 1 ) % BUFFERED_FRAMES;
           			reader->DoneWithFrame( frameBuffer[ oldest ] );
					frameBuffer[ oldest ] = NULL;
					frameCount --;
				}

				// Bung it in to the buffer
				int newest = ( framePosition + 1 ) % BUFFERED_FRAMES;
				frameBuffer[ newest ] = frame;
				frameCount ++;
				framePosition = newest;

				pthread_mutex_unlock( &threadlock );
				pthread_mutex_lock( &threadlock );
			}
			else {
				pthread_mutex_unlock( &threadlock );
				pthread_mutex_lock( &threadlock );
			}

			// Get the next frame
           	frame = reader->GetFrame();

			// Do with it what needs to be done
           	if (frame != NULL) {
				if ( audioOn )
					if (writer == NULL || (writer != NULL && common->getConfig().preview_capture) )
	              		displayer->PutSound(*frame);

				// All access to the writer is protected
				pthread_mutex_lock( &writerlock );
               	if (writer != NULL)
                   	writer->WriteFrame(*frame);
				pthread_mutex_unlock( &writerlock );
			}
		}

		// Release all buffered frames
		for (int index = 0; index < BUFFERED_FRAMES; index ++ ) {
			if ( frameBuffer[ index ] != NULL ) {
				reader->DoneWithFrame( frameBuffer[ index ] );
			}
		}

		pthread_mutex_unlock( &threadlock );
		// Set the buffer positional vars to defaults to allow restart
		frameCount = 0;
		framePosition = -1;

		g_nav_ctl.capture_active  = FALSE;

		return NULL;
	}

	static int videoIdler( gpointer p ) {
		static int lastPosition = -1;

        GtkWidget *drawingarea = (GtkWidget*)p;

		if ( framePosition == lastPosition ) {
			//cout << "Waiting on " << framePosition << " from " << lastPosition << endl;
			pthread_mutex_lock( &threadlock );
			pthread_mutex_unlock( &threadlock );
			//cout << "Waited on " << framePosition << " from " << lastPosition << endl;
		}

		// Get the current buffer position
		int position = framePosition;

		// If we have a valid position and it's different to the last iteration
		if ( g_nav_ctl.capture_active && position != -1 && position != lastPosition ) {

			// ... display it - note, the displayer and the frame may be destroyed
			// while waiting for the thread lock so additional checks are needed
			if ( ( displayer != NULL && frameBuffer[ position ] != NULL ) ) {
			    if ( writer == NULL || (writer != NULL && common->getConfig().preview_capture) ) {
	       			displayer->Put(*frameBuffer[ position ], drawingarea, TRUE);
	       		} else {
					common->getPageCapture()->windowChanged();
				}
	       	}

			// Remember the buffer position of the last frame displayed
			lastPosition = position;
		}

		// Update the frame info
		common->showFrameInfo(0);

		// If the state has changed and we're still active...
		if ( common->getPageCapture()->getAVCState() != avcStatus && avcActive ) {
			// Apply the changes to the GUI
			common->getPageCapture()->applyAVCState( avcStatus );
		}

		return 1;
	}

    gint avcStatusIdler(gpointer p) {
		PageCapture *me = static_cast< PageCapture* > (p);
		quadlet_t avcStatus = reader->AVCStatus( common->getConfig().phyID );
		if ( me->getAVCState() != avcStatus )
			me->applyAVCState( avcStatus );
		common->showFrameInfo(0);
		return 1;
	}

    void *avcStatusThread(gpointer p) {
		avcActive = TRUE;

		while ( avcActive ) {

			// Sleep for a while before doing this again
			struct timespec t;
			t.tv_sec = 0;
			t.tv_nsec = 250000000;
			nanosleep( &t, NULL );

			// Just to be safe, make sure the reader exists
			if ( avcActive && reader != NULL ) {

				// Make sure we have the lock on avc controller 
				pthread_mutex_lock( &avclock );
				avcStatus = reader->AVCStatus( common->getConfig().phyID );
				pthread_mutex_unlock( &avclock );
			}
		}

		return NULL;
	}

    gboolean
    on_capture_drawingarea_expose_event (GtkWidget *widget,
                                         GdkEventExpose *event,
                                         gpointer user_data) {

        //        GtkWidget *drawingarea = lookup_widget(widget, "capture_drawingarea");
        // displayer.Put(*frame2, drawingarea);
        return FALSE;
    }


    void
    on_capture_page_record_button_clicked (GtkButton *button,
                                            gpointer user_data) {
		startCapture();
    }


    void
    on_capture_page_stop_button_clicked (GtkButton *button,
                                           gpointer user_data) {
		stopCapture();
    }

	void
	on_capture_page_snapshot_button_clicked (GtkButton       *button,
	                                        gpointer         user_data)
	{
		saveFrame();
	}

	gboolean
	on_capture_page_file_entry_focus_in_event (GtkWidget       *widget,
	                                        GdkEventFocus   *event,
	                                        gpointer         user_data)
	{
		gtk_entry_set_editable( GTK_ENTRY( widget ), TRUE );
		g_nav_ctl.escaped = TRUE;
	  	return FALSE;
	}
	
	
	gboolean
	on_capture_page_file_entry_focus_out_event (GtkWidget       *widget,
	                                        GdkEventFocus   *event,
	                                        gpointer         user_data)
	{
	
		gtk_entry_set_editable( GTK_ENTRY( widget ), FALSE );
		g_nav_ctl.escaped = FALSE;
		strcpy( filename, gtk_entry_get_text( GTK_ENTRY( widget) ) );
	  	return FALSE;
	}
	
	
	void
	on_capture_page_file_entry_grab_focus  (GtkWidget       *widget,
	                                        gpointer         user_data)
	{
		gtk_entry_set_editable( GTK_ENTRY( widget ), TRUE );
		g_nav_ctl.escaped = TRUE;
	}
	
	gboolean
	on_capture_page_file_entry_button_press_event (GtkWidget       *widget,
	                                        GdkEventButton  *event,
	                                        gpointer         user_data)
	{
		g_nav_ctl.escaped = TRUE;
		gtk_entry_set_editable( GTK_ENTRY(lookup_widget( widget, "capture_page_file_entry" )), TRUE );
		return FALSE;
	}

	void
	on_capture_page_gnome_file_entry_browse_clicked
	                                        (GnomeFileEntry  *gnomefileentry,
	                                        gpointer         user_data)
	{
		gtk_entry_set_editable( GTK_ENTRY( gnome_file_entry_gtk_entry( gnomefileentry ) ), TRUE );
		g_nav_ctl.escaped = TRUE;
	}



	static bool buttonGuard = true;
	void
	on_capture_page_button_pressed         (GtkButton       *button,
                                        	gpointer         user_data)
	{
		buttonGuard = false;
	}


	void
	on_capture_page_button_released        (GtkButton       *button,
                                        	gpointer         user_data)
	{
		buttonGuard = true;
	}


	void
	on_capture_page_mute_button_toggled    (GtkToggleButton *togglebutton,
                                        	gpointer         user_data)
	{
		if ( ! buttonGuard ) {
			audioOn = ! audioOn;
			buttonGuard = true;
			gtk_toggle_button_set_active( togglebutton, !audioOn );
		}
	}
}
