#include "gpsdump.h"

GPSDump::GPSDump() {
    num_packets = 0;
    num_networks = 0;
    gpsf = NULL;
}

// Fit whatever the system defines 'timeval' as into our file
// struct
time_hdr GPSDump::CurTimeHdr() {
    timeval ts;
    time_hdr ret;

    gettimeofday(&ts, NULL);

    ret.tv_sec = ts.tv_sec;
    ret.tv_usec = ts.tv_usec;

    return ret;
}

// Convert a float to primary and mantissa for storage in consistent manner
void GPSDump::Float2Pair(float in_float, int16_t *primary, int64_t *mantissa) {
    *primary = (int) in_float;
    *mantissa = (long) (1000000 * ((in_float) - *primary));
}

int GPSDump::OpenDump(const char *in_fname) {
    if ((gpsf = fopen(in_fname, "wb")) == NULL) {
        snprintf(errstr, 1024, "GPSDump unable to open file: %s",
                 strerror(errno));
        return errno;
    }

    fname = in_fname;

    // Write the header data
    file_hdr fhdr;
    memset(&fhdr, 0, sizeof(file_hdr));

    fhdr.magic = GPS_MAGIC;
    fhdr.version = 1;
    fhdr.start = CurTimeHdr();

    if (fwrite(&fhdr, sizeof(file_hdr), 1, gpsf) < 1) {
        snprintf(errstr, 1024, "GPSDump unable to write file header: %s",
                 strerror(errno));
        return errno;
    }

    // Write the network header block
    net_hdr nblk[254];

    memset(&nblk, 0, sizeof(net_hdr) * 254);

    for (int x = 0; x < 254; x++)
        nblk[x].number = x;

    if (fwrite(&nblk, sizeof(net_hdr), 254, gpsf) < 1) {
        snprintf(errstr, 1024, "GPSDump unable to write network block header: %s",
                 strerror(errno));
        return errno;
    }

    return 1;
}

int GPSDump::CloseDump(int in_unlink) {
    int ret = 1;

    if (gpsf)
        fclose(gpsf);

    if (num_packets == 0 && in_unlink) {
        unlink(fname);
        ret = -1;
    }

    gpsf = NULL;
    num_packets = 0;

    return ret;
}

int GPSDump::DumpPacket(packet_info *in_packinfo) {
    float lat, lon, alt, spd;
    int fix;

    int netnum = 0xFF;
    data_pkt dpkt;
    memset(&dpkt, 0, sizeof(data_pkt));

    if (in_packinfo == NULL) {
        dpkt.number = 0xFF;
    } else {
        // Are we hashed already?
        netnum = FindOrHashNet(in_packinfo);
        if (netnum < 0)
            return -1;

        dpkt.number = netnum;
    }

    // Split the floats
    gps->FetchLoc(&lat, &lon, &alt, &spd, &fix);

    Float2Pair(lat, &dpkt.lat, &dpkt.lat_mant);
    Float2Pair(lon, &dpkt.lon, &dpkt.lon_mant);
    Float2Pair(alt, &dpkt.alt, &dpkt.alt_mant);
    Float2Pair(spd, &dpkt.spd, &dpkt.spd_mant);
    dpkt.fix = fix;
    dpkt.ts = CurTimeHdr();

    num_packets++;

    if (fwrite(&dpkt, sizeof(data_pkt), 1, gpsf) < 1) {
        snprintf(errstr, 1024, "GPSDump unable to write network data element: %s",
                 strerror(errno));
        return errno;
    }

    num_packets++;

    return netnum;
}

// Find our map, or insert us into the map and return our new number.
int GPSDump::FindOrHashNet(packet_info *in_packinfo) {
    string macaddr = Packetracker::Mac2String(in_packinfo->bssid_mac, ':');
    int ret = 0;

    if (bssid_map.find(macaddr) == bssid_map.end()) {
        if (num_networks >= 254) {
            snprintf(errstr, 1024, "GPSDump unable to add new network.  Too many networks in this file.");
            return -1;
        }

        ret = num_networks;
        num_networks++;
        bssid_map[macaddr] = ret;

        // Store the offset, calculate the new one, and jump to it
        fpos_t old_pos;
        fgetpos(gpsf, &old_pos);

        long new_pos = (long) sizeof(file_hdr) + (sizeof(net_hdr) * ret);
        fseek(gpsf, new_pos, SEEK_SET);

        net_hdr nhdr;
        memset(&nhdr, 0, sizeof(net_hdr));
        nhdr.number = ret;
        memcpy(nhdr.bssid, in_packinfo->bssid_mac, MAC_LEN);

        if (in_packinfo->ssid[0] != '\0') {
            memcpy(nhdr.ssid, in_packinfo->ssid, SSID_SIZE);
            ssid_map[macaddr] = ret;
        }

        if (fwrite(&nhdr, sizeof(net_hdr), 1, gpsf) < 1) {
            snprintf(errstr, 1024, "GPSDump unable to write network block header: %s",
                     strerror(errno));
            return errno;
        }

        fsetpos(gpsf, &old_pos);

        return ret;

    } else {
        // If we don't know the ssid, fill it in and write it back into the file.
        if (ssid_map.find(macaddr) == ssid_map.end() &&
            in_packinfo->ssid[0] != '\0') {

            ret = bssid_map[macaddr];

            ssid_map[macaddr] = ret;

            // Store the offset, calculate the new one, and jump to it
            fpos_t old_pos;
            fgetpos(gpsf, &old_pos);

            long new_pos = (long) sizeof(file_hdr) + (sizeof(net_hdr) * ret);
            fseek(gpsf, new_pos, SEEK_SET);

            net_hdr nhdr;
            memset(&nhdr, 0, sizeof(net_hdr));
            nhdr.number = ret;
            memcpy(nhdr.bssid, in_packinfo->bssid_mac, MAC_LEN);

            memcpy(nhdr.ssid, in_packinfo->ssid, SSID_SIZE);

            if (fwrite(&nhdr, sizeof(net_hdr), 1, gpsf) < 1) {
                snprintf(errstr, 1024, "GPSDump unable to write network block header: %s",
                         strerror(errno));
                return errno;
            }

            fsetpos(gpsf, &old_pos);
        }

        return bssid_map[macaddr];
    }

    return -1;
}

