/***************************************************************************
                          kcontainerimpl.cpp  -  description
                             -------------------
    begin                : Mon Oct 16 2000
    copyright            : (C) 2000 by Sergio Moretti
    email                : sermore@libero.it
    revision             : $Revision: 1.17 $
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <kdebug.h>
#include <klocale.h>
#include "kfactoryimpl.h"
#include "kcontainerimpl.h"
#include "krootcontainerimpl.h"

const char KContainerImpl::DOCID[] = "Container";
const char KContainerImpl::DOCTYPE[] = "<!DOCTYPE Container >";


KContainerImpl::KContainerImpl(int type) : KObjectImpl(type) 
{
   _items.setAutoDelete(false);
   _glbDict.setAutoDelete(true);
}

KContainerImpl::~KContainerImpl() 
{
   //kdDebug(D_INI) << name() << ": destroy" << endl;
   //_items.setAutoDelete(true);
   if (itemCount() > 0)
      clear(true);
   _domGlobals.clear();
   _domItems.clear();
}

void KContainerImpl::clear(bool destroying) 
{
   kdDebug(D_INI) << name() << ": clear" << endl;
   // kill active items
   kill(true); //FIXME togli kill
   root()->incrItemGlobal(-itemCount());
   root()->incrItemActiveGlobal(-itemActiveCount());
   setMod(MDI_CNT);
   setMod(MDI_CLEAR);
   _items.setAutoDelete(true);
   _items.clear();
   _items.setAutoDelete(false);
   for (QDomNode n = domItems().firstChild(); !n.isNull(); 
	n = domItems().firstChild())
      domItems().removeChild(n);
   if (!destroying)
   {
      root()->setModified();
      //FIXME emit before or after destroy content?
      emitMod();
   }
}

KObjectImpl * KContainerImpl::findGlobal(int type) 
{
   KObjectImpl * glb = _glbDict[type];
   kdError(glb == 0, D_INI) << name() << ": unknown global " << type << endl;
   return glb;
}

const KContainerImpl * KContainerImpl::global() const 
{
   return dynamic_cast<const KContainerImpl*>(KObjectImpl::global()); 
}

KObjectImpl * KContainerImpl::globalLoad(const QDomElement &e) 
{
   kdDebug(D_INI) << name() << ": global item load " << e.attribute("Name") << endl;
   // load and insert global instance
   int type = e.attribute("Type").toInt();
   KObjectImpl * obj = KFactoryImpl::factory()->createObject(type);
   obj->load(this, e);
   _glbDict.insert(type, obj);
   return obj;
}

void KContainerImpl::initGlobals() 
{
   kdDebug(D_INI) << name() << ": init globals" << endl;
   _domGlobals = dom().namedItem("Globals").toElement();
   if (_domGlobals.isNull()) 
   {
      // create globals
      _domGlobals = root()->domDoc().createElement("Globals");
      dom().appendChild(_domGlobals);
      IntList lst = KFactoryImpl::factory()->typeList(_itemType);
      for(IntList::Iterator i = lst.begin(); i != lst.end(); i++)
	 objectNew(*i, NEW_GLOBAL);
      root()->setModified();
   } 
   else 
   {
      // load globals
      QDomNode n = domGlobals().firstChild();
      while (!n.isNull()) 
      {
	 QDomElement item = n.toElement();
	 if (item.isNull())
	    /* throw */
	    new DomError(name(), i18n("node unknown"), n);
	 if (KFactoryImpl::factory()->prototype(item.attribute("Type").toInt()) != 0)
	    globalLoad(item);
	 else
	    kdWarning(D_RUN) << name() << " skip unknown global " << item.attribute("Name") << endl;
	 n = n.nextSibling();
      }
      // check to see if all globals are loaded
      IntList lst = KFactoryImpl::factory()->typeList(_itemType);
      for(IntList::Iterator i = lst.begin(); i != lst.end(); i++)
	 // if don't exists a global for this type, add it
	 if (findGlobal(*i) == 0) 
	 {
	    kdWarning(D_RUN) << name() << " add global " << KFactoryImpl::factory()->typeStr(*i) << endl;
	    objectNew(*i, NEW_GLOBAL);
	    root()->setModified();
	 }
   }
}

void KContainerImpl::initItems() 
{
   kdDebug(D_INI) << name() << ": init items" << endl;
   _domItems = dom().namedItem("Items").toElement();
   if (_domItems.isNull()) 
   {
      // no items
      _domItems = root()->domDoc().createElement("Items");
      dom().appendChild(_domItems);
   } 
   else 
   {
      // load items
      QDomNode n = domItems().firstChild();
      while (!n.isNull()) 
      {
	 QDomElement item = n.toElement();
	 if (item.isNull())
	    /* throw */
	    new DomError(name(), i18n("node unknown"), n);
	 if (KFactoryImpl::factory()->prototype(item.attribute("Type").toInt()) != 0)
	    itemLoad(item);
	 else
	    kdWarning(D_RUN) << name() << " skip unknown item " << item.attribute("Name") << endl;
	 n = n.nextSibling();
      }
   }
}

void KContainerImpl::incrItemActive(int v) 
{
   if (v == 0)
      return;
   if (_itemActiveCount == 0)
   {
      //kdDebug(D_RUN) << name() << ": start" << endl;
      setMod(MDI_START);
      if (!isRoot())
	 container()->incrItemActive(1);
   }
   _itemActiveCount += v;
   if (_itemActiveCount == 0)
   {
      //kdDebug(D_RUN) << name() << ": stop" << endl;
      setMod(MDI_STOP);
      if (!isRoot())
	 container()->incrItemActive(-1);
   }
   setMod(MDI_ACTCNT);
   if (!isRoot())
      root()->incrItemActiveGlobal(v);
}

bool KContainerImpl::itemAdd(KObjectImpl *item) 
{
   kdDebug(D_INI) << name() << ": item add " << item->name() << endl;
   if (_items.findRef(item) != -1)
      /* throw */
      new Error(name(), QString("item %1 duplicate").arg(item->name()));
   if (itemFindId(item->id()) != 0)
      /* throw */ 
      new Error(name(), QString("id %1 already present").arg(item->id()));
   if (!isItemInsertable(item))
      return false;
   _items.append(item);
   if (!(item->type() & TYP_CONTAINER_T))
      root()->incrItemGlobal(1);
   setMod(MDI_CNT);
   item->setMod(MDI_ADD);
   item->emitMod();
   emitMod();
   return true;
}

KObjectImpl * KContainerImpl::itemCopy(const QDomElement &e) 
{
   kdDebug(D_INI) << name() << ": item copy " << e.attribute("Name") << endl;
   if (e.attribute("Version") != DOCVERS)
      return 0;
   QDomElement i = e.cloneNode().toElement();
   bool isContainer = e.attribute("Type").toInt() & TYP_CONTAINER_T;
   QDomElement items;
   if (isContainer) 
   {
      // remove all items
      items = i.namedItem("Items").toElement();
      i.removeChild(items);
   }
   int objId = root()->assignId();
   int type = e.attribute("Type").toInt();
   QString name = 
      QString("%1-%2").arg(KFactoryImpl::factory()->typeStr(type)).arg(objId);
   i.setAttribute("Id", objId);
   i.setAttribute("Name", name);
   KObjectImpl *obj = itemLoad(i);
   if (obj == 0)
      return 0;
   // insert item
   domItems().appendChild(i);
   if (isContainer) 
   {
      // add items to new object
      QDomNode n = items.firstChild();
      while (!n.isNull()) 
      {
	 dynamic_cast<KContainerImpl*>(obj)->itemCopy(n.toElement());
	 items.removeChild(n);
	 n = items.firstChild();
      }
   }
   root()->setModified();
   return obj;
}

KObjectImpl * KContainerImpl::itemFindId(int id) 
{
   for (Iterator it(*this); it.current(); ++it)
      if (it.current()->id() == id)
	 return it.current();
   return 0;
}

KObjectImpl* KContainerImpl::itemLoad(const QDomElement &e) 
{
   kdDebug(D_INI) << name() << ": item load " << e.attribute("Name") << endl;
   if (e.attribute("Version") != DOCVERS)
      return 0;
   int type = e.attribute("Type").toInt();
   KObjectImpl *obj = KFactoryImpl::factory()->createObject(type);
   obj->load(this, e);
   if(!itemAdd(obj)) 
   {
      delete obj;
      return 0;
   }
   return obj;
}

bool KContainerImpl::itemMove(KObjectImpl *item, KContainerImpl *to) 
{
   kdDebug(D_INI) << name() << ": item move " << item->name() 
		  << " to " << to->name() << endl;
   if (to->itemCopy(item->dom()) != 0) 
   {
      itemRemove(item);
      return true;
   }
   return false;
}

void KContainerImpl::itemRemove(KObjectImpl *item) 
{
   kdDebug(D_INI) << name() << ": item remove " << item->name() << endl;
   if (_items.findRef(item) == -1)
      /* throw */
      new Error(name(), QString("%1 not in container").arg(item->name()));
   if (item->isRunning())
      item->kill(true);
   if (item->isMods(ANYMOD)) 
   {
      kdWarning(D_INI) << name() << ": removing modified " << item->name() << endl;
      //FIXME ????
      item->resetMods();
   }
   _items.remove(item);
   QDomNode old = domItems().removeChild(item->dom());
   kdError(old.isNull(), D_RUN) << name() << ": removing " << item->name() << " from dom failed" << endl;
   root()->setModified();
   kdFatal(old.isNull(), D_RUN) << name() << ": error removing dom element " << endl;
   if (!(item->type() & TYP_CONTAINER_T))
      root()->incrItemGlobal(-1);
   item->setMod(MDI_REMOVE);
   setMod(MDI_CNT);
   item->emitMod();
   emitMod();
   //delete item; you have to delete the instance in main event loop
}

KObjectImpl * KContainerImpl::itemSearch(int id)
{
   for (Iterator it(*this); it.current(); ++it)
   {
      if (it.current()->id() == id)
	 return it.current();
      KContainerImpl *c = dynamic_cast<KContainerImpl*>(it.current());
      if (c != 0)
      {
	 KObjectImpl *item = c->itemSearch(id);
	 if (item != 0)
	    return item;
      }
   }
   return 0;
}

IntList KContainerImpl::itemTypes() const 
{
   return KFactoryImpl::factory()->typeList(itemType());
}

bool KContainerImpl::kill(bool wait) 
{
   bool retval = true;
   for (Iterator it(*this); it.current(); ++it)
      if (it.current()->isRunning())
	 retval &= it.current()->kill(wait);
   return retval;
}

void KContainerImpl::loadData() 
{
   KObjectImpl::loadData();
   kdDebug(D_INI) << name() << ": container LoadData" << endl;
   if (!dom().hasAttribute("ItemType")) 
   {
      /* throw */
      new DomError(name(), "loadData: ItemType not found", dom());
      //kdFatal(D_INI) << name() << ": ItemType not found in:\n" << dom().ownerDocument().toString() << endl;
   }

   if (!dom().hasAttribute("PriorityPolicy"))
   {
      setPriorityPolicy(PRI_FIFO);
   }
   // data
   _itemType = dom().attribute("ItemType", QString::number(0)).toInt();
   _itemActiveCount = 0;

   // CONFIG
   _cfg_priorityPolicy = static_cast<PriorityPolicy>(
      dom().attribute("PriorityPolicy", QString::number(PRI_FIFO)).toInt());

   // containers
   if (!isNormal())
      return;
   initGlobals();
   initItems();
}

KObjectImpl * KContainerImpl::objectNew(int type, int newType) 
{
   kdDebug(D_INI) << name() << ": create object type " << type << endl;
   //kdFatal(findGlobal(type) == 0, D_INI) << name() << ": unknown global " << type << endl;
   kdFatal(newType != NEW_DEFAULT && (type & _itemType == 0), D_INI) << name() << " WRONG ITEM TYPE" << type << endl;
   // create new object without initializing it
   KObjectImpl *obj = KFactoryImpl::factory()->createObject(type);
   QString typeStr = KFactoryImpl::factory()->typeStr(type);
   QDomElement e;
   if (newType == NEW_DEFAULT)
      e = root()->domDoc().createElement(obj->docId());
   else
      e = root()->findDefault(type)->dom().cloneNode(false).toElement();
   kdFatal(e.isNull(), D_INI) << name() << ": dom null" << endl;
   switch (newType) 
   {
   case NEW_GLOBAL:
      domGlobals().appendChild(e);
      e.setAttribute("Id", 0); // code for global object
      e.setAttribute("Name", QString("G%1-%2").arg(id()).arg(typeStr));
      e.setAttribute("GlobalCfg", 0);
      obj->load(this, e);
      _glbDict.insert(type, obj);
      break;
   case NEW_DEFAULT:
      root()->domDefaults().appendChild(e);
      e.setAttribute("Version", DOCVERS);
      e.setAttribute("Id", 1); // code for default object
      e.setAttribute("Name", QString("D%1-%2").arg(id()).arg(typeStr));
      e.setAttribute("Type", type);
      obj->load(this, e);
      root()->defaults().insert(type, obj);
      break;
   case NEW_ITEM: // not used
   {
      domItems().appendChild(e);
      int objId = root()->assignId();
      e.setAttribute("Id", objId);
      e.setAttribute("Name", QString("%1-%2").arg(typeStr).arg(objId));
      obj = itemLoad(e);
      break;
   }
   default:
      kdFatal(D_INI) << name() << ": unknown new type " << newType << endl;
   }
   return obj;
}

void KContainerImpl::runPeriodically() 
{
   for (Iterator it(*this); it.current(); ++it)
      it.current()->runPeriodically();
   KObjectImpl::runPeriodically();
}

// config entries

PriorityPolicy KContainerImpl::getPriorityPolicy() const 
{
   return useGlobal() ? global()->getPriorityPolicy() : _cfg_priorityPolicy;
}

void KContainerImpl::setPriorityPolicy(PriorityPolicy p) { 
   if (_cfg_priorityPolicy != p)
   {
      _cfg_priorityPolicy = p; 
      dom().setAttribute("PriorityPolicy", _cfg_priorityPolicy);
      setMod(MDI_CFG_PRPOLICY);
      root()->setModified();
   }
}

#include "kcontainerimpl.moc"
