/***************************************************************************
                          transferentry.cpp  -  description
                             -------------------
    begin                : Wed Nov 3 2004
    copyright            : (C) 2004 by Diederik van der Boor
    email                : vdboor --at-- codingdomain.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "transferentryinterface.h"
#include "transferentry.h"

#include "../kmessdebug.h"

#include <qlabel.h>

#include <klocale.h>
#include <kdebug.h>

#include <kurllabel.h>
#include <kprogress.h>
#include <krun.h>

#include <kurl.h>
#include <kmimetype.h>
#include <kiconloader.h>
#include <ksqueezedtextlabel.h>


// Constructor
TransferEntry::TransferEntry(QWidget *parent, const char *name, const QString &filename, uint filesize, bool incoming)
: TransferEntryInterface(parent, name),
 isDone_(false),
 incoming_(incoming),
 filename_(filename),
 filesize_(filesize)
{
  initializeWidgets();
}

// Constructor without standard QWidget parameters
TransferEntry::TransferEntry(const QString &filename, uint filesize, bool incoming)
: TransferEntryInterface(0, "TransferEntry"),
 isDone_(false),
 incoming_(incoming),
 filename_(filename),
 filesize_(filesize)
{
  initializeWidgets();
}



// Destructor
TransferEntry::~TransferEntry()
{

}


// Initialize the widgets
void TransferEntry::initializeWidgets()
{
  // Get a icon pixmap
  KIconLoader *loader    = KGlobal::iconLoader();
  QString      iconTitle = KMimeType::iconForURL(KURL(filename_));

  // Set pixmap and filename
  iconLabel_->setPixmap( loader->loadIcon(iconTitle, KIcon::NoGroup, iconLabel_->width()) );
  filenameLabel_->setText("<b>" + filename_ + "</b>");

  // Enable urls?
  if(incoming_)
  {
    // Can't open it yet.
    openLabel_->hide();
    openLabel_->setEnabled(false);
  }

  cancelLabel_->setEnabled(true);
}


// Called when cancel is clicked.
void TransferEntry::cancelClicked()
{
  emit cancelTransfer();

  if(! isDone_)
  {
    // If a slot didn't call failTransfer(),
    // we do this with the correct (!) status message.
    failTransfer( i18n("File transfer dialog message", "Cancelled") );
  }
}


// Indicate the transfer is complete or cancelled.
bool TransferEntry::isDone() const
{
  return isDone_;
}


// Return the file name
QString TransferEntry::getFileName() const
{
  return filename_;
}


// Mark the transfer as failed
void TransferEntry::failTransfer(const QString &message)
{
  if(isDone_) return;

  // Update the widgets
  if(message == 0 || message.isEmpty())
  {
    statusLabel_->setText( i18n("File transfer dialog message", "File transfer failed!") );
  }
  else
  {
    statusLabel_->setText(message);
  }

  // Make sure the widget does not resize when the progressbar hides
  setMinimumHeight( height() );

  progressBar_->hide();
  cancelLabel_->setEnabled(false);
  cancelLabel_->hide();

  isDone_ = true;
}


// Mark the transfer as complete
void TransferEntry::finishTransfer()
{
#ifdef KMESSTEST
  ASSERT( ! isDone_ );
#endif
  if(isDone_) return;

  // Make sure the widget does not resize when the progressbar hides
  setMinimumHeight( height() );

  // Update the widgets
  statusLabel_->setText( i18n("File transfer dialog message", "File transfer complete") );
  progressBar_->hide();
  cancelLabel_->setEnabled(false);
  cancelLabel_->hide();
  openLabel_->setEnabled(true);
  openLabel_->show();

  isDone_ = true;
}


// Called when open is clicked
void TransferEntry::openClicked()
{
  if(filename_.startsWith("/"))
  {
    // Using "file://" causes problems.
    // It makes KDE think we want a smb //hostname/path file. A bit weird though.
    new KRun("file:"  + filename_); // This class has "autodelete".
  }
  else
  {
    new KRun("file:/" + filename_); // This class has "autodelete".
  }
}


// Convert a string to some more readable form
QString TransferEntry::toReadableBytes(uint bytes)
{
  QString format;
  if(bytes > 1048576)
  {
    // Using '%.2f' instead of '%.1f' removes the ".0" part, but it's less pretty.
    format.sprintf("%.1f", (double) bytes / 1048576.0);
    return i18n("%1 MB").arg(format);
  }
  else if(bytes > 1024)
  {
    format.sprintf("%.1f", (double) bytes / 1024.0);
    return i18n("%1 kB").arg(format);
  }
  else
  {
    return i18n("%1 bytes").arg(bytes);
  }
}


// Set a status message
void TransferEntry::setStatusMessage(const QString &message)
{
#ifdef KMESSTEST
  ASSERT( ! isDone_ );
#endif
  if(isDone_) return;

  statusLabel_->setText(message);
}


// Update the progress bar
void TransferEntry::updateProgress(uint bytesTransferred)
{
#ifdef KMESSTEST
  ASSERT( ! isDone_ );
#endif
  if(isDone_) return;

  int percent = (int) (((double) bytesTransferred / (double) filesize_ * 1000.0) + 0.5);
  progressBar_->setProgress( percent );

  QString transfered = toReadableBytes(bytesTransferred);
  QString complete   = toReadableBytes(filesize_);

  statusLabel_->setText( i18n("File transfer dialog message", "%1 of %2 complete").arg(transfered).arg(complete) );
}


#include "transferentry.moc"
