/* @(#) llist.c 1.16 @(#) */
/***************************************************************\
*	Copyright (c) 1999 First Step Internet Services, Inc.
*		All Rights Reserved
*	Distributed under the BSD Licenese
*
*	Module: CORE
*
*	Doubly, Circly linked list driver
\***************************************************************/

#define _KOALAMUD_LLIST_C "@(#) nitehawk@localhost.1ststep.net|lib/koala/llist.c|20000827025247|38883 @(#)"

#include "autoconf.h"

#include "version.h"
#include "llist.h"
#include "log.h"
#include "memory.h"

/* listcreate
 * 	Create the head pointer for the linked list and point the data
 * 	appropriatly.
 *
 * 	Return:
 * 		Pointer to the linked list or NULL on failure.
 */
listnodeptr listcreate(void *data)
{
	listnodeptr list;

	/* Allocate memory for the list pointer */
	if ((list = kmalloc(sizeof(listnode), ALLOC_LLIST)) == NULL)
	{
		logmsg(LOGCRIT, "Unable to allocate memory for linked list.");
		return NULL;
	}
	
	/* Set the pointers to point where things are. */
	list->data.generic = data;
	list->next = list;
	list->prev = list;
	list->head = list;
	
	/* Return a pointer to the new list. */
	return list;
}

/* listaddnode
 * 	Add a node to the linked list without sorting.
 *
 * 	Return 0 on success and non-zero on failure.
 */
int listaddnode(listnodeptr list, void *data)
{
	listnodeptr newnode;

	/* Validate the list pointer */
	if ( ! list)
	{
		logmsg(LOGERR, "llist_addn caught invalid pointer");
		return 1;
	}

	/* If the list is 'empty', simply use the head node for the new data */
	if (list->data.generic == NULL)
	{
			list->data.generic = data;
			return 0;
	}

	/* Allocate memory for the node */
	if ((newnode = kmalloc(sizeof(listnode), ALLOC_LLIST)) == NULL)
	{
		logmsg(LOGCRIT, "Unable to allocate memory for new llist node.");
		return 1;
	}

	/* Point things in the right direction */
	newnode->head = list;
	newnode->prev = list;
	newnode->next = list->next;
	list->next->prev = newnode;
	list->next = newnode;
	
	/* Hook the data in */
	newnode->data.generic = data;
	
	return 0;
}

/* llist_remove
 * 	Remove an entry out of the linked list.  Destroy the llist struct.
 *
 * 	Return 0 on success and non-zero on failure
 *
 * 	Problems:
 * 		Nothing to remove the only node of a list.  What needs to be
 * 		done in this case.
 */
int listremovenode(listnodeptr list, void *data)
{
	listnodeptr tmp= list;
	
	/* Validate the list and data */
	if ((!list) || (!data))
	{
		logmsg(LOGERR, "llist_remove caught caught bad list or"
				" data pointer");
		return 1;
	}

	/* Is the current node the one with the data? */
	if (list->data.generic == data)
	{
		/* Move the data from the next node and delete the
		 * next node.
		 */
		if ((tmp = list->next))
		{
			// If tmp == list, set the data to null because out list is empty
			if (tmp == list)
			{
				list->data.generic = NULL;
				return 0;
			}
			else
			{
				// Move the data.
				list->data.generic = tmp->data.generic;
			}
			
			// Remove 'tmp' (list->next)
			list->next = tmp->next;
			tmp->next->prev = list;
			kmfree(tmp, ALLOC_LLIST);

			/* Node has been removed */
			return 0;
		}
		else
		{
			/* Last Node in list.  How to remove without causing a
			 * destroy?
			 */

			list->data.generic = NULL;

			return 0;
		}
	}

	/* Start off one position offset */
	tmp = list->next;

	/* Loop through the list to find the data */
	while (tmp != list)
	{
		/* We found the node. */
		if (data == tmp->data.generic)
		{
			/* Remove the node from the linkage */
			tmp->prev->next = tmp->next;
			tmp->next->prev = tmp->prev;

			/* Free the memory for the node */
			kmfree(tmp, ALLOC_LLIST);

			/* return that we found the node */
			return 0;
		}
		tmp = tmp->next;
	}

	/* We didn't find the node */
	return 1;
}

/* listnextnode
 * 	Return the next node in the linked list.
 * 	Returns NULL if there is no next node.
 */
KPRODINLINE listnodeptr listnextnode(listnodeptr list)
{
	if (list->next == list->head)
		return NULL;
	return ((list) ? (list->next) : (NULL));	
}

/* listprevnode
 * 	Return the previous node in the linked list.
 */
KPRODINLINE listnodeptr listprevnode(listnodeptr list)
{
	return ((list) ? (list->prev) : (NULL));	
}

/* listdestroy
 * 	Cleanup memory allocated by the linked list functions.
 *
 * 	ASSUMES:
 * 		All data items are either already cleaned up or otherwise
 * 		tracked.  No cleanup is done for the data.
 */
void listdestroy(listnodeptr list)
{
	listnodeptr tmp = list;
	listnodeptr nxt = tmp->next;
	
	/* Validate the list */
	if (! list)
	{
		logmsg(LOGERR, "listdestroy caught invalid list pointer");
		return;
	}

	/* Loop through the list and free all the center nodes */
	while (nxt != list)
	{
		nxt = tmp->next;
		kmfree(tmp, ALLOC_LLIST);
		tmp = nxt;
	}
}
