/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <QInputDialog>
#include <QStringListModel>

#include "connection.h"
#include "itemview.h"
#include "tool.h"

#include "aliasfield.h"

using namespace Form;

class AliasModel : public QStringListModel
{
  public:
    AliasModel( QObject *parent = 0 )
      : QStringListModel( parent )
    {
    }

    virtual Qt::ItemFlags flags( const QModelIndex& ) const
    {
      return ( Qt::ItemIsSelectable | Qt::ItemIsEnabled );
    }
};

class AliasItemView : public ItemView
{
  public:
    AliasItemView( QWidget *parent )
      : ItemView( parent )
    {
    }

  protected:
    virtual void doAdd()
    {
      bool ok = true;

      QString mail = QInputDialog::getText( this, tr( "New Email Alias" ), tr( "Email Alias:" ),
                                            QLineEdit::Normal, QString(), &ok );

      if ( ok && !mail.isEmpty() ) {
        QAbstractItemModel *model = mListView->model();

        bool found = false;
        for ( int i = 0; i < model->rowCount(); ++i ) {
          if ( mail == model->data( model->index( i, 0 ), Qt::DisplayRole ).toString() ) {
            found = true;
            break;
          }
        }

        if ( !found ) {
          int pos = model->rowCount();
          model->insertRow( pos );
          model->setData( model->index( pos, 0 ), mail );
        }
      }
    }
};

AliasField::AliasField( const QString &name, QWidget *parent )
  : Field( name, parent )
{
  mView = new AliasItemView( this );

  mModel = new AliasModel( this );
  mView->setModel( mModel );

  fieldLayout()->addWidget( mView, 0, 0 );
}

void AliasField::loadEntry( const Entry &entry )
{
  mModel->setStringList( entry.values( "alias" ) );

  mDn = entry.id();
}

void AliasField::saveEntry( Entry &entry ) const
{
  entry.clearValue( "alias" );

  const QStringList alias = mModel->stringList();
  for ( int i = 0; i < alias.count(); ++i ) {
    entry.addValue( "alias", alias[ i ] );
  }
}

bool AliasField::isValid( QString &message, PagePolicy::State ) const
{
  /**
   * We have to make sure that no other entry in the LDAP has already one
   * of our alias set.
   */

  const QStringList aliasList = mModel->stringList();
  for ( int i = 0; i < aliasList.count(); ++i ) {
    const QString alias = aliasList[ i ];

    if ( Tool::numberOfMails( Connection::self(), Connection::self()->baseDn(), alias, QStringList( mDn ) ) > 0 ) {
      message = QObject::tr( "Alias '%1' already exists." ).arg( alias );
      return false;
    }
  }

  return true;
}

void AliasField::setEditable( bool editable )
{
  mView->setEnabled( editable );
}

