/***************************************************************************
 *   Copyright (C) 2005-2006 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qgrid.h>
#include <qlabel.h>
#include <qradiobutton.h>
#include <qtooltip.h>
#include <qvbox.h>

#include <kcombobox.h>
#include <kdatetbl.h>
#include <kiconloader.h>
#include <knuminput.h>
#include <klineedit.h>
#include <klocale.h>

#include "kdatecombo.h"
#include "komparatorfilefilter.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

KomparatorFileFilter::KomparatorFileFilter( QWidget *parent, const char *name ) :
	KPushButton( parent, name ), m_auto_apply( true ), m_filter_type( MISSING_TYPE ),
	m_popup_frame( NULL ), m_name_edit( NULL ), m_path_edit( NULL ),
	m_owner_edit( NULL ), m_group_edit( NULL ),
	m_min_size( NULL ), m_max_size( NULL ), m_min_size_combo( NULL ), m_max_size_combo( NULL ),
	m_newer_real( NULL ), m_newer_equal( NULL ), m_newer_same_time( NULL ),
	m_duplicates_normal( NULL ), m_duplicates_multiple( NULL )
{
	init();
}

KomparatorFileFilter::KomparatorFileFilter( const QString &text, QWidget *parent, const char *name ) :
	KPushButton( text, parent, name ), m_auto_apply( true ), m_filter_type( MISSING_TYPE ),
	m_popup_frame( NULL ), m_name_edit( NULL ), m_path_edit( NULL ),
	m_owner_edit( NULL ), m_group_edit( NULL ),
	m_min_size( NULL ), m_max_size( NULL ), m_min_size_combo( NULL ), m_max_size_combo( NULL ),
	m_newer_real( NULL ), m_newer_equal( NULL ), m_newer_same_time( NULL ),
	m_duplicates_normal( NULL ), m_duplicates_multiple( NULL )
{
	init();
}

KomparatorFileFilter::KomparatorFileFilter( const QIconSet &icon, const QString &text, QWidget *parent, const char *name ) :
	KPushButton( icon, text, parent, name ), m_auto_apply( true ), m_filter_type( MISSING_TYPE ),
	m_popup_frame( NULL ), m_name_edit( NULL ), m_path_edit( NULL ),
	m_owner_edit( NULL ), m_group_edit( NULL ),
	m_min_size( NULL ), m_max_size( NULL ), m_min_size_combo( NULL ), m_max_size_combo( NULL ),
	m_newer_real( NULL ), m_newer_equal( NULL ), m_newer_same_time( NULL ),
	m_duplicates_normal( NULL ), m_duplicates_multiple( NULL )
{
	init();
}

KomparatorFileFilter::KomparatorFileFilter( const KGuiItem &item, QWidget *parent, const char *name ) :
	KPushButton( item, parent, name ), m_auto_apply( true ), m_filter_type( MISSING_TYPE ),
	m_popup_frame( NULL ), m_name_edit( NULL ), m_path_edit( NULL ),
	m_owner_edit( NULL ), m_group_edit( NULL ),
	m_min_size( NULL ), m_max_size( NULL ), m_min_size_combo( NULL ), m_max_size_combo( NULL ),
	m_newer_real( NULL ), m_newer_equal( NULL ), m_newer_same_time( NULL ),
	m_duplicates_normal( NULL ), m_duplicates_multiple( NULL )
{
	init();
}

KomparatorFileFilter::~KomparatorFileFilter()
{
	delete m_name_edit;
	m_name_edit = NULL;
	delete m_path_edit;
	m_path_edit = NULL;
	delete m_owner_edit;
	m_owner_edit = NULL;
	delete m_group_edit;
	m_group_edit = NULL;
	delete m_min_size;
	m_min_size = NULL;
	delete m_max_size;
	m_max_size = NULL;
	delete m_min_size_combo;
	m_min_size_combo = NULL;
	delete m_max_size_combo;
	m_max_size_combo = NULL;
	delete m_newer_real;
	m_newer_real = NULL;
	delete m_newer_equal;
	m_newer_equal = NULL;
	delete m_newer_same_time;
	m_newer_same_time = NULL;
	delete m_popup_frame;
	m_popup_frame = NULL;
}

void KomparatorFileFilter::init()
{
	setIconSet( SmallIcon( "filter" ) );
	setFixedSize( iconSet()->iconSize( QIconSet::Small ) );
	setToggleButton( true );

	m_popup_frame = new KPopupFrame( this );
	m_popup_frame->installEventFilter( this );
	m_main_box = new QHBox( m_popup_frame );

	QVBox *buttons_box = new QVBox( m_main_box );
	KPushButton *clear_button = new KPushButton( SmallIcon( "clear_left" ), "", buttons_box );
	clear_button->setFixedSize( clear_button->iconSet()->iconSize( QIconSet::Small ) );
	connect( clear_button, SIGNAL( clicked() ), this, SLOT( slotClear() ) );
	new QLabel( "", buttons_box );
	KPushButton *autoapply_button = new KPushButton( SmallIcon( "apply" ), "", buttons_box );
	autoapply_button->setFixedSize( autoapply_button->iconSet()->iconSize( QIconSet::Small ) );
	autoapply_button->setToggleButton( true );
	connect( autoapply_button, SIGNAL( toggled( bool ) ), this, SLOT( slotAutoApply( bool ) ) );
	autoapply_button->setOn( true );
	buttons_box->setMargin( 2 );
	
	QGrid *string_box = new QGrid( 2/*columns*/, m_main_box );
	string_box->setSpacing( 4 );
	string_box->setMargin( 4 );
	new QLabel( i18n( "Name:" ), string_box );
	m_name_edit = new KLineEdit( "", string_box );
	m_name_edit->setMinimumWidth( 80 );
	connect( m_name_edit, SIGNAL( textChanged( const QString & ) ), this, SLOT( slotEmitChanged( const QString & ) ) );
	new QLabel( i18n( "Path:" ), string_box );
	m_path_edit = new KLineEdit( "", string_box );
	m_path_edit->setMinimumWidth( 80 );
	connect( m_path_edit, SIGNAL( textChanged( const QString & ) ), this, SLOT( slotEmitChanged( const QString & ) ) );
	new QLabel( i18n( "Owner:" ), string_box );
	m_owner_edit = new KLineEdit( "", string_box );
	m_owner_edit->setMinimumWidth( 80 );
	connect( m_owner_edit, SIGNAL( textChanged( const QString & ) ), this, SLOT( slotEmitChanged( const QString & ) ) );
	new QLabel( i18n( "Group:" ), string_box );
	m_group_edit = new KLineEdit( "", string_box );
	m_group_edit->setMinimumWidth( 80 );
	connect( m_group_edit, SIGNAL( textChanged( const QString & ) ), this, SLOT( slotEmitChanged( const QString & ) ) );

	QFrame *separator = new QFrame( m_main_box );
	separator->setFrameShape( QFrame::VLine );

	m_misc_box = new QVBox( m_main_box );
	m_misc_box->setSpacing( 4 );
	m_misc_box->setMargin( 4 );
	QGrid *size_box = new QGrid( 3/*columns*/, m_misc_box );
	size_box->setSpacing( 4 );
	new QLabel( i18n( "Size between" ), size_box );
	m_min_size = new KIntSpinBox( 0/*lower*/, 9999/*upper*/, 1/*step*/, 0/*value*/, 10/*base*/, size_box/*parent*/ );
	m_min_size_combo = new KComboBox( size_box );
	m_min_size_combo->insertItem( i18n( "Byte(s)" ) );
	m_min_size_combo->insertItem( i18n( "KB" ) );
	m_min_size_combo->insertItem( i18n( "MB" ) );
	QLabel *and_label = new QLabel( i18n( "and" ), size_box );
	and_label->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
	m_max_size = new KIntSpinBox( 0/*lower*/, 9999/*upper*/, 1/*step*/, 9999/*value*/, 10/*base*/, size_box/*parent*/ );
	m_max_size_combo = new KComboBox( size_box );
	m_max_size_combo->insertItem( i18n( "Byte(s)" ) );
	m_max_size_combo->insertItem( i18n( "KB" ) );
	m_max_size_combo->insertItem( i18n( "MB" ) );
	m_max_size_combo->setCurrentItem( 2 );
	connect( m_min_size, SIGNAL( valueChanged( int ) ), this, SLOT( slotEmitChanged() ) );
	connect( m_min_size_combo, SIGNAL( activated( const QString & ) ), this, SLOT( slotEmitChanged( const QString & ) ) );
	connect( m_max_size, SIGNAL( valueChanged( int ) ), this, SLOT( slotEmitChanged() ) );
	connect( m_max_size_combo, SIGNAL( activated( const QString & ) ), this, SLOT( slotEmitChanged( const QString & ) ) );
	QHBox *time_box = new QHBox( m_misc_box );
	time_box->setSpacing( 4 );
	m_date_checkbox = new QCheckBox( i18n( "Modified or created" ), time_box );
	m_date_combo = new KDateCombo( QDate::currentDate(), time_box );
	m_date_combo->setEnabled( false );
	connect( m_date_combo, SIGNAL( activated( const QString & ) ), this, SLOT( slotEmitChanged( const QString & ) ) );
	connect( m_date_checkbox, SIGNAL( toggled( bool ) ), this, SLOT( slotConsiderDate( bool ) ) );

	m_main_box->adjustSize();
	m_popup_frame->setMainWidget( m_main_box );

	const QString clear_tt = i18n( "Reset all filters to default values" );
	QToolTip::add( clear_button, clear_tt );
	const QString autoapply_tt = i18n( "Apply filters automatically with every change that is made, otherwise only, if the filter dialog is closed" );
	QToolTip::add( autoapply_button, autoapply_tt );
	const QString name_tt = i18n( "File name contains..." );
	QToolTip::add( m_name_edit, name_tt );
	const QString path_tt = i18n( "Path contains..." );
	QToolTip::add( m_path_edit, path_tt );
	const QString owner_tt = i18n( "Owner contains..." );
	QToolTip::add( m_owner_edit, owner_tt );
	const QString group_tt = i18n( "Group contains..." );
	QToolTip::add( m_group_edit, group_tt );
	const QString size_min_tt = i18n( "Minimum size of files to be displayed" );
	QToolTip::add( m_min_size, size_min_tt );
	QToolTip::add( m_min_size_combo, size_min_tt );
	const QString size_max_tt = i18n( "Maximum size of files to be displayed" );
	QToolTip::add( m_max_size, size_max_tt );
	QToolTip::add( m_max_size_combo, size_max_tt );
	const QString date_tt = i18n( "Files created or modified on..." );
	QToolTip::add( m_date_checkbox, date_tt );
	QToolTip::add( m_date_combo, date_tt );

	connect( this, SIGNAL( toggled( bool ) ), this, SLOT( slotToggled( bool ) ) );
}

void KomparatorFileFilter::setFilterType( int filter_type )
{
	m_filter_type = filter_type;

	if ( m_filter_type == NEWER_TYPE )
	{
		QFrame *separator = new QFrame( m_misc_box );
		separator->setFrameShape( QFrame::HLine );
		QHBox *result_box = new QHBox( m_misc_box );
		m_newer_real = new QCheckBox( i18n( "Real newer files" ), result_box );
		m_newer_real->setChecked( true );
		connect( m_newer_real, SIGNAL( clicked() ), this, SLOT( slotEmitChanged() ) );
		m_newer_equal = new QCheckBox( i18n( "Newer but equal files" ), result_box );
		m_newer_equal->setChecked( true );
		connect( m_newer_equal, SIGNAL( clicked() ), this, SLOT( slotEmitChanged() ) );
		m_newer_same_time = new QCheckBox( i18n( "Different files with same time stamp" ), m_misc_box );
		m_newer_same_time->setChecked( true );
		connect( m_newer_same_time, SIGNAL( clicked() ), this, SLOT( slotEmitChanged() ) );

		m_main_box->adjustSize();
		m_popup_frame->setMainWidget( m_main_box );

		const QString newer_real_tt = i18n( "Files which are different and one is newer than the other" );
		QToolTip::add( m_newer_real, newer_real_tt );
		const QString newer_equal_tt = i18n( "Files which are equal but one is newer than the other" );
		QToolTip::add( m_newer_equal, newer_equal_tt );
		const QString newer_same_time_tt = i18n( "Files which are different but have the same time stamp" );
		QToolTip::add( m_newer_same_time, newer_same_time_tt );
	}
	else if ( m_filter_type == DUPLICATES_TYPE )
	{
		QFrame *separator = new QFrame( m_misc_box );
		separator->setFrameShape( QFrame::HLine );
		m_duplicates_normal = new QCheckBox( i18n( "Duplicate files pairs" ), m_misc_box );
		m_duplicates_normal->setChecked( true );
		connect( m_duplicates_normal, SIGNAL( clicked() ), this, SLOT( slotEmitChanged() ) );
		m_duplicates_multiple = new QCheckBox( i18n( "Multiple duplicates" ), m_misc_box );
		m_duplicates_multiple->setChecked( true );
		connect( m_duplicates_multiple, SIGNAL( clicked() ), this, SLOT( slotEmitChanged() ) );

		const QString duplicates_normal_tt = i18n( "Files that have one duplicate in the other parent url and the same relative path" );
		QToolTip::add( m_duplicates_normal, duplicates_normal_tt );
		const QString duplicates_multiple_tt = i18n( "Files that have several duplicates and/or different relative paths" );
		QToolTip::add( m_duplicates_multiple, duplicates_multiple_tt );

		m_main_box->adjustSize();
		m_popup_frame->setMainWidget( m_main_box );
	}
	else
	{
		new QLabel( "", m_misc_box );
	}
}

void KomparatorFileFilter::slotToggled( bool )
{
	if ( m_popup_frame->isHidden() )
	{
		m_popup_frame->popup( mapToGlobal( QPoint( 0, -m_popup_frame->height() ) ) );
		m_name_edit->setFocus();
	}
}

void KomparatorFileFilter::slotConsiderDate( bool on )
{
	m_date_combo->setEnabled( on );
	slotEmitChanged();
}

void KomparatorFileFilter::slotEmitFilterChanged()
{
	emitChanged();
}

void KomparatorFileFilter::slotEmitChanged( const QString & )
{
	if ( m_auto_apply ) emitChanged();
}

void KomparatorFileFilter::emitChanged()
{
	QDate date;
	m_date_combo->getDate( &date );

	if ( m_filter_type == DUPLICATES_TYPE )
	{
		emit( signalChanged( this, m_name_edit->text(), m_path_edit->text(), m_owner_edit->text(), m_group_edit->text(),
			m_min_size->value() << (10*m_min_size_combo->currentItem()), m_max_size->value() << (10*m_max_size_combo->currentItem()),
			m_date_checkbox->isChecked(), date, m_duplicates_normal->isChecked(), m_duplicates_multiple->isChecked() ) );
	}
	else if ( m_filter_type == NEWER_TYPE )
	{
		emit( signalChanged( this, m_name_edit->text(), m_path_edit->text(), m_owner_edit->text(), m_group_edit->text(),
			m_min_size->value() << (10*m_min_size_combo->currentItem()), m_max_size->value() << (10*m_max_size_combo->currentItem()),
			m_date_checkbox->isChecked(), date, m_newer_real->isChecked(), m_newer_equal->isChecked(), m_newer_same_time->isChecked() ) );
	}
	else
	{
		emit( signalChanged( this, m_name_edit->text(), m_path_edit->text(), m_owner_edit->text(), m_group_edit->text(),
			m_min_size->value() << (10*m_min_size_combo->currentItem()), m_max_size->value() << (10*m_max_size_combo->currentItem()),
			m_date_checkbox->isChecked(), date ) );
	}
}

void KomparatorFileFilter::slotClear()
{
	m_name_edit->setText( "" );
	m_path_edit->setText( "" );
	m_owner_edit->setText( "" );
	m_group_edit->setText( "" );

	m_min_size->setValue( 0 );
	m_max_size->setValue( 9999 );
	m_min_size_combo->setCurrentItem( 0 );
	m_max_size_combo->setCurrentItem( 2 );

	m_date_checkbox->setChecked( false );
	m_date_combo->setDate( QDate::currentDate() );
	
	if ( m_newer_real ) m_newer_real->setChecked( true );
	if ( m_newer_equal ) m_newer_equal->setChecked( true );
	if ( m_newer_same_time ) m_newer_same_time->setChecked( true );

	if ( m_duplicates_normal ) m_duplicates_normal->setChecked( true );
	if ( m_duplicates_multiple ) m_duplicates_multiple->setChecked( true );

	if ( m_name_edit ) slotEmitChanged();
}

void KomparatorFileFilter::slotAutoApply( bool apply )
{
	m_auto_apply = apply;

	if ( m_name_edit ) slotEmitChanged();
}

bool KomparatorFileFilter::eventFilter( QObject*, QEvent *e )
{
	if ( e->type() == QEvent::Hide )
	{
		if ( !m_popup_frame->isHidden() ) // This hide event should actually really hide the frame.
		{
			m_popup_frame->hide();
		}
		else if ( isOn() ) // frame is hidden now; "on" state shows us, that these events were not yet emitted.
		{
			// emit these events always, if closed. Auto apply is only taken into account during modification of filters.
			emitChanged();
			setOn( false );
		}
		
		return true;
	}
	else if ( e->type() == QEvent::KeyRelease || e->type() == QEvent::KeyPress )
	{
		if( ((QKeyEvent *)e)->key() == Qt::Key_Escape ) // Esc would close main window, if it wasn't caught here !!!
		{
			if ( !m_popup_frame->isHidden() ) m_popup_frame->hide();
			return true;
		}
	}

  return false;
}

QStringList KomparatorFileFilter::getStringList()
{
	QStringList ret;
	ret.push_back( m_name_edit->text() );
	ret.push_back( m_path_edit->text() );
	ret.push_back( m_owner_edit->text() );
	ret.push_back( m_group_edit->text() );
	return ret;
}

QValueList<int> KomparatorFileFilter::getIntList()
{
	QValueList<int> ret;
	ret.push_back( m_min_size->value() );
	ret.push_back( m_min_size_combo->currentItem() );
	ret.push_back( m_max_size->value() );
	ret.push_back( m_max_size_combo->currentItem() );
	ret.push_back( m_date_checkbox->isChecked() );
	QDate date;
	m_date_combo->getDate( &date );
	ret.push_back( date.year() );
	ret.push_back( date.month() );
	ret.push_back( date.day() );
	if ( m_filter_type == DUPLICATES_TYPE )
	{
		ret.push_back( m_duplicates_normal->isChecked() ? 1 : 0 );
		ret.push_back( m_duplicates_multiple->isChecked() ? 1 : 0 );
	}
	else if ( m_filter_type == NEWER_TYPE )
	{
		ret.push_back( m_newer_real->isChecked() ? 1 : 0 );
		ret.push_back( m_newer_equal->isChecked() ? 1 : 0 );
		ret.push_back( m_newer_same_time->isChecked() ? 1 : 0 );
	}
	return ret;
}

void KomparatorFileFilter::setStringList( QStringList &string_list )
{
	if ( string_list.size() != 4 ) return;

	m_name_edit->setText( string_list[0] );
	m_path_edit->setText( string_list[1] );
	m_owner_edit->setText( string_list[2] );
	m_group_edit->setText( string_list[3] );
}

void KomparatorFileFilter::setIntList( QValueList<int> &int_list )
{
	if ( ( m_filter_type == DUPLICATES_TYPE && int_list.size() != 10 ) ||
		( m_filter_type == NEWER_TYPE && int_list.size() != 11 ) ||
		( m_filter_type == MISSING_TYPE && int_list.size() != 8 ) ) return;

	m_min_size->setValue( int_list[0] );
	m_min_size_combo->setCurrentItem( int_list[1] );
	m_max_size->setValue( int_list[2] );
	m_max_size_combo->setCurrentItem( int_list[3] );
	m_date_checkbox->setChecked( int_list[4] );
	QDate date;
	date.setYMD( int_list[5], int_list[6], int_list[7] );
	m_date_combo->setDate( date );
	if ( m_filter_type == DUPLICATES_TYPE )
	{
		m_duplicates_normal->setChecked( (bool)int_list[8] );
		m_duplicates_multiple->setChecked( (bool)int_list[9] );
	}
	else if ( m_filter_type == NEWER_TYPE )
	{
		m_newer_real->setChecked( int_list[8] );
		m_newer_equal->setChecked( int_list[9] );
		m_newer_same_time->setChecked( int_list[10] );
	}
}
