//                          -*- mode: C++ -*-
//
// Copyright(C) 2005,2006,2007 Stefan Siegl <stesie@brokenpipe.de>
// Copyright(C) 2006 Martin Albrecht <malb@informatik.uni-bremen.de>
//
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <iostream>

#include <kopetemetacontact.h>
#include <kopeteuiglobal.h>
#include <kopetechatsession.h>

#include <kdebug.h>
#include <kmessagebox.h>
#include <klocale.h>

#include "silcaccount.h"
#include "silccontactmanager.h"
#include "silcbuddycontact.h"
#include "silcchannelcontact.h"
#include "silcservercontact.h"

template <class T> int 
SilcContactList<T>::compareItems(QPtrCollection::Item item1,
				 QPtrCollection::Item item2)
{
  T *contact1 = (T *) item1;
  T *contact2 = (T *) item2;
  return contact1->nickName().compare(contact2->nickName());
}

template <class T> T *
SilcContactList<T>::lookup(const QString &name) 
{
  typename SilcContactList<T>::iterator iter = begin();
  while(iter != end()) {
    if((*iter)->nickName().compare(name) == 0)
      return static_cast<T*>(*iter);
    else
      iter ++;
  }
  
  return NULL;
}

template <> SilcBuddyContact *
SilcContactList<SilcBuddyContact>::lookupByFingerprint(const QString &finger)
{
  SilcContactList<SilcBuddyContact>::iterator iter = begin();

  while(iter != end()) {
    if(! static_cast<SilcBuddyContact *>(*iter)->fingerprint().compare(finger))
      return static_cast<SilcBuddyContact *>(*iter);
    else
      iter ++;
  }
  
  return NULL;
}

template <> SilcBuddyContact *
SilcContactList<SilcBuddyContact>::lookupById(const QString &id)
{
  SilcContactList<SilcBuddyContact>::iterator iter = begin();

  while(iter != end()) {
    if(! static_cast<SilcBuddyContact *>(*iter)->contactId().compare(id))
      return static_cast<SilcBuddyContact *>(*iter);
    else
      iter ++;
  }
  
  return NULL;
}

template <class T> void
SilcContactList<T>::setStatus(Kopete::OnlineStatus status)
{
  typename SilcContactList<T>::iterator iter = begin();
  for(; iter != end(); iter ++)
    (*iter)->setOnlineStatus(status);
}

SilcContactManager::SilcContactManager(SilcAccount *account)
  : _channels(), _account(account), _outstandingWhoisRequests(0)
{
  QObject::connect(account, SIGNAL(connected()),
		   this, SLOT(slotConnected()));
  QObject::connect(account, SIGNAL(disconnected()),
		   this, SLOT(slotDisconnected()));
}

SilcContactManager::~SilcContactManager() { }


void
SilcContactManager::getClientsCallback(SilcTK::SilcClient,
				       SilcTK::SilcClientConnection,
                                       SilcTK::SilcStatus status,
                                       SilcTK::SilcDList celist,
				       void *context)
{
  SilcTK::SilcClientEntry client_entry;
  SilcBuddyContact *req = (SilcBuddyContact *) context;
  SilcContactManager *cm = req->account()->contactManager();
  SilcAccount *account = req->account();

  if (status == SILC_STATUS_ERR_TIMEDOUT) {
    SilcTK::silc_client_get_clients_whois(account->client(),
					  account->conn(), 
					  req->nickName().utf8(),
					  NULL, NULL, getClientsCallback,
					  (void *) req);
    return;
  }
  
  if (!celist) return;

  SilcTK::silc_dlist_start(celist);

  // add each found entry and watch it ...
  while ((client_entry = (SilcTK::SilcClientEntry) SilcTK::silc_dlist_get(celist))
         != SILC_LIST_END) {
    SilcBuddyContact *buddy = (SilcBuddyContact *) client_entry->context;

    if(! buddy)
      buddy =
	cm->createBuddy(QString::fromUtf8(client_entry->nickname),
			NULL, client_entry);

    req->account()->setBuddyOnlineStatus(cm, buddy, client_entry->mode);
  }

  if(-- cm->_outstandingWhoisRequests == 0) {
    // no more outstanding whois requests, fire watch requests ...
    cm->watchAllBuddies(1);
  }
}


void
SilcContactManager::watchAllBuddies(bool watch)
{
  SilcContactList<SilcBuddyContact>::iterator iter = _buddies.begin();
  for(; iter != _buddies.end(); iter ++) {
    if((* iter) == account()->myself())
      continue;
    
    SilcBuddyContact *buddy = static_cast<SilcBuddyContact*>(* iter);
    buddy->watchme(watch);
  }
}


void
SilcContactManager::slotConnected(void)
{
  // set online status again, the first call initiated by 
  // SilcAccount::setOnlineStatus probably was not successful, since
  // the connection wasn't up then ...
  account()->setOnlineStatus(account()->myself()->onlineStatus());

  // mark all channels online ...
  _channels.setStatus(SilcProtocol::protocol()->statusOnlineChannel);

  // send a whois request for every buddy we've heard of and put this buddy
  // on the watch list ...
  //
  // we obviously need to do this one after another, i.e. send the /watch
  // commands as soon as any other /whois request has been answered ...
  // ... this is probably some work around for some hidden bug ...
  SilcContactList<SilcBuddyContact>::iterator iter = _buddies.begin();
  for(; iter != _buddies.end(); iter ++) {
    if((* iter) == account()->myself())
       continue;

    SilcBuddyContact *buddy = static_cast<SilcBuddyContact*>(* iter);

    _outstandingWhoisRequests ++;
    SilcTK::silc_client_get_clients_whois(account()->client(),
					  account()->conn(), 
					  buddy->nickName().utf8(),
					  NULL, NULL, getClientsCallback,
					  (void *) buddy);
  } 
}

void
SilcContactManager::setOnlineStatus(SilcBuddyContact *buddy, 
				    const Kopete::OnlineStatus &status)
{
  if(!buddy) {
    return;
  }
  buddy->setOnlineStatus(status);

  // finally notify open chat windows to update their information ...
  SilcContactList<SilcChannelContact>::iterator iter = _channels.begin();

  for(; iter != _channels.end(); iter ++) {
    if(! (* iter)->manager()) continue; // ignore channels without a manager
    const Kopete::ContactPtrList members = (* iter)->manager()->members();

    if(members.contains(buddy))
      static_cast<SilcChannelContact *>(* iter)->updateBuddyOnlineStatus(buddy);
  }
}


void
SilcContactManager::slotDisconnected(void)
{
  // mark all contacts as offline ...
  _channels.setStatus(SilcProtocol::protocol()->statusOffline);
  _buddies.setStatus(SilcProtocol::protocol()->statusOffline);

  _outstandingWhoisRequests = 0;
}

SilcServerContact *
SilcContactManager::createServer(const QString &hostname)
{
  Kopete::MetaContact *meta = new Kopete::MetaContact();
  meta->setTemporary(true);

  return new SilcServerContact(_account, hostname, meta);
}

SilcChannelContact *
SilcContactManager::createChannel(const QString &channel,
				  Kopete::MetaContact *meta,
				  SilcTK::SilcChannelEntry entry)
{
  if(entry && entry->context) return (SilcChannelContact *) entry->context;

  if(! meta) {
    meta = new Kopete::MetaContact();
    meta->setTemporary(true);
  }

  SilcChannelContact *c = new SilcChannelContact(_account, channel, meta);
  if(entry) c->setChannelEntry(entry);
  _channels.append(c);

  return c;
}

SilcBuddyContact *
SilcContactManager::createBuddy(Kopete::MetaContact *meta,
				const QString &finger)
{
  // check whether there already is a buddy with this fingerprint ...
  SilcBuddyContact *bc = _buddies.lookupByFingerprint(finger);
  if(bc) 
    return bc;

  if(! meta) {
    meta = new Kopete::MetaContact();
    meta->setTemporary(true);
  }

  SilcBuddyContact *contact =
    new SilcBuddyContact(_account, QString::null, finger, meta);
  
  _buddies.append(contact);
  return contact;
}

SilcBuddyContactData *
SilcContactManager::popPendingBuddy(const QString &nickname)
{

  SilcBuddyContactData *buddy;
  for ( buddy = _pending_buddies.first(); buddy; buddy =
        _pending_buddies.next() ) {
    if(buddy->nickName().compare(nickname) == 0) {
      _pending_buddies.remove(buddy);
      break;
    }
  }
  return buddy;
}

SilcBuddyContact *
SilcContactManager::createBuddy(const QString &nickname,
				Kopete::MetaContact *meta,
                                SilcTK::SilcClientEntry entry
                                )
{
   QString finger;
   if(!_buddies.lookupByFingerprint("self")) {
     finger = "self"; //we are "self"
   } else {
     finger = nickname;
   }

  if(entry) {
    if(entry->context)
      return (SilcBuddyContact *) entry->context;
    finger = SilcBuddyContact::convFingerprint
      ((const char *) entry->fingerprint);

    SilcBuddyContact *bc = _buddies.lookupByFingerprint(finger);
    if(bc) {
      //if(! bc->clientEntry())
	bc->setNickName(QString::fromUtf8(entry->nickname));

      bc->addClientEntry(entry);
      return bc;
    }
  }

  if(! meta) {
    meta = new Kopete::MetaContact();
    meta->setTemporary(true);
  }

  //incomplete Buddy
  if(! nickname.compare(finger)) {
    SilcBuddyContactData *buddy = new SilcBuddyContactData(_account, nickname,
							   finger, meta);
    this->_pending_buddies.append(buddy);
    return NULL;
  } 

  SilcBuddyContact *contact =
    new SilcBuddyContact(_account, nickname, finger, meta);  

  contact->setClientEntry(entry);
  _buddies.append(contact);

  contact->watchme(true);
  
  return contact;
}

void 
SilcContactManager::buddySignedOff(SilcBuddyContact *buddy,
				   const QString &reason)
{
  // now check all channels for the quitting buddy
  SilcContactList<SilcChannelContact>::iterator iter = _channels.begin();

  for(; iter != _channels.end(); iter ++) {
    if(! (* iter)->manager()) continue; // no manager, i.e. no window
    const Kopete::ContactPtrList members = (* iter)->manager()->members();

    if(members.contains(buddy)
       && !static_cast<SilcChannelContact*>(* iter)->isJoined(buddy))
      (* iter)->manager()->removeContact(buddy, reason);
  }
}

SilcBuddyContact *
SilcContactManager::lookupBuddy(const QString &name)
{
  return _buddies.lookup(name);
}

SilcBuddyContact *
SilcContactManager::lookupBuddyByFingerprint(const QString &finger)
{
  return _buddies.lookupByFingerprint(finger);
}

bool
SilcContactManager::addBuddy(SilcBuddyContact *buddy)
{
  if(this->_buddies.lookupByFingerprint(buddy->fingerprint())) {
   return false;
  } else {
   _buddies.append(buddy);
    return true;
  }
}

SilcChannelContact *
SilcContactManager::lookupChannel(const QString &name)
{
  return _channels.lookup(name);
}

SilcBuddyContact *
SilcContactManager::lookupBuddyById(const QString &id)
{
  return _buddies.lookupById(id);
}

#include "silccontactmanager.moc"
