//                       -*- mode: C++ -*-
//
// Copyright(C) 2005,2006,2007,2008 Stefan Siegl <stesie@brokenpipe.de>
// Copyright(C) 2006 Martin Albrecht <malb@informatik.uni-bremen.de>
// Copyright(C) 2007 Christian Dietrich <stettberger@brokenpipe.de>
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef KOPETESILC_SILCBUDDYCONTACT_H
#define KOPETESILC_SILCBUDDYCONTACT_H

#include <qdatetime.h>
#include <qstringlist.h>

#include "silccontact.h"

//#include "silccontactmanager.h"
//#include "silcbuddycliententry.h"
//#include "silcbuddyattributes.h"

class SilcBuddyAttributes;
class SilcBuddyClientEntry;

class KAction;
class KActionMenu;
class KActionCollection;

/**
 * @brief Reimplementation of Kopete::Contact class, used for SILC Users
 *
 * Every user of the connected SILC network will have it's very own
 * SilcBuddyContact class associated as soon as he or she is known to
 * kopete_silc.  This is as soon as he or she joins a joined chat session,
 * incoming private message, etc.
 * 
 * @author Stefan Siegl <stesie@brokenpipe.de>
 */
class SilcBuddyContact : public SilcContact {
  Q_OBJECT;

public:

  SignatureStatus verifySignature(SilcTK::SilcMessagePayload payload);

  SilcBuddyContact(SilcAccount *account,
		   const QString &nickname,
		   const QString &fingerprint,
		   Kopete::MetaContact *meta,
		   const QString &icon = QString::null);
  virtual ~SilcBuddyContact();


  /**
   * @brief return a reference to the SilcBuddyClientEntry class
   */
  SilcBuddyClientEntry &clientEntries(void)
  {
    return *_clientEntry;
  }

  /**
   * @brief return a const reference to the SilcBuddyClientEntry class
   */
  const SilcBuddyClientEntry &clientEntries(void) const
  {
    return *_clientEntry;
  }

  /**
   * @brief Return the fingerprint of the buddy contact.
   */
  inline const QString fingerprint(void) const 
  {
    return contactId().mid(1);
  }

  /**
   * @brief Return whether the fingerprint is marked as trusted.
   */
  bool fpTrusted(void) const
  {
    return _fpTrusted;
  }

  /**
   * @brief Mark the fingerprint is to be considered as trusted.
   */
  void setFpTrusted(bool trusted);

  /**
   * @brief Return the names of those channels this buddy is joined to.
   */
  inline const QStringList &channelList(void) const
  {
    return _channels;
  }
 
  /**
   * @brief Return the username of this buddy (from WHOIS information).
   */
  inline const QString userName(void) const
  {
    return _username;
  }

  /**
   * @brief Return the real name of this buddy (from WHOIS information).
   */
  inline const QString realName(void) const
  {
    return _realname;
  }
  
  /**
   * @brief Return whether to allow rich text messages to be sent to the buddy.
   */
  inline bool allowRichText(void) const { return _allowRichText; };

  /**
   * @brief Set whether messages to the buddy may be sent as rich text.
   */
  inline void setAllowRichText(bool v) { _allowRichText = v; };

  /**
   * @brief Set the list of joined channels.
   */
  void setChannelList(QStringList &list) 
  {
    _channels = list;
  }

  /**
   * @brief Convert raw fingerprint into readable hex version.
   */
  static QString convFingerprint(const char *);

  /**
   * @brief Callback function for incoming private messages.
   */
  static void silc_private_message(SilcTK::SilcClient client,
				   SilcTK::SilcClientConnection conn,
				   SilcTK::SilcClientEntry sender,
				   SilcTK::SilcMessagePayload payload,
				   SilcTK::SilcMessageFlags flags,
				   const unsigned char *message,
				   SilcTK::SilcUInt32 message_len);

  /**
   * @brief Return the path of the public key file for the provided
   * fingerprint.
   */
  static const QString publicKeyPath(QString fp);

  /**
   * @brief Return the path of the public key file of this buddy.
   */
  const QString publicKeyPath(void) const;

  /**
   * @brief Return whether a public key file for the provided fingerprint
   * is available.
   */
  static bool havePublicKey(QString fp);

  /**
   * @brief Return whether a public key file is available for this buddy.
   */
  bool havePublicKey(void);

  /**
   * @brief Performs a WHOIS request for this buddy.
   */
  bool whoami(void);

  /**
   * @brief Update the local cache of WHOIS information.
   */
  void updateWhois(QString username, QString realname);

  /**
   * @brief either add or remote the buddy from the watch list
   */
  bool watchme(bool watch);
  static bool watchme_callback(SilcTK::SilcClient client,
                 SilcTK::SilcClientConnection conn,
                 SilcTK::SilcCommand command,
                 SilcTK::SilcStatus status,
                 SilcTK::SilcStatus error,
                 void *context,
                 va_list ap);

  inline bool watched(void) const  { return _watched; };

  virtual void serialize(QMap<QString, QString>&, QMap<QString, QString>&);

  virtual QList<KAction *> *customContextMenuActions(Kopete::ChatSession *);

  virtual void sendFile(const KUrl &sourceURL = KUrl(),
			const QString &fileName = QString::null,
			uint fileSize = 0L);

  virtual void slotUserInfo(void);

  void mimeAlternateToMsg( Kopete::Message &, SilcTK::SilcMime, bool) const;

  /**
   * @brief send file as mime to contact
   */
  virtual void sendFileAsMime(const QString &fileName);

  /**
   * @brief return a reference to the SilcBuddyAttributes of this buddy
   */
  inline SilcBuddyAttributes &attributes(void) 
  {
    return *_attributes;
  }

  /**
   * @brief return a const reference to the SilcBuddyAttributes of this buddy
   */
  inline const SilcBuddyAttributes &attributes(void) const
  {
    return *_attributes;
  }

signals:
  void signalWhois(QString nickname, QString username, QString realname);

public slots:
  void slotIsOp(void);
  void slotIsQuiet(void);
  void slotKick(void);
  void slotSelectClientEntry(void);

protected slots:
  /**
   * @brief send private message to the SILC network
   */
  virtual void slotSendMessage(Kopete::Message &, Kopete::ChatSession *);

private slots:
  /**
   * @brief reset _clientEntry to empty if we went offline
   */
  void slotOnlineStatusChanged(Kopete::Contact *contact,
			       const Kopete::OnlineStatus &status,
			       const Kopete::OnlineStatus &oldStatus);

  void slotPropertyChanged(Kopete::PropertyContainer *, const QString &key,
			   const QVariant &oldValue,
			   const QVariant &newValue);

private:
  SilcBuddyClientEntry *_clientEntry;
  bool _fpTrusted;
  bool _watched;
  bool _allowRichText;

  /* this data is provided by WHOIS */
  QString _username;
  QString _realname;
  QStringList _channels;
  QDateTime _update;
  
  /**
   * @brief SilcBuddyAttributes of this buddy
   */
  SilcBuddyAttributes *_attributes;

  Kopete::ChatSession *_activeManager;

  KAction *_actionIsOp;
  KAction *_actionIsQuiet;
  KAction *_actionKick;
  KActionMenu *_actionClientSelectMenu;
  QList<KAction *> _actionClientSelect;
};





/**
 * Pending Buddies have this format
 *
 */

class SilcBuddyContactData {
  public:
  QString nickname;
  QString finger;
  Kopete::MetaContact *meta;
  SilcAccount *account; 

  SilcBuddyContactData(SilcAccount * account, 
                       QString nickname,
                       QString finger,
                       Kopete::MetaContact *meta);
  inline QString &nickName() { return nickname; }

};

#endif // KOPETESILC_SILCBUDDYCONTACT_H
