/***************************************************************************
                          lmsensor.cpp  -  description
                             -------------------
    begin                : Mon Aug 6 2001
    copyright            : (C) 2001 by 
    email                : 
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "lmsensor.h"


LMSensor::LMSensor(LMSensors *parent): QObject(parent)
{
  feature      = -1;
  description  = "";
  compensation = 0;
  multiplicator= 1;
  monitorize   = false;
}


LMSensor::~LMSensor(){
}


bool LMSensor::init(const sensors_feature_data **data, int *nr1,int *nr2)
{
 const sensors_chip_name *chip_name= getChipName();

 switch ( (*data)->name[0]) {
   case 't': type      = lmTemp;
             valMax    =  65;
             valMin    =  0;
             break;
   case 'f': type      = lmFan;
             valMax    =  10000;
             valMin    =  3000;
             break;
   default:  type      = lmVoltage;
						 valMax    =   16;
             valMin    =  -16;
             break;
 }

 feature= (*data)->number;

 setName( (*data)->name );

 char *label;
 sensors_get_label(*chip_name,feature,&label);
 description= label;

 QString str;
 bool valid= false;
 while( (*data= sensors_get_all_features(*chip_name, nr1, nr2)) && (*data)->mapping!=SENSORS_NO_MAPPING) {
   str= (*data)->name;
   if(str.find("_min")>=0) {
       sensors_get_feature(*chip_name, (*data)->number, &valMin);
       valid= true;
   } else
       if(str.find("_max")>=0 || str.find("_over")>=0) {
          sensors_get_feature(*chip_name, (*data)->number, &valMax);
          valid= true;
       }
 }

 if(valid) {
    double newVal;
    valid= (sensors_get_feature(*chip_name, feature, &newVal)==0);
    if(valid) {
        if(valMin>valMax) {
          double pivot= valMin;
          valMin= valMax;
          valMax= pivot;
        }
        val= (valMax+valMin)/2;
        readConfig();
        updateValue();
    }
 }

 return valid;
}

QString LMSensor::getPrintMask(bool addSufix)
{
 return LMSensors::getSensorPrintMask(type, addSufix, ((LMSensors*)parent())->getTempScale() );
}

KConfig *LMSensor::getConfig()
{
 return ((LMSensors*)parent())->getConfig();
}

const sensors_chip_name *LMSensor::getChipName()
{
  return ((LMSensors *)parent())->getChipName();
}

bool LMSensor::monitorized()
{
  return monitorize;
}

void LMSensor::setMonitorized(bool enable)
{
 if( monitorize!=enable ) monitorize= enable;
}

void LMSensor::setValueIdeal(double value)
{
 valIdeal= current2Celsius(value);
}

	
void LMSensor::setValueMax(double value)
{
  valMax= current2Celsius(value);
}

void LMSensor::setValueMin(double value)
{
  valMin= current2Celsius(value);
}

void LMSensor::setCompensation(double value)
{
  compensation= current2Celsius(value);
}

void LMSensor::setMultiplicator(double value)
{
 multiplicator= value;
}

void LMSensor::setDescription(const QString &str)
{
 description= str;
}

void LMSensor::updateValue()
{
 double newVal= readValueFromLMSensor();
 if(val!=newVal) {
   valPrevious= val;
   val        = newVal;
   emit valueChanged( celsius2Current(val));
 }
}

void LMSensor::writeConfig()
{
KConfig *ksConfig= getConfig();

  if(ksConfig) {
    ksConfig->setGroup( QString(name()) );
    ksConfig->writeEntry("description"  ,description);
    ksConfig->writeEntry("valueIdeal"   ,valIdeal);
    ksConfig->writeEntry("valueMax"     ,valMax);
    ksConfig->writeEntry("valueMin"     ,valMin);
    ksConfig->writeEntry("compensation" ,compensation);
    ksConfig->writeEntry("multiplicator",multiplicator);
    ksConfig->writeEntry("monitorize"   ,monitorize);
    emit configChanged();
  }
}

void LMSensor::readConfig()
{
KConfig *ksConfig= getConfig();

  if(ksConfig) {
    ksConfig->setGroup( QString(name()) );
    valMax       = ksConfig->readDoubleNumEntry("valueMax"     ,valMax);
    valMin       = ksConfig->readDoubleNumEntry("valueMin"     ,valMin);
    compensation = ksConfig->readDoubleNumEntry("compensation" ,compensation);
    multiplicator= ksConfig->readDoubleNumEntry("multiplicator",multiplicator);
    description  = ksConfig->readEntry         ("description"  ,description);
    valIdeal     = ksConfig->readDoubleNumEntry("valueIdeal"   ,-1000000);
    if(valIdeal<=-1000000) {
      valIdeal= calculateIdealValue();
    }
    setMonitorized( ksConfig->readBoolEntry    ("monitorize",monitorize) );
  }
}

#define ABSOLUTE_VALUE(n)  ( (n)>=0 ? (n) : -(n) )
#define TRUNCATE_VALUE(n)  ( (double)(int)(n)    )

double LMSensor::calculateIdealValue()
{
 double value= readValueFromLMSensor();

 if(type==lmVoltage) {
     double decimals= 10;
     double module = ABSOLUTE_VALUE(value);
     if(module>3.0) {
       if( ABSOLUTE_VALUE( 12.0-value)<2.0 ) return  12.0;
       if( ABSOLUTE_VALUE(-12.0-value)<2.0 ) return -12.0;
       if( ABSOLUTE_VALUE(  5.0-value)<1.0 ) return   5.0;
       if( ABSOLUTE_VALUE( -5.0-value)<1.0 ) return  -5.0;
       if( ABSOLUTE_VALUE(  3.3-value)<0.3 ) return   3.3;
       if( ABSOLUTE_VALUE( -3.3-value)<0.3 ) return  -3.3;
       if(module>4.0) decimals= 1;
     }
     return TRUNCATE_VALUE((value * decimals)) / decimals;
  } else {
     return value;
  }
}


double LMSensor::readValueFromLMSensor()
{
 double newVal;
 const sensors_chip_name *chip_name= getChipName();
 sensors_get_feature(*chip_name, feature, &newVal);
 newVal= newVal * multiplicator + compensation;
 return newVal;
}

double LMSensor::current2Celsius(double val)
{
  if(type!=lmTemp) return val;
  switch( ((LMSensors *)parent())->getTempScale() ) {
   case LMSensors::dgCelsius   : return val;
   case LMSensors::dgFahrenheit: return  (5.0/9.0)*(val-32.0);
  }
  return  val - 273.16;  // Kelvin to Celsius
}

double LMSensor::celsius2Current(double val)
{
  if(type!=lmTemp) return val;
  switch( ((LMSensors *)parent())->getTempScale() ) {
   case LMSensors::dgCelsius   : return val;
   case LMSensors::dgFahrenheit: return  (9.0/5.0)*val+32.0;
  }
  return  val + 273.16;  // Celsius to Kelvin
}
