/***************************************************************************
                          lmsensorscfg.cpp  -  description
                             -------------------
    begin                : Mon Aug 13 2001
    copyright            : (C) 2001 by 
    email                : 
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "lmsensorscfg.h"

#include "lmsensorsalarms.h"
#include "lmsensorpanel.h"
#include "lmsensordockpanel.h"
#include "lmsensorswidget.h"

#include <qcolor.h>
#include <qlistbox.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qradiobutton.h>
#include <qpushbutton.h>
#include <qobjcoll.h>
#include <qtabwidget.h>
#include <qcombobox.h>
#include <qspinbox.h>
#include <kcolorbutton.h>
#include <kiconloader.h>
#include <klocale.h>

#include <kaboutapplication.h>

LMSensorsCfg::LMSensorsCfg(LMSensors *lsensors,QWidget *parent, const char *name) : LMSensorsCfgDesign(parent,name,WDestructiveClose)
{
 sensors  = lsensors;

 curSensor= 0;
 indexCurSensor= -1;

 connect( MainTab, SIGNAL(currentChanged(QWidget*)), this, SLOT(slotMainTabPageChanged(QWidget *)));
 connect( listSensors ,SIGNAL(highlighted(int)),this,SLOT(slotCurrentSensorChanged(int)) );
 connect( buttonApply ,SIGNAL(clicked()),this,SLOT(slotApplyChanges()));
 connect( buttonCancel,SIGNAL(clicked()),this,SLOT(close()));
 connect( buttonAccept,SIGNAL(clicked()),this,SLOT(slotAccept()));
 connect( buttonTest  ,SIGNAL(clicked()),this,SLOT(slotTestAlarm()));
 connect( editMinValue,SIGNAL(activated(int)) ,this,SLOT(slotComboMinMaxSelected(int)));
 connect( editMaxValue,SIGNAL(activated(int)) ,this,SLOT(slotComboMinMaxSelected(int)));

 QObjectList *list= sensors->getSensors();
 if(list) {
   QPixmap pm;
   for(LMSensor *sensor=(LMSensor*)list->first(); sensor!=0; sensor= (LMSensor*)list->next()) {
     switch(sensor->getSensorType()) {
       case LMSensor::lmTemp   : pm= UserIcon("thermometer18x18.png"); break;
       case LMSensor::lmFan    : pm= UserIcon("cooler18x18.png");      break;
       case LMSensor::lmVoltage: pm= UserIcon("battery18x18.png");     break;
     }
     listSensors->insertItem(pm,sensor->getDescription());
   }
   listSensors->setCurrentItem(0);
 }

}

LMSensorsCfg::~LMSensorsCfg()
{
}


void LMSensorsCfg::readSensorInfo(int index)
{
 if(index<0) return;

 LMSensor *oldSensor= curSensor;

 curSensor= sensors->getSensor(index);

 if(!curSensor) return;

 indexCurSensor= index;

 KConfig *cfg= sensors->getConfig();

 if( oldSensor==0 || curSensor->getSensorType()!=oldSensor->getSensorType() )
   switch(curSensor->getSensorType()) {
     case LMSensor::lmTemp   : GroupBoxDes->setTitle(i18n("Temperature"));
															 pixmapSensor->setPixmap(UserIcon("thermometer.png"));
															 break;
     case LMSensor::lmFan    : GroupBoxDes->setTitle(i18n("Fan"));
															 pixmapSensor->setPixmap(UserIcon("cooler.png"));
															 break;
     case LMSensor::lmVoltage: GroupBoxDes->setTitle(i18n("Voltage"));
                               pixmapSensor->setPixmap(UserIcon("battery.png"));
															 break;
   }

 editDescription ->setText(curSensor->getDescription());

 QString str;

 const char *mask= curSensor->getSensorType()==LMSensor::lmVoltage ? "%.2lf":"%.0lf";

 str.sprintf(curSensor->getPrintMask(true), curSensor->getValue());
 editValue->setText(str);

 str.sprintf(mask, curSensor->getValueIdeal());
 editIdealValue->setText(str);

 str.sprintf(mask, curSensor->getValueMin());
 editMinValue->setEditText(str);

 str.sprintf(mask, curSensor->getValueMax());
 editMaxValue->setEditText(str);

 editSum->setText(QString::number(curSensor->getCompensation()));
 editMul->setText(QString::number(curSensor->getMultiplicator()));

 checkShowInDock->setEnabled( curSensor->getSensorType()==LMSensor::lmTemp );

 CheckAllSensors->setChecked(false);

 if(cfg) {
   colorAlarm     ->setColor  ( LMSensorDockPanel::readColorAlarm(cfg,curSensor->name()) );
   colorNormal    ->setColor  ( LMSensorDockPanel::readColorNormal(cfg,curSensor->name()) );
   checkShow      ->setChecked( LMSensorPanel::readShow(cfg,curSensor->name()) );
   checkShowInDock->setChecked( checkShowInDock->isEnabled() && LMSensorDockPanel::readShowInDock(cfg,curSensor->name()) );

   editLaunch->setText( LMSensorsAlarms::readAlarmCommand(cfg,curSensor->name()) );
   editPlay  ->setText( LMSensorsAlarms::readAlarmSound  (cfg,curSensor->name()) );
   switch( LMSensorsAlarms::readAlarm(cfg,curSensor->name()) )
   {
     case LMSensorsAlarms::acNothing : radioAlarm1->setChecked(true); break;
     case LMSensorsAlarms::acSound   : radioAlarm2->setChecked(true); break;
     case LMSensorsAlarms::acCommand : radioAlarm3->setChecked(true); break;
   }
 }
}


void LMSensorsCfg::readSystemInfo()
{
 KConfig *cfg= sensors->getConfig();

 CheckCPU    ->setChecked( LMSensorPanel::readShow(cfg,"proc.CPUINFO" ) );
 CheckCPULoad->setChecked( LMSensorPanel::readShow(cfg,"proc.CPULOAD" ) );
 CheckRAM    ->setChecked( LMSensorPanel::readShow(cfg,"proc.RAMINFO" ) );
 CheckSWAP   ->setChecked( LMSensorPanel::readShow(cfg,"proc.SWAPINFO") );
 CheckUpTime ->setChecked( LMSensorPanel::readShow(cfg,"proc.UPTIME"  ) );
}

void LMSensorsCfg::readPreferencesInfo()
{
 switch(sensors->getTempScale()) {
  case LMSensors::dgCelsius   :  RadioC->setChecked(true); break;
  case LMSensors::dgFahrenheit:  RadioF->setChecked(true); break;
  case LMSensors::dgKelvin    :  RadioK->setChecked(true); break;
 }
 SpinUpdateTime ->setValue(sensors->getUpdateInterval());
 SpinUpdateTime2->setValue(LMSensorsWidget::cfgReadUpdateInterval(sensors->getConfig()));
}


void LMSensorsCfg::applySensorChanges()
{
 if(!curSensor) return;

 KConfig *cfg= sensors->getConfig();
 const QString sName(curSensor->name());
 // Apply sensor configuration
 curSensor->setDescription(editDescription->text());
 curSensor->setCompensation(editSum->text().toDouble());
 curSensor->setMultiplicator(editMul->text().toDouble());
 curSensor->setValueIdeal(editIdealValue->text().toDouble());
 curSensor->setValueMax(editMaxValue->currentText().toDouble());
 curSensor->setValueMin(editMinValue->currentText().toDouble());
 curSensor->setMonitorized(
                            checkShow->isChecked()       ||
                            checkShowInDock->isChecked() ||
                            !radioAlarm1->isChecked()
                          );
 curSensor->writeConfig();
 // Apply show in main window configuration
 LMSensorPanel::writeShow(cfg,sName,checkShow->isChecked() );
 emit itemShowChanged(curSensor->name());
 // Apply dock configuration
 LMSensorDockPanel::writeShowInDock(cfg,sName,checkShowInDock->isChecked());
 LMSensorDockPanel::writeColorNormal(cfg,sName,colorNormal->color());
 LMSensorDockPanel::writeColorAlarm(cfg,sName,colorAlarm->color());
 emit itemDockChanged(curSensor->name());
 // Apply alarms configuration
 if( CheckAllSensors->isChecked() ) {
    QObjectList *list= sensors->getSensors();
    for(LMSensor *sensor=(LMSensor*)list->first(); sensor!=0; sensor= (LMSensor*)list->next())
       applySensorAlarmConfig(cfg,sensor->name());
 } else {
    applySensorAlarmConfig(cfg,curSensor->name());
 }
 // This line always must be at the end of the method
 listSensors->changeItem( *listSensors->pixmap(indexCurSensor),editDescription->text(), indexCurSensor );
}


void LMSensorsCfg::applySensorAlarmConfig(KConfig *cfg,const char *name)
{
 LMSensorsAlarms::writeAlarmCommand(cfg,name,editLaunch->text());
 LMSensorsAlarms::writeAlarmSound  (cfg,name,editPlay  ->text());
 int code= LMSensorsAlarms::acNothing;
 if(radioAlarm1->isChecked()) code= LMSensorsAlarms::acNothing;
 else if(radioAlarm2->isChecked()) code= LMSensorsAlarms::acSound;
 else    if(radioAlarm3->isChecked()) code= LMSensorsAlarms::acCommand;
 LMSensorsAlarms::writeAlarm(cfg,name,code);
 emit itemAlarmChanged(name);
}


void LMSensorsCfg::applyOneSystemInfoItem(KConfig *cfg, QCheckBox *check,const char *name)
{
 if(check->isChecked()!=LMSensorPanel::readShow(cfg,name)) {
   LMSensorPanel::writeShow(cfg,name,check->isChecked() );
   emit itemShowChanged(name);
 }
}

void LMSensorsCfg::applySystemInfoChanges()
{
 KConfig *cfg= sensors->getConfig();

 applyOneSystemInfoItem(cfg, CheckCPU    , "proc.CPUINFO" );
 applyOneSystemInfoItem(cfg, CheckCPULoad, "proc.CPULOAD" );
 applyOneSystemInfoItem(cfg, CheckRAM    , "proc.RAMINFO" );
 applyOneSystemInfoItem(cfg, CheckSWAP   , "proc.SWAPINFO");
 applyOneSystemInfoItem(cfg, CheckUpTime , "proc.UPTIME"  );
}


void LMSensorsCfg::applyPreferencesChanges()
{
 LMSensors::TempScale scale= LMSensors::dgCelsius;
 if(RadioK->isChecked()) scale= LMSensors::dgKelvin;
 else if(RadioF->isChecked()) scale= LMSensors::dgFahrenheit;
 sensors->setTempScale(scale);
 sensors->setUpdateInterval(SpinUpdateTime->value());
 sensors->writeConfig();

 LMSensorsWidget::cfgWriteUpdateInterval(sensors->getConfig(),SpinUpdateTime2->value());

 emit systemUpdateIntervalChanged();
}



void LMSensorsCfg::slotCurrentSensorChanged(int index)
{
 if(index==indexCurSensor) return;
 readSensorInfo(index);
}

void LMSensorsCfg::slotApplyChanges()
{
  switch(MainTab->currentPageIndex())
  {
    case 0: applySensorChanges();      break;
    case 1: applySystemInfoChanges();  break;
    case 2: applyPreferencesChanges(); break;
  }
}

void LMSensorsCfg::slotAccept()
{
 slotApplyChanges();
 close();
}

void LMSensorsCfg::slotTestAlarm()
{
  if( radioAlarm2->isChecked() ) {
    LMSensorsAlarms::playSound(editPlay->text());
  } else
  if( radioAlarm3->isChecked() ) {
    LMSensorsAlarms::runSensorCommand("TEST","100",editLaunch->text());
  }
}

void LMSensorsCfg::slotMainTabPageChanged(QWidget *)
{
  switch(MainTab->currentPageIndex())
  {
    case 0: readSensorInfo(listSensors->currentItem());  break;
    case 1: readSystemInfo();      											 break;
    case 2: readPreferencesInfo();                       break;
  }
}

void LMSensorsCfg::slotComboMinMaxSelected(int index)
{
 QComboBox *combo  = (QComboBox*)sender();
 QString       str = combo->text(index);
 double     percent= str.left(str.length()-1).toDouble();
 double     value  = editIdealValue->text().toDouble();

 if(value<0) percent= -percent;
 value= value + value * percent / 100;

 const char *mask= curSensor->getSensorType()==LMSensor::lmVoltage ? "%.2lf":"%.0lf";
 str.sprintf(mask,value);

 combo->setEditText(str);
}


